"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Validators = void 0;
const core_1 = require("aws-cdk-lib/core");
class Validators {
    /**
     * Validates whether a supplied value conforms to the allowedPattern, granted neither is an unresolved token.
     *
     * @param value          the value to be validated.
     * @param allowedPattern the regular expression to use for validation.
     *
     * @throws if the ``value`` does not conform to the ``allowedPattern`` and neither is an unresolved token (per
     *         ``cdk.unresolved``).
     */
    static parameterValue(value, allowedPattern) {
        if (core_1.Token.isUnresolved(value) || core_1.Token.isUnresolved(allowedPattern)) {
            // Unable to perform validations against unresolved tokens
            return;
        }
        if (!new RegExp(allowedPattern).test(value)) {
            throw new Error(`The supplied value (${value}) does not match the specified allowedPattern (${allowedPattern})`);
        }
    }
    static parameterName(parameterName) {
        if (core_1.Token.isUnresolved(parameterName)) {
            throw new Error("Unable to determine parameter name for SSM parameter since the parameter name is an unresolved token.");
        }
        if (parameterName.length > 2048) {
            throw new Error("Name cannot be longer than 2048 characters.");
        }
        if (!parameterName.match(/^\/[\/\w.-]+$/)) {
            throw new Error(`Name must begin with / and must only contain letters, numbers, and the following 4 symbols .-_/; got ${parameterName}`);
        }
    }
    static description(description) {
        if (description && description?.length > 1024) {
            throw new Error("Description cannot be longer than 1024 characters.");
        }
    }
}
exports.Validators = Validators;
//# sourceMappingURL=data:application/json;base64,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