"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringParameter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ssm = require("aws-cdk-lib/aws-ssm");
const cxschema = require("aws-cdk-lib/cloud-assembly-schema");
const custom_resources = require("aws-cdk-lib/custom-resources");
const parameter_base_1 = require("./parameter-base");
const validators_1 = require("./validators");
const RESOURCE_TYPE = "Custom::SSM_String_Parameter_Cross_Region";
class StringParameter extends parameter_base_1.ParameterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.parameterName,
        });
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.KEY_VALUE, RESOURCE_TYPE);
        if (props.allowedPattern) {
            validators_1.Validators.parameterValue(props.stringValue, props.allowedPattern);
        }
        validators_1.Validators.parameterName(props.parameterName);
        validators_1.Validators.description(props.description);
        this.parameterArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: "ssm",
            resource: `parameter${props.parameterName}`,
            region: props.region,
        });
        const removalPolicy = props.removalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        new custom_resources.AwsCustomResource(this, "Resource", {
            resourceType: RESOURCE_TYPE,
            onCreate: {
                region: props.region,
                service: "SSM",
                action: "putParameter",
                parameters: {
                    Name: props.parameterName,
                    Value: props.stringValue,
                    AllowedPattern: props.allowedPattern,
                    DataType: props.dataType,
                    Description: props.description,
                    Tags: this.tags.renderedTags,
                    Tier: props.tier,
                    Type: ssm.ParameterType.STRING,
                },
                physicalResourceId: custom_resources.PhysicalResourceId.of(this.parameterArn),
            },
            onUpdate: {
                region: props.region,
                service: "SSM",
                action: "putParameter",
                parameters: {
                    Name: props.parameterName,
                    Value: props.stringValue,
                    AllowedPattern: props.allowedPattern,
                    DataType: props.dataType,
                    Description: props.description,
                    Tags: this.tags.renderedTags,
                    Tier: props.tier,
                    Type: ssm.ParameterType.STRING,
                },
                physicalResourceId: custom_resources.PhysicalResourceId.of(this.parameterArn),
            },
            onDelete: removalPolicy != aws_cdk_lib_1.RemovalPolicy.RETAIN
                ? {
                    region: props.region,
                    service: "SSM",
                    action: "deleteParameter",
                    parameters: {
                        Name: props.parameterName,
                    },
                    physicalResourceId: custom_resources.PhysicalResourceId.of(this.parameterArn),
                }
                : undefined,
            policy: custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        this.parameterName = props.parameterName;
        this.parameterType = ssm.ParameterType.STRING;
        this.stringValue = props.stringValue;
    }
    /**
     * Imports an external string parameter with name and optional version.
     */
    static fromStringParameterAttributes(scope, id, attrs) {
        if (!attrs.parameterName) {
            throw new Error("parameterName cannot be an empty string");
        }
        const parameterType = attrs.type ?? ssm.ParameterType.STRING;
        const parameterName = attrs.version
            ? `${attrs.parameterName}:${aws_cdk_lib_1.Tokenization.stringifyNumber(attrs.version)}`
            : attrs.parameterName;
        class Import extends parameter_base_1.ParameterBase {
            constructor() {
                super(scope, id);
                this.parameterArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: "ssm",
                    resource: `parameter${parameterName}`,
                    region: attrs.region,
                });
                const parameter = new custom_resources.AwsCustomResource(this, "Resource", {
                    resourceType: RESOURCE_TYPE,
                    onCreate: {
                        region: attrs.region,
                        service: "SSM",
                        action: "getParameter",
                        parameters: {
                            Name: parameterName,
                        },
                        physicalResourceId: custom_resources.PhysicalResourceId.of(this.parameterArn),
                    },
                    onUpdate: {
                        region: attrs.region,
                        service: "SSM",
                        action: "getParameter",
                        parameters: {
                            Name: parameterName,
                        },
                        physicalResourceId: custom_resources.PhysicalResourceId.of(this.parameterArn),
                    },
                    policy: custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                        resources: custom_resources.AwsCustomResourcePolicy.ANY_RESOURCE,
                    }),
                });
                this.parameterName = attrs.parameterName;
                this.parameterType = parameterType;
                this.stringValue = parameter.getResponseField("Parameter.Value");
            }
        }
        return new Import();
    }
    /**
     * Imports an external string parameter by name and region.
     */
    static fromStringParameterName(scope, id, region, parameterName) {
        return this.fromStringParameterAttributes(scope, id, { parameterName: parameterName, region: region });
    }
    /**
     * Reads the value of an SSM parameter during synthesis through an
     * environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account information. Otherwise, it will fail during synthesis.
     */
    static valueFromLookup(scope, region, parameterName) {
        return aws_cdk_lib_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SSM_PARAMETER_PROVIDER,
            props: { parameterName: parameterName, region: region },
            dummyValue: `dummy-value-for-${parameterName}`,
        }).value;
    }
}
exports.StringParameter = StringParameter;
_a = JSII_RTTI_SYMBOL_1;
StringParameter[_a] = { fqn: "@pepperize/cdk-ssm-parameters-cross-region.StringParameter", version: "0.0.37" };
//# sourceMappingURL=data:application/json;base64,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