# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['kafkaesk', 'kafkaesk.ext', 'kafkaesk.ext.logging']

package_data = \
{'': ['*']}

install_requires = \
['aiokafka>=0.7.0,<0.8.0',
 'async-timeout>=3.0.1,<4.0.0',
 'jsonschema>=3.2.0,<4.0.0',
 'kafka-python>=2.0.1,<3.0.0',
 'opentracing>=2.3.0,<3.0.0',
 'orjson>=3.3.1,<4.0.0',
 'prometheus_client>=0.8.0',
 'pydantic>=1.5.1,<2.0.0']

entry_points = \
{'console_scripts': ['kafkaesk = kafkaesk.app:run']}

setup_kwargs = {
    'name': 'kafkaesk',
    'version': '0.7.4',
    'description': 'Easy publish and subscribe to events with python and Kafka.',
    'long_description': '<!-- PROJECT LOGO -->\n<h1 align="center">\n  <br>\n  <img src="https://onna.com/wp-content/uploads/2020/03/h-onna-solid.png" alt="Onna Logo"></a>\n</h1>\n\n<h2 align="center">kafkaesk</h2>\n\n<!-- TABLE OF CONTENTS -->\n## Table Of Contents\n\n- [About the Project](#about-the-project)\n- [Publish](#publish)\n- [Subscribe](#subscribe)\n- [Avoiding global object](#avoiding-global-object)\n- [Manual commit](#manual-commit)\n- [kafkaesk contract](#kafkaesk-contract)\n- [Worker](#worker)\n- [Development](#development)\n- [Extensions](#extensions)\n- [Naming](#naming)\n\n\n## About The Project\n\nThis project is meant to help facilitate effortless publishing and subscribing to events with Python and Kafka.\n\n### Guiding principal\n\n- HTTP\n- Language agnostic\n- Contracts built on top of [Kafka](https://kafka.apache.org/)\n\n\n### Alternatives\n - [aiokafka](https://aiokafka.readthedocs.io/en/stable/): can be complex to scale correctly\n - [guillotina_kafka](https://github.com/onna/guillotina_kafka): complex, tied to [Guillotina](https://guillotina.readthedocs.io/en/latest/)\n - [faust](https://faust.readthedocs.io/en/latest/): requires additional data layers, not language agnostic\n - confluent kafka + avro: close but ends up being like grpc. compilation for languages. No asyncio.\n\n> Consider this Python project as syntactic sugar around these ideas.\n\n## Publish\n\nUsing [pydantic](https://pydantic-docs.helpmanual.io/) but can be done with pure JSON.\n\n```python\nimport kafkaesk\nfrom pydantic import BaseModel\n\napp = kafkaesk.Application()\n\n@app.schema("Content", version=1, retention=24 * 60 * 60)\nclass ContentMessage(BaseModel):\n    foo: str\n\n\nasync def foobar():\n    # ...\n    # doing something in an async func\n    await app.publish("content.edited.Resource", data=ContentMessage(foo="bar"))\n```\n\nA convenience method is available in the `subscriber` dependency instance, this allow to header\npropagation from the consumed message.\n\n```python\nimport kafkaesk\nfrom pydantic import BaseModel\n\napp = kafkaesk.Application()\n\n@app.schema("Content", version=1, retention=24 * 60 * 60)\nclass ContentMessage(BaseModel):\n    foo: str\n\n\n@app.subscribe("content.*", "group_id")\nasync def get_messages(data: ContentMessage, subscriber):\n    print(f"{data.foo}")\n    # This will propagate `data` record headers\n    await subscriber.publish("content.edited.Resource", data=ContentMessage(foo="bar"))\n\n```\n\n## Subscribe\n\n```python\nimport kafkaesk\nfrom pydantic import BaseModel\n\napp = kafkaesk.Application()\n\n@app.schema("Content", version=1, retention=24 * 60 * 60)\nclass ContentMessage(BaseModel):\n    foo: str\n\n\n@app.subscribe("content.*", "group_id")\nasync def get_messages(data: ContentMessage):\n    print(f"{data.foo}")\n\n```\n\n## Avoiding global object\n\nIf you do not want to have global application configuration, you can lazily configure\nthe application and register schemas/subscribers separately.\n\n```python\nimport kafkaesk\nfrom pydantic import BaseModel\n\nrouter = kafkaesk.Router()\n\n@router.schema("Content", version=1, retention=24 * 60 * 60)\nclass ContentMessage(BaseModel):\n    foo: str\n\n\n@router.subscribe("content.*", "group_id")\nasync def get_messages(data: ContentMessage):\n    print(f"{data.foo}")\n\n\nif __name__ == "__main__":\n    app = kafkaesk.Application()\n    app.mount(router)\n    kafkaesk.run(app)\n\n```\n\nOptional consumer injected parameters:\n\n- schema: str\n- record: aiokafka.structs.ConsumerRecord\n- app: kafkaesk.app.Application\n- subscriber: kafkaesk.app.BatchConsumer\n\nDepending on the type annotation for the first parameter, you will get different data injected:\n\n- `async def get_messages(data: ContentMessage)`: parses pydantic schema\n- `async def get_messages(data: bytes)`: give raw byte data\n- `async def get_messages(record: aiokafka.structs.ConsumerRecord)`: give kafka record object\n- `async def get_messages(data)`: raw json data in message\n\n## Manual commit\n\nTo accomplish a manual commit strategy yourself:\n\n```python\napp = kafkaesk.Application(auto_commit=False)\n\n@app.subscribe("content.*", "group_id")\nasync def get_messages(data: ContentMessage, subscriber):\n    print(f"{data.foo}")\n    await subscriber.consumer.commit()\n```\n\n## SSL\nAdd these values to your `kafak_settings`:\n    `ssl_context` - this should be a placeholder as the SSL Context is generally created within the application\n    `security_protocol` - one of SSL or PLAINTEXT\n    `sasl_mechanism` - one of PLAIN, GSSAPI, SCRAM-SHA-256, SCRAM-SHA-512, OAUTHBEARER\n    `sasl_plain_username`\n    `sasl_plain_password`\n\n## kafkaesk contract\n\nThis is a library around using kafka.\nKafka itself does not enforce these concepts.\n\n- Every message must provide a json schema\n- Messages produced will be validated against json schema\n- Each topic will have only one schema\n- A single schema can be used for multiple topics\n- Consumed message schema validation is up to the consumer\n- Messages will be consumed at least once. Considering this, your handling should be idempotent\n\n### Message format\n\n```json\n{\n    "schema": "schema_name:1",\n    "data": { ... }\n}\n```\n\n## Worker\n\n```bash\nkafkaesk mymodule:app --kafka-servers=localhost:9092\n```\n\nOptions:\n\n - --kafka-servers: comma separated list of kafka servers\n - --kafka-settings: json encoded options to be passed to https://aiokafka.readthedocs.io/en/stable/api.html#aiokafkaconsumer-class\n - --topic-prefix: prefix to use for topics\n - --replication-factor: what replication factor topics should be created with. Defaults to min(number of servers, 3).\n\n### Application.publish\n\n- stream_id: str: name of stream to send data to\n- data: class that inherits from pydantic.BaseModel\n- key: Optional[bytes]: key for message if it needs one\n\n### Application.subscribe\n\n- stream_id: str: fnmatch pattern of streams to subscribe to\n- group: Optional[str]: consumer group id to use. Will use name of function if not provided\n\n### Application.schema\n\n- id: str: id of the schema to store\n- version: Optional[int]: version of schema to store\n- streams: Optional[List[str]]: if streams are known ahead of time, you can pre-create them before you push data\n- retention: Optional[int]: retention policy in seconds\n\n### Application.configure\n\n- kafka_servers: Optional[List[str]]: kafka servers to connect to\n- topic_prefix: Optional[str]: topic name prefix to subscribe to\n- kafka_settings: Optional[Dict[str, Any]]: additional aiokafka settings to pass in\n- replication_factor: Optional[int]: what replication factor topics should be created with. Defaults to min(number of servers, 3).\n- kafka_api_version: str: default `auto`\n- auto_commit: bool: default `True`\n- auto_commit_interval_ms: int: default `5000`\n\n## Development\n\n### Requirements\n\n- [Docker](https://www.docker.com/)\n- [Poetry](https://python-poetry.org/)\n\n```bash\npoetry install\n```\n\nRun tests:\n\n```bash\ndocker-compose up\nKAFKA=localhost:9092 poetry run pytest tests\n```\n\n## Extensions\n\n### Logging\nThis extension includes classes to extend Python\'s logging framework to publish structured log messages to a Kafka topic.\nThis extension is made up of three main components: an extended `logging.LogRecord` and some custom `logging.Handler`s.\n\nSee `logger.py` in examples directory.\n\n#### Log Record\n`kafkaesk.ext.logging.record.factory` is a function that will return `kafkaesk.ext.logging.record.PydanticLogRecord` objects.\nThe `factory()` function scans through any `args` passed to a logger and checks each item to determine if it is a subclass of `pydantid.BaseModel`.\n\nIf it is a base model instance and `model._is_log_model` evaluates to `True` the model will be removed from `args` and added to `record._pydantic_data`.\nAfter that `factory()` will use logging\'s existing logic to finish creating the log record.\n\n### Handler\nThis extensions ships with two handlers capable of handling `kafkaesk.ext.logging.handler.PydanticLogModel` classes: `kafakesk.ext.logging.handler.PydanticStreamHandler` and `kafkaesk.ext.logging.handler.PydanticKafkaeskHandler`.\n\nThe stream handler is a very small wrapper around `logging.StreamHandler`, the signature is the same, the only difference is that the handler will attempt to convert any pydantic models it receives to a human readable log message.\n\nThe kafkaesk handler has a few more bits going on in the background.\n\nThe handler has two required inputs, a `kafkaesk.app.Application` instance and a stream name.\n\nOnce initialized any logs emitted by the handler will be saved into an internal queue.\nThere is a worker task that handles pulling logs from the queue and writing those logs to the specified topic.\n\n# Naming\n\nIt\'s hard and "kafka" is already a fun name.\nHopefully this library isn\'t literally "kafkaesque" for you.\n',
    'author': 'vangheem',
    'author_email': 'vangheem@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
