"use strict";
const nodeunit_1 = require("nodeunit");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
class SomeToken {
    constructor() {
        this.foo = 60;
        this.goo = 40;
    }
    resolve() {
        return this.foo + this.goo;
    }
}
class Nested extends lib_1.Stack {
    constructor(scope, id) {
        const resource = new lib_1.CfnResource(scope, `${id}+NestedStackResource`, { type: 'AWS::CloudFormation::Stack' });
        super(scope, id);
        this.nestedStackResource = resource;
    }
}
module.exports = nodeunit_1.testCase({
    'capitalizeResourceProperties capitalizes all keys of an object (recursively) from camelCase to PascalCase'(test) {
        const c = new lib_1.Stack();
        test.equal(util_1.capitalizePropertyNames(c, undefined), undefined);
        test.equal(util_1.capitalizePropertyNames(c, 12), 12);
        test.equal(util_1.capitalizePropertyNames(c, 'hello'), 'hello');
        test.deepEqual(util_1.capitalizePropertyNames(c, ['hello', 88]), ['hello', 88]);
        test.deepEqual(util_1.capitalizePropertyNames(c, { Hello: 'world', hey: 'dude' }), { Hello: 'world', Hey: 'dude' });
        test.deepEqual(util_1.capitalizePropertyNames(c, [1, 2, { three: 3 }]), [1, 2, { Three: 3 }]);
        test.deepEqual(util_1.capitalizePropertyNames(c, { Hello: 'world', recursive: { foo: 123, there: { another: ['hello', { world: 123 }] } } }), { Hello: 'world', Recursive: { Foo: 123, There: { Another: ['hello', { World: 123 }] } } });
        // make sure tokens are resolved and result is also capitalized
        test.deepEqual(util_1.capitalizePropertyNames(c, { hello: { resolve: () => ({ foo: 'bar' }) }, world: new SomeToken() }), { Hello: { Foo: 'bar' }, World: 100 });
        test.done();
    },
    ignoreEmpty: {
        '[]'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty([])), undefined);
            test.done();
        },
        '{}'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty({})), undefined);
            test.done();
        },
        'undefined/null'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(undefined)), undefined);
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(null)), null);
            test.done();
        },
        'primitives'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(12)), 12);
            test.strictEqual(stack.resolve(util_1.ignoreEmpty('12')), '12');
            test.done();
        },
        'non-empty arrays/objects'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(util_1.ignoreEmpty([1, 2, 3, undefined])), [1, 2, 3]); // undefined array values is cleaned up by "resolve"
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ o: 1, b: 2, j: 3 })), { o: 1, b: 2, j: 3 });
            test.done();
        },
        'resolve first'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => 123 } })), { xoo: 123 });
            test.strictEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => undefined } })), undefined);
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => [] } })), { xoo: [] });
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => [undefined, undefined] } })), { xoo: [] });
            test.done();
        },
    },
    filterUnderined: {
        'is null-safe (aka treats null and undefined the same)'(test) {
            test.deepEqual(util_1.filterUndefined({ 'a null': null, 'a not null': true }), { 'a not null': true });
            test.done();
        },
        'removes undefined, but leaves the rest'(test) {
            test.deepEqual(util_1.filterUndefined({ 'an undefined': undefined, 'yes': true }), { yes: true });
            test.done();
        },
    },
    'pathToTopLevelStack returns the array of stacks that lead to a stack'(test) {
        const a = new lib_1.Stack(undefined, 'a');
        const aa = new Nested(a, 'aa');
        const aaa = new Nested(aa, 'aaa');
        test.deepEqual(path(aaa), ['a', 'aa', 'aaa']);
        test.deepEqual(path(aa), ['a', 'aa']);
        test.deepEqual(path(a), ['a']);
        test.done();
        function path(s) {
            return util_1.pathToTopLevelStack(s).map(x => x.node.id);
        }
    },
    'findCommonStack returns the lowest common stack between two stacks or undefined'(test) {
        const a = new lib_1.Stack(undefined, 'a');
        const aa = new Nested(a, 'aa');
        const ab = new Nested(a, 'ab');
        const aaa = new Nested(aa, 'aaa');
        const aab = new Nested(aa, 'aab');
        const aba = new Nested(ab, 'aba');
        const b = new lib_1.Stack(undefined, 'b');
        const ba = new Nested(b, 'ba');
        const baa = new Nested(ba, 'baa');
        test.equal(lca(a, b), undefined);
        test.equal(lca(aa, ab), 'a');
        test.equal(lca(ab, aa), 'a');
        test.equal(lca(aa, aba), 'a');
        test.equal(lca(aba, aa), 'a');
        test.equal(lca(ab, aba), 'ab');
        test.equal(lca(aba, ab), 'ab');
        test.equal(lca(aba, aba), 'aba');
        test.equal(lca(aa, aa), 'aa');
        test.equal(lca(a, aaa), 'a');
        test.equal(lca(aaa, aab), 'aa');
        test.equal(lca(aaa, b), undefined);
        test.equal(lca(aaa, ba), undefined);
        test.equal(lca(baa, ba), 'ba');
        test.done();
        function lca(s1, s2) {
            const res = util_1.findLastCommonElement(util_1.pathToTopLevelStack(s1), util_1.pathToTopLevelStack(s2));
            if (!res) {
                return undefined;
            }
            return res.node.id;
        }
    },
});
//# sourceMappingURL=data:application/json;base64,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