"use strict";
const lib_1 = require("../lib");
const encoding_1 = require("../lib/private/encoding");
const intrinsic_1 = require("../lib/private/intrinsic");
const resolve_1 = require("../lib/private/resolve");
const evaluate_cfn_1 = require("./evaluate-cfn");
const util_1 = require("./util");
class Promise2 {
    constructor() {
        this.creationStack = [];
    }
    resolve() {
        return {
            Data: {
                stringProp: 'hello',
                numberProp: 1234,
            },
            Recurse: new intrinsic_1.Intrinsic(42),
        };
    }
}
class Promise1 {
    constructor() {
        this.creationStack = [];
        this.p2 = [new Promise2(), new Promise2()];
    }
    resolve() {
        return this.p2;
    }
}
class BaseDataType {
    constructor(foo) {
        this.foo = foo;
    }
}
class DataType extends BaseDataType {
    constructor() {
        super(12);
        this.goo = 'hello';
    }
}
/**
 * Return Tokens in both flavors that resolve to the given string
 */
function tokensThatResolveTo(value) {
    return [
        new intrinsic_1.Intrinsic(value),
        lib_1.Lazy.anyValue({ produce: () => value }),
    ];
}
/**
 * Wrapper for resolve that creates an throwaway Construct to call it on
 *
 * So I don't have to change all call sites in this file.
 */
function resolve(x) {
    return new lib_1.Stack().resolve(x);
}
module.exports = {
    'resolve a plain old object should just return the object'(test) {
        const obj = { PlainOldObject: 123, Array: [1, 2, 3] };
        test.deepEqual(resolve(obj), obj);
        test.done();
    },
    'if a value is an object with a token value, it will be evaluated'(test) {
        const obj = {
            RegularValue: 'hello',
            LazyValue: new intrinsic_1.Intrinsic('World'),
        };
        test.deepEqual(resolve(obj), {
            RegularValue: 'hello',
            LazyValue: 'World',
        });
        test.done();
    },
    'tokens are evaluated anywhere in the object tree'(test) {
        const obj = new Promise1();
        const actual = resolve({ Obj: obj });
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
            ],
        });
        test.done();
    },
    'tokens are evaluated recursively'(test) {
        const obj = new Promise1();
        const actual = resolve(new intrinsic_1.Intrinsic({ Obj: obj }));
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
            ],
        });
        test.done();
    },
    'empty arrays or objects are kept'(test) {
        test.deepEqual(resolve({}), {});
        test.deepEqual(resolve([]), []);
        const obj = {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [undefined, undefined],
                    PropD: 'Yoohoo',
                },
            },
        };
        test.deepEqual(resolve(obj), {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [],
                    PropD: 'Yoohoo',
                },
            },
        });
        test.done();
    },
    'if an object has a "resolve" property that is not a function, it is not considered a token'(test) {
        test.deepEqual(resolve({ a_token: { resolve: () => 78787 } }), { a_token: 78787 });
        test.deepEqual(resolve({ not_a_token: { resolve: 12 } }), { not_a_token: { resolve: 12 } });
        test.done();
    },
    // eslint-disable-next-line max-len
    'if a resolvable object inherits from a class that is also resolvable, the "constructor" function will not get in the way (uses Object.keys instead of "for in")'(test) {
        test.deepEqual(resolve({ prop: new DataType() }), { prop: { foo: 12, goo: 'hello' } });
        test.done();
    },
    'isToken(obj) can be used to determine if an object is a token'(test) {
        test.ok(lib_1.isResolvableObject({ resolve: () => 123 }));
        test.ok(lib_1.isResolvableObject({ a: 1, b: 2, resolve: () => 'hello' }));
        test.ok(!lib_1.isResolvableObject({ a: 1, b: 2, resolve: 3 }));
        test.done();
    },
    'Token can be used to create tokens that contain a constant value'(test) {
        test.equal(resolve(new intrinsic_1.Intrinsic(12)), 12);
        test.equal(resolve(new intrinsic_1.Intrinsic('hello')), 'hello');
        test.deepEqual(resolve(new intrinsic_1.Intrinsic(['hi', 'there'])), ['hi', 'there']);
        test.done();
    },
    'resolving leaves a Date object in working order'(test) {
        const date = new Date('2000-01-01');
        const resolved = resolve(date);
        test.equal(date.toString(), resolved.toString());
        test.done();
    },
    'tokens can be stringified and evaluated to conceptual value'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('woof woof');
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'The dog says: woof woof');
        test.done();
    },
    'tokens stringification can be reversed'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('woof woof');
        // THEN
        test.equal(token, lib_1.Tokenization.reverseString(`${token}`).firstToken);
        test.done();
    },
    'Tokens stringification and reversing of CloudFormation Tokens is implemented using Fn::Join'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic(({ woof: 'woof' }));
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(resolved, {
            'Fn::Join': ['', ['The dog says: ', { woof: 'woof' }]],
        });
        test.done();
    },
    'Doubly nested strings evaluate correctly in scalar context'(test) {
        // GIVEN
        const token1 = new intrinsic_1.Intrinsic('world');
        const token2 = new intrinsic_1.Intrinsic(`hello ${token1}`);
        // WHEN
        const resolved1 = resolve(token2.toString());
        const resolved2 = resolve(token2);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved1), 'hello world');
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved2), 'hello world');
        test.done();
    },
    'integer Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const token of tokensThatResolveTo(1)) {
            // WHEN
            const stringified = `the number is ${token}`;
            const resolved = resolve(stringified);
            // THEN
            test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'the number is 1');
        }
        test.done();
    },
    'intrinsic Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const bucketName of tokensThatResolveTo({ Ref: 'MyBucket' })) {
            // WHEN
            const resolved = resolve(`my bucket is named ${bucketName}`);
            // THEN
            const context = { MyBucket: 'TheName' };
            test.equal(evaluate_cfn_1.evaluateCFN(resolved, context), 'my bucket is named TheName');
        }
        test.done();
    },
    'tokens resolve properly in initial position'(test) {
        // GIVEN
        for (const token of tokensThatResolveTo('Hello')) {
            // WHEN
            const resolved = resolve(`${token} world`);
            // THEN
            test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
        }
        test.done();
    },
    'side-by-side Tokens resolve correctly'(test) {
        // GIVEN
        for (const token1 of tokensThatResolveTo('Hello ')) {
            for (const token2 of tokensThatResolveTo('world')) {
                // WHEN
                const resolved = resolve(`${token1}${token2}`);
                // THEN
                test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
            }
        }
        test.done();
    },
    'tokens can be used in hash keys but must resolve to a string'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('I am a string');
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`,
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested in hash keys'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic(lib_1.Lazy.stringValue({ produce: () => lib_1.Lazy.stringValue({ produce: (() => 'I am a string') }) }));
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`,
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested and concatenated in hash keys'(test) {
        // GIVEN
        const innerToken = new intrinsic_1.Intrinsic('toot');
        const token = new intrinsic_1.Intrinsic(`${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: 'boom chicago',
        };
        // THEN
        test.deepEqual(resolve(s), { 'toot the woot': 'boom chicago' });
        test.done();
    },
    'can find nested tokens in hash keys'(test) {
        // GIVEN
        const innerToken = new intrinsic_1.Intrinsic('toot');
        const token = new intrinsic_1.Intrinsic(`${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: 'boom chicago',
        };
        // THEN
        const tokens = resolve_1.findTokens(new lib_1.Stack(), () => s);
        test.ok(tokens.some(t => t === innerToken), 'Cannot find innerToken');
        test.ok(tokens.some(t => t === token), 'Cannot find token');
        test.done();
    },
    'fails if token in a hash key resolves to a non-string'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`,
        };
        // THEN
        test.throws(() => resolve(s), 'The key "${Token[TOKEN.19]}" has been resolved to {"Ref":"Other"} but must be resolvable to a string');
        test.done();
    },
    'list encoding': {
        'can encode Token to string and resolve the encoding'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const struct = {
                XYZ: lib_1.Token.asList(token),
            };
            // THEN
            test.deepEqual(resolve(struct), {
                XYZ: { Ref: 'Other' },
            });
            test.done();
        },
        'cannot add to encoded list'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const encoded = lib_1.Token.asList(token);
            encoded.push('hello');
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /Cannot add elements to list token/);
            test.done();
        },
        'cannot add to strings in encoded list'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const encoded = lib_1.Token.asList(token);
            encoded[0] += 'hello';
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /concatenate strings in/);
            test.done();
        },
        'can pass encoded lists to FnSelect'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.select(1, encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Select': [1, { Ref: 'Other' }],
            });
            test.done();
        },
        'can pass encoded lists to FnJoin'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.join('/', encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }],
            });
            test.done();
        },
        'can pass encoded lists to FnJoin, even if join is stringified'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.join('/', encoded).toString();
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }],
            });
            test.done();
        },
    },
    'number encoding': {
        'basic integer encoding works'(test) {
            test.equal(16, encoding_1.extractTokenDouble(encoding_1.createTokenDouble(16)));
            test.done();
        },
        'arbitrary integers can be encoded, stringified, and recovered'(test) {
            for (let i = 0; i < 100; i++) {
                // We can encode all numbers up to 2^48-1
                const x = Math.floor(Math.random() * (Math.pow(2, 48) - 1));
                const encoded = encoding_1.createTokenDouble(x);
                // Roundtrip through JSONification
                const roundtripped = JSON.parse(JSON.stringify({ theNumber: encoded })).theNumber;
                const decoded = encoding_1.extractTokenDouble(roundtripped);
                test.equal(decoded, x, `Fail roundtrip encoding of ${x}`);
            }
            test.done();
        },
        'arbitrary numbers are correctly detected as not being tokens'(test) {
            test.equal(undefined, encoding_1.extractTokenDouble(0));
            test.equal(undefined, encoding_1.extractTokenDouble(1243));
            test.equal(undefined, encoding_1.extractTokenDouble(4835e+532));
            test.done();
        },
        'can number-encode and resolve Token objects'(test) {
            // GIVEN
            const x = new intrinsic_1.Intrinsic(123);
            // THEN
            const encoded = lib_1.Token.asNumber(x);
            test.equal(false, lib_1.isResolvableObject(encoded), 'encoded number does not test as token');
            test.equal(true, lib_1.Token.isUnresolved(encoded), 'encoded number does not test as token');
            // THEN
            const resolved = resolve({ value: encoded });
            test.deepEqual(resolved, { value: 123 });
            test.done();
        },
    },
    'stack trace is captured at token creation'(test) {
        function fn1() {
            function fn2() {
                class ExposeTrace extends intrinsic_1.Intrinsic {
                    get creationTrace() {
                        return this.creationStack;
                    }
                }
                return new ExposeTrace('hello');
            }
            return fn2();
        }
        const previousValue = util_1.reEnableStackTraceCollection();
        const token = fn1();
        util_1.restoreStackTraceColection(previousValue);
        test.ok(token.creationTrace.find(x => x.includes('fn1')));
        test.ok(token.creationTrace.find(x => x.includes('fn2')));
        test.done();
    },
    'newError returns an error with the creation stack trace'(test) {
        function fn1() {
            function fn2() {
                function fn3() {
                    class ThrowingToken extends intrinsic_1.Intrinsic {
                        throwError(message) {
                            throw this.newError(message);
                        }
                    }
                    return new ThrowingToken('boom');
                }
                return fn3();
            }
            return fn2();
        }
        const previousValue = util_1.reEnableStackTraceCollection();
        const token = fn1();
        util_1.restoreStackTraceColection(previousValue);
        test.throws(() => token.throwError('message!'), /Token created:/);
        test.done();
    },
    'type coercion': (() => {
        const tests = {};
        const inputs = [
            'a string',
            1234,
            { an_object: 1234 },
            [1, 2, 3],
            false,
        ];
        for (const input of inputs) {
            // GIVEN
            const stringToken = lib_1.Token.asString(new intrinsic_1.Intrinsic(input));
            const numberToken = lib_1.Token.asNumber(new intrinsic_1.Intrinsic(input));
            const listToken = lib_1.Token.asList(new intrinsic_1.Intrinsic(input));
            // THEN
            const expected = input;
            tests[`${input}<string>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(stringToken))), expected);
                test.done();
            };
            tests[`${input}<list>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(listToken))), expected);
                test.done();
            };
            tests[`${input}<number>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(numberToken))), expected);
                test.done();
            };
            tests[`${input}<string>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(stringToken).toString()), expected);
                test.done();
            };
            tests[`${input}<list>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(listToken).toString()), expected);
                test.done();
            };
            tests[`${input}<number>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(numberToken).toString()), expected);
                test.done();
            };
            tests[`${input}<string>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(stringToken))), expected);
                test.done();
            };
            tests[`${input}<list>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(listToken))), expected);
                test.done();
            };
            tests[`${input}<number>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(numberToken))), expected);
                test.done();
            };
        }
        return tests;
    })(),
    'creation stack is attached to errors emitted during resolve'(test) {
        function showMeInTheStackTrace() {
            return lib_1.Lazy.stringValue({ produce: () => { throw new Error('fooError'); } });
        }
        const previousValue = util_1.reEnableStackTraceCollection();
        const x = showMeInTheStackTrace();
        let message;
        try {
            resolve(x);
        }
        catch (e) {
            message = e.message;
        }
        finally {
            util_1.restoreStackTraceColection(previousValue);
        }
        test.ok(message && message.includes('showMeInTheStackTrace'));
        test.done();
    },
    'stringifyNumber': {
        'converts number to string'(test) {
            test.equal(lib_1.Tokenization.stringifyNumber(100), '100');
            test.done();
        },
        'converts tokenized number to string'(test) {
            test.equal(resolve(lib_1.Tokenization.stringifyNumber({
                resolve: () => 100,
            })), '100');
            test.done();
        },
        'string remains the same'(test) {
            test.equal(lib_1.Tokenization.stringifyNumber('123'), '123');
            test.done();
        },
        'Ref remains the same'(test) {
            const val = { Ref: 'SomeLogicalId' };
            test.deepEqual(lib_1.Tokenization.stringifyNumber(val), val);
            test.done();
        },
        'lazy Ref remains the same'(test) {
            const resolvedVal = { Ref: 'SomeLogicalId' };
            const tokenizedVal = lib_1.Lazy.anyValue({
                produce: () => resolvedVal,
            });
            const res = lib_1.Tokenization.stringifyNumber(tokenizedVal);
            test.notDeepEqual(res, resolvedVal);
            test.deepEqual(resolve(res), resolvedVal);
            test.done();
        },
        'tokenized Ref remains the same'(test) {
            const resolvedVal = { Ref: 'SomeLogicalId' };
            const tokenizedVal = lib_1.Token.asNumber(resolvedVal);
            const res = lib_1.Tokenization.stringifyNumber(tokenizedVal);
            test.notDeepEqual(res, resolvedVal);
            test.deepEqual(resolve(res), resolvedVal);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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