"use strict";
const cfn_resource_1 = require("../lib/cfn-resource");
const tag_manager_1 = require("../lib/tag-manager");
module.exports = {
    'TagManagerOptions can set tagPropertyName'(test) {
        const tagPropName = 'specialName';
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.MAP, 'Foo', undefined, { tagPropertyName: tagPropName });
        test.deepEqual(mgr.tagPropertyName, tagPropName);
        test.done();
    },
    '#setTag() supports setting a tag regardless of Type'(test) {
        const notTaggable = new tag_manager_1.TagManager(cfn_resource_1.TagType.NOT_TAGGABLE, 'AWS::Resource::Type');
        notTaggable.setTag('key', 'value');
        test.deepEqual(notTaggable.renderTags(), undefined);
        test.done();
    },
    'when a tag does not exist': {
        '#removeTag() does not throw an error'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.doesNotThrow(() => (mgr.removeTag('dne', 0)));
            test.done();
        },
        '#setTag() creates the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'notanymore' }]);
            test.done();
        },
    },
    'when a tag does exist': {
        '#removeTag() deletes the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore', 0);
            mgr.removeTag('dne', 0);
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
        '#setTag() overwrites the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            mgr.setTag('dne', 'iwin');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'iwin' }]);
            test.done();
        },
    },
    'when there are no tags': {
        '#renderTags() returns undefined'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
        '#hasTags() returns false'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.equal(mgr.hasTags(), false);
            test.done();
        },
    },
    '#renderTags() handles standard, map, keyValue, and ASG tag formats'(test) {
        const tagged = [];
        const standard = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        const asg = new tag_manager_1.TagManager(cfn_resource_1.TagType.AUTOSCALING_GROUP, 'AWS::Resource::Type');
        const keyValue = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'AWS::Resource::Type');
        const mapper = new tag_manager_1.TagManager(cfn_resource_1.TagType.MAP, 'AWS::Resource::Type');
        tagged.push(standard);
        tagged.push(asg);
        tagged.push(keyValue);
        tagged.push(mapper);
        for (const res of tagged) {
            res.setTag('foo', 'bar');
            res.setTag('asg', 'only', 0, false);
        }
        test.deepEqual(standard.renderTags(), [
            { key: 'asg', value: 'only' },
            { key: 'foo', value: 'bar' },
        ]);
        test.deepEqual(asg.renderTags(), [
            { key: 'asg', value: 'only', propagateAtLaunch: false },
            { key: 'foo', value: 'bar', propagateAtLaunch: true },
        ]);
        test.deepEqual(keyValue.renderTags(), [
            { Key: 'asg', Value: 'only' },
            { Key: 'foo', Value: 'bar' },
        ]);
        test.deepEqual(mapper.renderTags(), {
            foo: 'bar',
            asg: 'only',
        });
        test.done();
    },
    'when there are tags it hasTags returns true'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        mgr.setTag('key', 'myVal', 2);
        mgr.setTag('key', 'newVal', 1);
        test.equal(mgr.hasTags(), true);
        test.done();
    },
    'tags with higher or equal priority always take precedence'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        mgr.setTag('key', 'myVal', 2);
        mgr.setTag('key', 'newVal', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 2);
        test.deepEqual(mgr.renderTags(), undefined);
        test.done();
    },
    'tags are always ordered by key name'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        mgr.setTag('key', 'foo');
        mgr.setTag('aardvark', 'zebra');
        mgr.setTag('name', 'test');
        test.deepEqual(mgr.renderTags(), [
            { key: 'aardvark', value: 'zebra' },
            { key: 'key', value: 'foo' },
            { key: 'name', value: 'test' },
        ]);
        mgr.setTag('myKey', 'myVal');
        test.deepEqual(mgr.renderTags(), [
            { key: 'aardvark', value: 'zebra' },
            { key: 'key', value: 'foo' },
            { key: 'myKey', value: 'myVal' },
            { key: 'name', value: 'test' },
        ]);
        test.done();
    },
    'excludeResourceTypes only tags resources that do not match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Fake::Resource');
        test.equal(false, mgr.applyTagAspectHere([], ['AWS::Fake::Resource']));
        test.equal(true, mgr.applyTagAspectHere([], ['AWS::Wrong::Resource']));
        test.done();
    },
    'includeResourceTypes only tags resources that match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Fake::Resource');
        test.equal(true, mgr.applyTagAspectHere(['AWS::Fake::Resource'], []));
        test.equal(false, mgr.applyTagAspectHere(['AWS::Wrong::Resource'], []));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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