"use strict";
const lib_1 = require("../lib");
const synthesis_1 = require("../lib/private/synthesis");
class TaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.STANDARD, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class AsgTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.AUTOSCALING_GROUP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class MapTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.MAP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
module.exports = {
    'Tag visit all children of the applied node'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Aspects.of(res).add(new lib_1.Tag('foo', 'bar'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(map.tags.renderTags(), { foo: 'bar' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'foo', value: 'bar', propagateAtLaunch: true }]);
        test.done();
    },
    'The last aspect applied takes precedence'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Aspects.of(res).add(new lib_1.Tag('foo', 'bar'));
        lib_1.Aspects.of(res).add(new lib_1.Tag('foo', 'foobar'));
        lib_1.Aspects.of(res).add(new lib_1.Tag('foo', 'baz'));
        lib_1.Aspects.of(res2).add(new lib_1.Tag('foo', 'good'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'baz' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'good' }]);
        test.done();
    },
    'RemoveTag will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Aspects.of(root).add(new lib_1.Tag('root', 'was here'));
        lib_1.Aspects.of(res).add(new lib_1.Tag('first', 'there is only 1'));
        lib_1.Aspects.of(res).add(new lib_1.RemoveTag('root'));
        lib_1.Aspects.of(res).add(new lib_1.RemoveTag('doesnotexist'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'add will add a tag and remove will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Tags.of(root).add('root', 'was here');
        lib_1.Tags.of(res).add('first', 'there is only 1');
        lib_1.Tags.of(res).remove('root');
        lib_1.Tags.of(res).remove('doesnotexist');
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'the #visit function is idempotent'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Aspects.of(res).add(new lib_1.Tag('foo', 'bar'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.done();
    },
    'removeTag Aspects by default will override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Aspects.of(res).add(new lib_1.RemoveTag('key'));
        lib_1.Aspects.of(res2).add(new lib_1.Tag('key', 'value'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), undefined);
        test.done();
    },
    'removeTag Aspects with priority 0 will not override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Aspects.of(res).add(new lib_1.RemoveTag('key', { priority: 0 }));
        lib_1.Aspects.of(res2).add(new lib_1.Tag('key', 'value'));
        synthesis_1.synthesize(root);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'key', value: 'value' }]);
        test.done();
    },
    'Aspects are merged with tags created by L1 Constructor'(test) {
        const root = new lib_1.Stack();
        const aspectBranch = new TaggableResource(root, 'FakeBranchA', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite' },
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        const asgResource = new AsgTaggableResource(aspectBranch, 'FakeAsg', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite', propagateAtLaunch: false },
                    { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                ],
            },
        });
        const mapTaggable = new MapTaggableResource(aspectBranch, 'FakeSam', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: {
                    aspects: 'overwrite',
                    cfn: 'is cool',
                },
            },
        });
        const cfnBranch = new TaggableResource(root, 'FakeBranchB', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        lib_1.Aspects.of(aspectBranch).add(new lib_1.Tag('aspects', 'rule'));
        synthesis_1.synthesize(root);
        test.deepEqual(aspectBranch.testProperties().tags, [{ key: 'aspects', value: 'rule' }, { key: 'cfn', value: 'is cool' }]);
        test.deepEqual(asgResource.testProperties().tags, [
            { key: 'aspects', value: 'rule', propagateAtLaunch: true },
            { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
        ]);
        test.deepEqual(mapTaggable.testProperties().tags, {
            aspects: 'rule',
            cfn: 'is cool',
        });
        test.deepEqual(cfnBranch.testProperties().tags, [{ key: 'cfn', value: 'is cool' }]);
        test.done();
    },
    'when invalid tag properties are passed from L1s': {
        'map passed instead of array it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new TaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                        },
                    },
                });
            });
            test.throws(() => {
                new AsgTaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                            propagateAtLaunch: true,
                        },
                    },
                });
            });
            test.done();
        },
        'if array is passed instead of map it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new MapTaggableResource(root, 'FakeSam', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: [
                            { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                            { key: 'aspects', value: 'overwrite' },
                        ],
                    },
                });
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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