"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
class TouchingAspect {
    constructor() {
        this.visits = new Array();
    }
    visit(node) {
        this.visits.push(node);
    }
}
class BogusStack extends lib_1.Stack {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'Resource', {
            type: 'CDK::Test::Resource',
        });
    }
}
function acctRegion(s) {
    return [s.account, s.region];
}
module.exports = {
    'Stack inherits unspecified part of the env from Stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'Stage', {
            env: { account: 'account', region: 'region' },
        });
        // WHEN
        const stack1 = new lib_1.Stack(stage, 'Stack1', { env: { region: 'elsewhere' } });
        const stack2 = new lib_1.Stack(stage, 'Stack2', { env: { account: 'tnuocca' } });
        // THEN
        test.deepEqual(acctRegion(stack1), ['account', 'elsewhere']);
        test.deepEqual(acctRegion(stack2), ['tnuocca', 'region']);
        test.done();
    },
    'envs are inherited deeply'(test) {
        // GIVEN
        const app = new lib_1.App();
        const outer = new lib_1.Stage(app, 'Stage', {
            env: { account: 'account', region: 'region' },
        });
        // WHEN
        const innerAcct = new lib_1.Stage(outer, 'Acct', { env: { account: 'tnuocca' } });
        const innerRegion = new lib_1.Stage(outer, 'Rgn', { env: { region: 'elsewhere' } });
        const innerNeither = new lib_1.Stage(outer, 'Neither');
        // THEN
        test.deepEqual(acctRegion(new lib_1.Stack(innerAcct, 'Stack')), ['tnuocca', 'region']);
        test.deepEqual(acctRegion(new lib_1.Stack(innerRegion, 'Stack')), ['account', 'elsewhere']);
        test.deepEqual(acctRegion(new lib_1.Stack(innerNeither, 'Stack')), ['account', 'region']);
        test.done();
    },
    'The Stage Assembly is in the app Assembly\'s manifest'(test) {
        // WHEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'Stage');
        new BogusStack(stage, 'Stack2');
        // THEN -- app manifest contains a nested cloud assembly
        const appAsm = app.synth();
        const artifact = appAsm.artifacts.find(x => x instanceof cxapi.NestedCloudAssemblyArtifact);
        test.ok(artifact);
        test.done();
    },
    'Stacks in Stage are in a different cxasm than Stacks in App'(test) {
        // WHEN
        const app = new lib_1.App();
        const stack1 = new BogusStack(app, 'Stack1');
        const stage = new lib_1.Stage(app, 'Stage');
        const stack2 = new BogusStack(stage, 'Stack2');
        // THEN
        const stageAsm = stage.synth();
        test.deepEqual(stageAsm.stacks.map(s => s.stackName), [stack2.stackName]);
        const appAsm = app.synth();
        test.deepEqual(appAsm.stacks.map(s => s.stackName), [stack1.stackName]);
        test.done();
    },
    'Can nest Stages inside other Stages'(test) {
        // WHEN
        const app = new lib_1.App();
        const outer = new lib_1.Stage(app, 'Outer');
        const inner = new lib_1.Stage(outer, 'Inner');
        const stack = new BogusStack(inner, 'Stack');
        // WHEN
        const appAsm = app.synth();
        const outerAsm = appAsm.getNestedAssembly(outer.artifactId);
        const innerAsm = outerAsm.getNestedAssembly(inner.artifactId);
        test.ok(innerAsm.tryGetArtifact(stack.artifactId));
        test.done();
    },
    'Default stack name in Stage objects incorporates the Stage name and no hash'(test) {
        // WHEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'MyStack');
        // THEN
        test.equal(stage.stageName, 'MyStage');
        test.equal(stack.stackName, 'MyStage-MyStack');
        test.done();
    },
    'Can not have dependencies to stacks outside the nested asm'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new BogusStack(app, 'Stack1');
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack2 = new BogusStack(stage, 'Stack2');
        // WHEN
        test.throws(() => {
            stack2.addDependency(stack1);
        }, /dependency cannot cross stage boundaries/);
        test.done();
    },
    'When we synth() a stage, prepare must be called on constructs in the stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        let prepared = false;
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'Stack');
        class HazPrepare extends lib_1.Construct {
            prepare() {
                prepared = true;
            }
        }
        new HazPrepare(stack, 'Preparable');
        // WHEN
        stage.synth();
        // THEN
        test.equals(prepared, true);
        test.done();
    },
    'When we synth() a stage, aspects inside it must have been applied'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'Stack');
        // WHEN
        const aspect = new TouchingAspect();
        lib_1.Aspects.of(stack).add(aspect);
        // THEN
        app.synth();
        test.deepEqual(aspect.visits.map(c => c.node.path), [
            'MyStage/Stack',
            'MyStage/Stack/Resource',
        ]);
        test.done();
    },
    'Aspects do not apply inside a Stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        new BogusStack(stage, 'Stack');
        // WHEN
        const aspect = new TouchingAspect();
        lib_1.Aspects.of(app).add(aspect);
        // THEN
        app.synth();
        test.deepEqual(aspect.visits.map(c => c.node.path), [
            '',
            'Tree',
        ]);
        test.done();
    },
    'Automatic dependencies inside a stage are available immediately after synth'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack1 = new lib_1.Stack(stage, 'Stack1');
        const stack2 = new lib_1.Stack(stage, 'Stack2');
        // WHEN
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', {
            type: 'CDK::Test::Resource',
        });
        new lib_1.CfnResource(stack2, 'Resource', {
            type: 'CDK::Test::Resource',
            properties: {
                OtherThing: resource1.ref,
            },
        });
        const asm = stage.synth();
        // THEN
        test.deepEqual(asm.getStackArtifact(stack2.artifactId).dependencies.map(d => d.id), [stack1.artifactId]);
        test.done();
    },
    'Assemblies can be deeply nested'(test) {
        // GIVEN
        const app = new lib_1.App({ treeMetadata: false });
        const level1 = new lib_1.Stage(app, 'StageLevel1');
        const level2 = new lib_1.Stage(level1, 'StageLevel2');
        new lib_1.Stage(level2, 'StageLevel3');
        // WHEN
        const rootAssembly = app.synth();
        // THEN
        test.deepEqual(rootAssembly.manifest.artifacts, {
            'assembly-StageLevel1': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1',
                    displayName: 'StageLevel1',
                },
            },
        });
        const assemblyLevel1 = rootAssembly.getNestedAssembly('assembly-StageLevel1');
        test.deepEqual(assemblyLevel1.manifest.artifacts, {
            'assembly-StageLevel1-StageLevel2': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1-StageLevel2',
                    displayName: 'StageLevel1/StageLevel2',
                },
            },
        });
        const assemblyLevel2 = assemblyLevel1.getNestedAssembly('assembly-StageLevel1-StageLevel2');
        test.deepEqual(assemblyLevel2.manifest.artifacts, {
            'assembly-StageLevel1-StageLevel2-StageLevel3': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1-StageLevel2-StageLevel3',
                    displayName: 'StageLevel1/StageLevel2/StageLevel3',
                },
            },
        });
        test.done();
    },
    'stage name validation'(test) {
        const app = new lib_1.App();
        new lib_1.Stage(app, 'abcd');
        new lib_1.Stage(app, 'abcd123');
        new lib_1.Stage(app, 'abcd123-588dfjjk');
        new lib_1.Stage(app, 'abcd123-588dfjjk.sss');
        new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid');
        test.throws(() => new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid '), /invalid stage name "abcd123-588dfjjk.sss_ajsid "/);
        test.throws(() => new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid/dfo'), /invalid stage name "abcd123-588dfjjk.sss_ajsid\/dfo"/);
        test.throws(() => new lib_1.Stage(app, '&'), /invalid stage name "&"/);
        test.throws(() => new lib_1.Stage(app, '45hello'), /invalid stage name "45hello"/);
        test.throws(() => new lib_1.Stage(app, 'f'), /invalid stage name "f"/);
        test.done();
    },
    'outdir cannot be specified for nested stages'(test) {
        // WHEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stage(app, 'mystage', { outdir: '/tmp/foo/bar' }), /"outdir" cannot be specified for nested stages/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zdGFnZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3Quc3RhZ2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLHlDQUF5QztBQUV6QyxnQ0FBaUc7QUEwUmpHLE1BQU0sY0FBYztJQUFwQjtRQUNrQixXQUFNLEdBQUcsSUFBSSxLQUFLLEVBQWMsQ0FBQztJQUluRCxDQUFDO0lBSFEsS0FBSyxDQUFDLElBQWdCO1FBQzNCLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3pCLENBQUM7Q0FDRjtBQUVELE1BQU0sVUFBVyxTQUFRLFdBQUs7SUFDNUIsWUFBWSxLQUFnQixFQUFFLEVBQVU7UUFDdEMsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLGlCQUFXLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNoQyxJQUFJLEVBQUUscUJBQXFCO1NBQzVCLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQUVELFNBQVMsVUFBVSxDQUFDLENBQVE7SUFDMUIsT0FBTyxDQUFDLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQy9CLENBQUM7QUEzU0QsaUJBQVM7SUFDUCx1REFBdUQsQ0FBQyxJQUFVO1FBQ2hFLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUU7WUFDcEMsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFO1NBQzlDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUM1RSxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUUzRSxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUM3RCxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRTFELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyQkFBMkIsQ0FBQyxJQUFVO1FBQ3BDLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUU7WUFDcEMsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFO1NBQzlDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLFNBQVMsR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUM1RSxNQUFNLFdBQVcsR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUM5RSxNQUFNLFlBQVksR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFakQsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFDakYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsSUFBSSxXQUFLLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUN0RixJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRXBGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1REFBdUQsQ0FBQyxJQUFVO1FBQ2hFLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN0QyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFaEMsd0RBQXdEO1FBQ3hELE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUUzQixNQUFNLFFBQVEsR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsWUFBWSxLQUFLLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUM1RixJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRWxCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2REFBNkQsQ0FBQyxJQUFVO1FBQ3RFLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksVUFBVSxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDdEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxVQUFVLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRS9DLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDL0IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBRTFFLE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUMzQixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7UUFFeEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFDQUFxQyxDQUFDLElBQVU7UUFDOUMsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3RDLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFN0MsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUMzQixNQUFNLFFBQVEsR0FBRyxNQUFNLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzVELE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFOUQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBRW5ELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2RUFBNkUsQ0FBQyxJQUFVO1FBQ3RGLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFL0MsT0FBTztRQUNQLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztRQUUvQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNERBQTRELENBQUMsSUFBVTtRQUNyRSxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLE1BQU0sR0FBRyxJQUFJLFVBQVUsQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDN0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksVUFBVSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUUvQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixNQUFNLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9CLENBQUMsRUFBRSwwQ0FBMEMsQ0FBQyxDQUFDO1FBRS9DLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0RUFBNEUsQ0FBQyxJQUFVO1FBQ3JGLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLElBQUksUUFBUSxHQUFHLEtBQUssQ0FBQztRQUNyQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDeEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzdDLE1BQU0sVUFBVyxTQUFRLGVBQVM7WUFDdEIsT0FBTztnQkFDZixRQUFRLEdBQUcsSUFBSSxDQUFDO1lBQ2xCLENBQUM7U0FDRjtRQUNELElBQUksVUFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztRQUVwQyxPQUFPO1FBQ1AsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWQsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBRTVCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtRUFBbUUsQ0FBQyxJQUFVO1FBQzVFLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFN0MsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxFQUFFLENBQUM7UUFDcEMsYUFBTyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFOUIsT0FBTztRQUNQLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUNaLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2xELGVBQWU7WUFDZix3QkFBd0I7U0FDekIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFDQUFxQyxDQUFDLElBQVU7UUFDOUMsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hDLElBQUksVUFBVSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUUvQixPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxjQUFjLEVBQUUsQ0FBQztRQUNwQyxhQUFPLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUU1QixPQUFPO1FBQ1AsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ1osSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDbEQsRUFBRTtZQUNGLE1BQU07U0FDUCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkVBQTZFLENBQUMsSUFBVTtRQUN0RixRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDeEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzFDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUUxQyxPQUFPO1FBQ1AsTUFBTSxTQUFTLEdBQUcsSUFBSSxpQkFBVyxDQUFDLE1BQU0sRUFBRSxVQUFVLEVBQUU7WUFDcEQsSUFBSSxFQUFFLHFCQUFxQjtTQUM1QixDQUFDLENBQUM7UUFDSCxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRTtZQUNsQyxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLFVBQVUsRUFBRTtnQkFDVixVQUFVLEVBQUUsU0FBUyxDQUFDLEdBQUc7YUFDMUI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFMUIsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQ1osR0FBRyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUNuRSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBRXZCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpQ0FBaUMsQ0FBQyxJQUFVO1FBQzFDLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxFQUFFLFlBQVksRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBRTdDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUM3QyxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDaEQsSUFBSSxXQUFLLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRWpDLE9BQU87UUFDUCxNQUFNLFlBQVksR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFakMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUU7WUFDOUMsc0JBQXNCLEVBQUU7Z0JBQ3RCLElBQUksRUFBRSxvQkFBb0I7Z0JBQzFCLFVBQVUsRUFBRTtvQkFDVixhQUFhLEVBQUUsc0JBQXNCO29CQUNyQyxXQUFXLEVBQUUsYUFBYTtpQkFDM0I7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQzlFLElBQUksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUU7WUFDaEQsa0NBQWtDLEVBQUU7Z0JBQ2xDLElBQUksRUFBRSxvQkFBb0I7Z0JBQzFCLFVBQVUsRUFBRTtvQkFDVixhQUFhLEVBQUUsa0NBQWtDO29CQUNqRCxXQUFXLEVBQUUseUJBQXlCO2lCQUN2QzthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxjQUFjLEdBQUcsY0FBYyxDQUFDLGlCQUFpQixDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDNUYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtZQUNoRCw4Q0FBOEMsRUFBRTtnQkFDOUMsSUFBSSxFQUFFLG9CQUFvQjtnQkFDMUIsVUFBVSxFQUFFO29CQUNWLGFBQWEsRUFBRSw4Q0FBOEM7b0JBQzdELFdBQVcsRUFBRSxxQ0FBcUM7aUJBQ25EO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdUJBQXVCLENBQUMsSUFBVTtRQUNoQyxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBRXRCLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN2QixJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDMUIsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDbkMsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFDdkMsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLDRCQUE0QixDQUFDLENBQUM7UUFFN0MsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsNkJBQTZCLENBQUMsRUFBRSxrREFBa0QsQ0FBQyxDQUFDO1FBQ3JILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLGdDQUFnQyxDQUFDLEVBQUUsc0RBQXNELENBQUMsQ0FBQztRQUM1SCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxFQUFFLDhCQUE4QixDQUFDLENBQUM7UUFDN0UsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztRQUVqRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOENBQThDLENBQUMsSUFBVTtRQUN2RCxPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUV0QixPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLEVBQUUsTUFBTSxFQUFFLGNBQWMsRUFBRSxDQUFDLEVBQUUsZ0RBQWdELENBQUMsQ0FBQztRQUMzSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGN4YXBpIGZyb20gJ0Bhd3MtY2RrL2N4LWFwaSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQXBwLCBDZm5SZXNvdXJjZSwgQ29uc3RydWN0LCBJQXNwZWN0LCBJQ29uc3RydWN0LCBTdGFjaywgU3RhZ2UsIEFzcGVjdHMgfSBmcm9tICcuLi9saWInO1xuXG5leHBvcnQgPSB7XG4gICdTdGFjayBpbmhlcml0cyB1bnNwZWNpZmllZCBwYXJ0IG9mIHRoZSBlbnYgZnJvbSBTdGFnZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlKGFwcCwgJ1N0YWdlJywge1xuICAgICAgZW52OiB7IGFjY291bnQ6ICdhY2NvdW50JywgcmVnaW9uOiAncmVnaW9uJyB9LFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhzdGFnZSwgJ1N0YWNrMScsIHsgZW52OiB7IHJlZ2lvbjogJ2Vsc2V3aGVyZScgfSB9KTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soc3RhZ2UsICdTdGFjazInLCB7IGVudjogeyBhY2NvdW50OiAndG51b2NjYScgfSB9KTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChhY2N0UmVnaW9uKHN0YWNrMSksIFsnYWNjb3VudCcsICdlbHNld2hlcmUnXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYWNjdFJlZ2lvbihzdGFjazIpLCBbJ3RudW9jY2EnLCAncmVnaW9uJ10pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2VudnMgYXJlIGluaGVyaXRlZCBkZWVwbHknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBvdXRlciA9IG5ldyBTdGFnZShhcHAsICdTdGFnZScsIHtcbiAgICAgIGVudjogeyBhY2NvdW50OiAnYWNjb3VudCcsIHJlZ2lvbjogJ3JlZ2lvbicgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBpbm5lckFjY3QgPSBuZXcgU3RhZ2Uob3V0ZXIsICdBY2N0JywgeyBlbnY6IHsgYWNjb3VudDogJ3RudW9jY2EnIH0gfSk7XG4gICAgY29uc3QgaW5uZXJSZWdpb24gPSBuZXcgU3RhZ2Uob3V0ZXIsICdSZ24nLCB7IGVudjogeyByZWdpb246ICdlbHNld2hlcmUnIH0gfSk7XG4gICAgY29uc3QgaW5uZXJOZWl0aGVyID0gbmV3IFN0YWdlKG91dGVyLCAnTmVpdGhlcicpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKGFjY3RSZWdpb24obmV3IFN0YWNrKGlubmVyQWNjdCwgJ1N0YWNrJykpLCBbJ3RudW9jY2EnLCAncmVnaW9uJ10pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFjY3RSZWdpb24obmV3IFN0YWNrKGlubmVyUmVnaW9uLCAnU3RhY2snKSksIFsnYWNjb3VudCcsICdlbHNld2hlcmUnXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYWNjdFJlZ2lvbihuZXcgU3RhY2soaW5uZXJOZWl0aGVyLCAnU3RhY2snKSksIFsnYWNjb3VudCcsICdyZWdpb24nXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnVGhlIFN0YWdlIEFzc2VtYmx5IGlzIGluIHRoZSBhcHAgQXNzZW1ibHlcXCdzIG1hbmlmZXN0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFnZSA9IG5ldyBTdGFnZShhcHAsICdTdGFnZScpO1xuICAgIG5ldyBCb2d1c1N0YWNrKHN0YWdlLCAnU3RhY2syJyk7XG5cbiAgICAvLyBUSEVOIC0tIGFwcCBtYW5pZmVzdCBjb250YWlucyBhIG5lc3RlZCBjbG91ZCBhc3NlbWJseVxuICAgIGNvbnN0IGFwcEFzbSA9IGFwcC5zeW50aCgpO1xuXG4gICAgY29uc3QgYXJ0aWZhY3QgPSBhcHBBc20uYXJ0aWZhY3RzLmZpbmQoeCA9PiB4IGluc3RhbmNlb2YgY3hhcGkuTmVzdGVkQ2xvdWRBc3NlbWJseUFydGlmYWN0KTtcbiAgICB0ZXN0Lm9rKGFydGlmYWN0KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTdGFja3MgaW4gU3RhZ2UgYXJlIGluIGEgZGlmZmVyZW50IGN4YXNtIHRoYW4gU3RhY2tzIGluIEFwcCcodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IEJvZ3VzU3RhY2soYXBwLCAnU3RhY2sxJyk7XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2UoYXBwLCAnU3RhZ2UnKTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgQm9ndXNTdGFjayhzdGFnZSwgJ1N0YWNrMicpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHN0YWdlQXNtID0gc3RhZ2Uuc3ludGgoKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFnZUFzbS5zdGFja3MubWFwKHMgPT4gcy5zdGFja05hbWUpLCBbc3RhY2syLnN0YWNrTmFtZV0pO1xuXG4gICAgY29uc3QgYXBwQXNtID0gYXBwLnN5bnRoKCk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXBwQXNtLnN0YWNrcy5tYXAocyA9PiBzLnN0YWNrTmFtZSksIFtzdGFjazEuc3RhY2tOYW1lXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQ2FuIG5lc3QgU3RhZ2VzIGluc2lkZSBvdGhlciBTdGFnZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IG91dGVyID0gbmV3IFN0YWdlKGFwcCwgJ091dGVyJyk7XG4gICAgY29uc3QgaW5uZXIgPSBuZXcgU3RhZ2Uob3V0ZXIsICdJbm5lcicpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IEJvZ3VzU3RhY2soaW5uZXIsICdTdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFwcEFzbSA9IGFwcC5zeW50aCgpO1xuICAgIGNvbnN0IG91dGVyQXNtID0gYXBwQXNtLmdldE5lc3RlZEFzc2VtYmx5KG91dGVyLmFydGlmYWN0SWQpO1xuICAgIGNvbnN0IGlubmVyQXNtID0gb3V0ZXJBc20uZ2V0TmVzdGVkQXNzZW1ibHkoaW5uZXIuYXJ0aWZhY3RJZCk7XG5cbiAgICB0ZXN0Lm9rKGlubmVyQXNtLnRyeUdldEFydGlmYWN0KHN0YWNrLmFydGlmYWN0SWQpKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdEZWZhdWx0IHN0YWNrIG5hbWUgaW4gU3RhZ2Ugb2JqZWN0cyBpbmNvcnBvcmF0ZXMgdGhlIFN0YWdlIG5hbWUgYW5kIG5vIGhhc2gnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlKGFwcCwgJ015U3RhZ2UnKTtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBCb2d1c1N0YWNrKHN0YWdlLCAnTXlTdGFjaycpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZXF1YWwoc3RhZ2Uuc3RhZ2VOYW1lLCAnTXlTdGFnZScpO1xuICAgIHRlc3QuZXF1YWwoc3RhY2suc3RhY2tOYW1lLCAnTXlTdGFnZS1NeVN0YWNrJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQ2FuIG5vdCBoYXZlIGRlcGVuZGVuY2llcyB0byBzdGFja3Mgb3V0c2lkZSB0aGUgbmVzdGVkIGFzbScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBCb2d1c1N0YWNrKGFwcCwgJ1N0YWNrMScpO1xuICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlKGFwcCwgJ015U3RhZ2UnKTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgQm9ndXNTdGFjayhzdGFnZSwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIHN0YWNrMi5hZGREZXBlbmRlbmN5KHN0YWNrMSk7XG4gICAgfSwgL2RlcGVuZGVuY3kgY2Fubm90IGNyb3NzIHN0YWdlIGJvdW5kYXJpZXMvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdXaGVuIHdlIHN5bnRoKCkgYSBzdGFnZSwgcHJlcGFyZSBtdXN0IGJlIGNhbGxlZCBvbiBjb25zdHJ1Y3RzIGluIHRoZSBzdGFnZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGxldCBwcmVwYXJlZCA9IGZhbHNlO1xuICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlKGFwcCwgJ015U3RhZ2UnKTtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBCb2d1c1N0YWNrKHN0YWdlLCAnU3RhY2snKTtcbiAgICBjbGFzcyBIYXpQcmVwYXJlIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICAgIHByb3RlY3RlZCBwcmVwYXJlKCkge1xuICAgICAgICBwcmVwYXJlZCA9IHRydWU7XG4gICAgICB9XG4gICAgfVxuICAgIG5ldyBIYXpQcmVwYXJlKHN0YWNrLCAnUHJlcGFyYWJsZScpO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YWdlLnN5bnRoKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5lcXVhbHMocHJlcGFyZWQsIHRydWUpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1doZW4gd2Ugc3ludGgoKSBhIHN0YWdlLCBhc3BlY3RzIGluc2lkZSBpdCBtdXN0IGhhdmUgYmVlbiBhcHBsaWVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2UoYXBwLCAnTXlTdGFnZScpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IEJvZ3VzU3RhY2soc3RhZ2UsICdTdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFzcGVjdCA9IG5ldyBUb3VjaGluZ0FzcGVjdCgpO1xuICAgIEFzcGVjdHMub2Yoc3RhY2spLmFkZChhc3BlY3QpO1xuXG4gICAgLy8gVEhFTlxuICAgIGFwcC5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzcGVjdC52aXNpdHMubWFwKGMgPT4gYy5ub2RlLnBhdGgpLCBbXG4gICAgICAnTXlTdGFnZS9TdGFjaycsXG4gICAgICAnTXlTdGFnZS9TdGFjay9SZXNvdXJjZScsXG4gICAgXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQXNwZWN0cyBkbyBub3QgYXBwbHkgaW5zaWRlIGEgU3RhZ2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFnZSA9IG5ldyBTdGFnZShhcHAsICdNeVN0YWdlJyk7XG4gICAgbmV3IEJvZ3VzU3RhY2soc3RhZ2UsICdTdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFzcGVjdCA9IG5ldyBUb3VjaGluZ0FzcGVjdCgpO1xuICAgIEFzcGVjdHMub2YoYXBwKS5hZGQoYXNwZWN0KTtcblxuICAgIC8vIFRIRU5cbiAgICBhcHAuc3ludGgoKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3BlY3QudmlzaXRzLm1hcChjID0+IGMubm9kZS5wYXRoKSwgW1xuICAgICAgJycsXG4gICAgICAnVHJlZScsXG4gICAgXSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0F1dG9tYXRpYyBkZXBlbmRlbmNpZXMgaW5zaWRlIGEgc3RhZ2UgYXJlIGF2YWlsYWJsZSBpbW1lZGlhdGVseSBhZnRlciBzeW50aCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlKGFwcCwgJ015U3RhZ2UnKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soc3RhZ2UsICdTdGFjazEnKTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soc3RhZ2UsICdTdGFjazInKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCByZXNvdXJjZTEgPSBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2sxLCAnUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQ0RLOjpUZXN0OjpSZXNvdXJjZScsXG4gICAgfSk7XG4gICAgbmV3IENmblJlc291cmNlKHN0YWNrMiwgJ1Jlc291cmNlJywge1xuICAgICAgdHlwZTogJ0NESzo6VGVzdDo6UmVzb3VyY2UnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBPdGhlclRoaW5nOiByZXNvdXJjZTEucmVmLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGFzbSA9IHN0YWdlLnN5bnRoKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoXG4gICAgICBhc20uZ2V0U3RhY2tBcnRpZmFjdChzdGFjazIuYXJ0aWZhY3RJZCkuZGVwZW5kZW5jaWVzLm1hcChkID0+IGQuaWQpLFxuICAgICAgW3N0YWNrMS5hcnRpZmFjdElkXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQXNzZW1ibGllcyBjYW4gYmUgZGVlcGx5IG5lc3RlZCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCh7IHRyZWVNZXRhZGF0YTogZmFsc2UgfSk7XG5cbiAgICBjb25zdCBsZXZlbDEgPSBuZXcgU3RhZ2UoYXBwLCAnU3RhZ2VMZXZlbDEnKTtcbiAgICBjb25zdCBsZXZlbDIgPSBuZXcgU3RhZ2UobGV2ZWwxLCAnU3RhZ2VMZXZlbDInKTtcbiAgICBuZXcgU3RhZ2UobGV2ZWwyLCAnU3RhZ2VMZXZlbDMnKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCByb290QXNzZW1ibHkgPSBhcHAuc3ludGgoKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChyb290QXNzZW1ibHkubWFuaWZlc3QuYXJ0aWZhY3RzLCB7XG4gICAgICAnYXNzZW1ibHktU3RhZ2VMZXZlbDEnOiB7XG4gICAgICAgIHR5cGU6ICdjZGs6Y2xvdWQtYXNzZW1ibHknLFxuICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgZGlyZWN0b3J5TmFtZTogJ2Fzc2VtYmx5LVN0YWdlTGV2ZWwxJyxcbiAgICAgICAgICBkaXNwbGF5TmFtZTogJ1N0YWdlTGV2ZWwxJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBhc3NlbWJseUxldmVsMSA9IHJvb3RBc3NlbWJseS5nZXROZXN0ZWRBc3NlbWJseSgnYXNzZW1ibHktU3RhZ2VMZXZlbDEnKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseUxldmVsMS5tYW5pZmVzdC5hcnRpZmFjdHMsIHtcbiAgICAgICdhc3NlbWJseS1TdGFnZUxldmVsMS1TdGFnZUxldmVsMic6IHtcbiAgICAgICAgdHlwZTogJ2NkazpjbG91ZC1hc3NlbWJseScsXG4gICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICBkaXJlY3RvcnlOYW1lOiAnYXNzZW1ibHktU3RhZ2VMZXZlbDEtU3RhZ2VMZXZlbDInLFxuICAgICAgICAgIGRpc3BsYXlOYW1lOiAnU3RhZ2VMZXZlbDEvU3RhZ2VMZXZlbDInLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGFzc2VtYmx5TGV2ZWwyID0gYXNzZW1ibHlMZXZlbDEuZ2V0TmVzdGVkQXNzZW1ibHkoJ2Fzc2VtYmx5LVN0YWdlTGV2ZWwxLVN0YWdlTGV2ZWwyJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHlMZXZlbDIubWFuaWZlc3QuYXJ0aWZhY3RzLCB7XG4gICAgICAnYXNzZW1ibHktU3RhZ2VMZXZlbDEtU3RhZ2VMZXZlbDItU3RhZ2VMZXZlbDMnOiB7XG4gICAgICAgIHR5cGU6ICdjZGs6Y2xvdWQtYXNzZW1ibHknLFxuICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgZGlyZWN0b3J5TmFtZTogJ2Fzc2VtYmx5LVN0YWdlTGV2ZWwxLVN0YWdlTGV2ZWwyLVN0YWdlTGV2ZWwzJyxcbiAgICAgICAgICBkaXNwbGF5TmFtZTogJ1N0YWdlTGV2ZWwxL1N0YWdlTGV2ZWwyL1N0YWdlTGV2ZWwzJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhZ2UgbmFtZSB2YWxpZGF0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuXG4gICAgbmV3IFN0YWdlKGFwcCwgJ2FiY2QnKTtcbiAgICBuZXcgU3RhZ2UoYXBwLCAnYWJjZDEyMycpO1xuICAgIG5ldyBTdGFnZShhcHAsICdhYmNkMTIzLTU4OGRmamprJyk7XG4gICAgbmV3IFN0YWdlKGFwcCwgJ2FiY2QxMjMtNTg4ZGZqamsuc3NzJyk7XG4gICAgbmV3IFN0YWdlKGFwcCwgJ2FiY2QxMjMtNTg4ZGZqamsuc3NzX2Fqc2lkJyk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgU3RhZ2UoYXBwLCAnYWJjZDEyMy01ODhkZmpqay5zc3NfYWpzaWQgJyksIC9pbnZhbGlkIHN0YWdlIG5hbWUgXCJhYmNkMTIzLTU4OGRmamprLnNzc19hanNpZCBcIi8pO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBTdGFnZShhcHAsICdhYmNkMTIzLTU4OGRmamprLnNzc19hanNpZC9kZm8nKSwgL2ludmFsaWQgc3RhZ2UgbmFtZSBcImFiY2QxMjMtNTg4ZGZqamsuc3NzX2Fqc2lkXFwvZGZvXCIvKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgU3RhZ2UoYXBwLCAnJicpLCAvaW52YWxpZCBzdGFnZSBuYW1lIFwiJlwiLyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFN0YWdlKGFwcCwgJzQ1aGVsbG8nKSwgL2ludmFsaWQgc3RhZ2UgbmFtZSBcIjQ1aGVsbG9cIi8pO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBTdGFnZShhcHAsICdmJyksIC9pbnZhbGlkIHN0YWdlIG5hbWUgXCJmXCIvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdvdXRkaXIgY2Fubm90IGJlIHNwZWNpZmllZCBmb3IgbmVzdGVkIHN0YWdlcycodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFN0YWdlKGFwcCwgJ215c3RhZ2UnLCB7IG91dGRpcjogJy90bXAvZm9vL2JhcicgfSksIC9cIm91dGRpclwiIGNhbm5vdCBiZSBzcGVjaWZpZWQgZm9yIG5lc3RlZCBzdGFnZXMvKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG5cbmNsYXNzIFRvdWNoaW5nQXNwZWN0IGltcGxlbWVudHMgSUFzcGVjdCB7XG4gIHB1YmxpYyByZWFkb25seSB2aXNpdHMgPSBuZXcgQXJyYXk8SUNvbnN0cnVjdD4oKTtcbiAgcHVibGljIHZpc2l0KG5vZGU6IElDb25zdHJ1Y3QpOiB2b2lkIHtcbiAgICB0aGlzLnZpc2l0cy5wdXNoKG5vZGUpO1xuICB9XG59XG5cbmNsYXNzIEJvZ3VzU3RhY2sgZXh0ZW5kcyBTdGFjayB7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgbmV3IENmblJlc291cmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdDREs6OlRlc3Q6OlJlc291cmNlJyxcbiAgICB9KTtcbiAgfVxufVxuXG5mdW5jdGlvbiBhY2N0UmVnaW9uKHM6IFN0YWNrKSB7XG4gIHJldHVybiBbcy5hY2NvdW50LCBzLnJlZ2lvbl07XG59Il19