"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const synthesis_1 = require("../lib/private/synthesis");
const util_1 = require("./util");
class Counter extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, { type: 'My::Counter', properties: { Count: props.Count } });
        this.arn = this.getAtt('Arn').toString();
        this.url = this.getAtt('URL').toString();
        this.count = props.Count;
    }
    increment(by = 1) {
        this.count += by;
    }
    get cfnProperties() {
        return { Count: this.count };
    }
}
function withoutHash(logId) {
    return logId.substr(0, logId.length - 8);
}
class CustomizableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, { type: 'MyResourceType', properties: props });
        if (props !== undefined) {
            this.prop1 = props.prop1;
            this.prop2 = props.prop2;
            this.prop3 = props.prop3;
        }
    }
    renderProperties() {
        const props = this.updatedProperites;
        const render = {};
        for (const key of Object.keys(props)) {
            render[key.toUpperCase()] = props[key];
        }
        return render;
    }
    get updatedProperites() {
        const props = {
            prop1: this.prop1,
            prop2: this.prop2,
            prop3: this.prop3,
        };
        const cleanProps = {};
        for (const key of Object.keys(props)) {
            if (props[key] === undefined) {
                continue;
            }
            cleanProps[key] = props[key];
        }
        return cleanProps;
    }
}
module.exports = {
    'all resources derive from Resource, which derives from Entity'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnResource(stack, 'MyResource', {
            type: 'MyResourceType',
            properties: {
                Prop1: 'p1', Prop2: 123,
            },
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: {
                    Type: 'MyResourceType',
                    Properties: {
                        Prop1: 'p1',
                        Prop2: 123,
                    },
                },
            },
        });
        test.done();
    },
    'resources must reside within a Stack and fail upon creation if not'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.CfnResource(root, 'R1', { type: 'ResourceType' }));
        test.done();
    },
    'all entities have a logical ID calculated based on their full path in the tree'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack');
        const level1 = new lib_1.Construct(stack, 'level1');
        const level2 = new lib_1.Construct(level1, 'level2');
        const level3 = new lib_1.Construct(level2, 'level3');
        const res1 = new lib_1.CfnResource(level1, 'childoflevel1', { type: 'MyResourceType1' });
        const res2 = new lib_1.CfnResource(level3, 'childoflevel3', { type: 'MyResourceType2' });
        test.equal(withoutHash(stack.resolve(res1.logicalId)), 'level1childoflevel1');
        test.equal(withoutHash(stack.resolve(res2.logicalId)), 'level1level2level3childoflevel3');
        test.done();
    },
    'resource.props can only be accessed by derived classes'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 10 });
        res.increment();
        res.increment(2);
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 13 } },
            },
        });
        test.done();
    },
    'resource attributes can be retrieved using getAtt(s) or attribute properties'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 10 });
        new lib_1.CfnResource(stack, 'YourResource', {
            type: 'Type',
            properties: {
                CounterName: res.getAtt('Name'),
                CounterArn: res.arn,
                CounterURL: res.url,
            },
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 10 } },
                YourResource: {
                    Type: 'Type',
                    Properties: {
                        CounterName: { 'Fn::GetAtt': ['MyResource', 'Name'] },
                        CounterArn: { 'Fn::GetAtt': ['MyResource', 'Arn'] },
                        CounterURL: { 'Fn::GetAtt': ['MyResource', 'URL'] },
                    },
                },
            },
        });
        test.done();
    },
    'ARN-type resource attributes have some common functionality'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 1 });
        new lib_1.CfnResource(stack, 'MyResource2', {
            type: 'Type',
            properties: {
                Perm: res.arn,
            },
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 1 } },
                MyResource2: {
                    Type: 'Type',
                    Properties: {
                        Perm: {
                            'Fn::GetAtt': ['MyResource', 'Arn'],
                        },
                    },
                },
            },
        });
        test.done();
    },
    'resource.addDependency(e) can be used to add a DependsOn on another resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new Counter(stack, 'Counter1', { Count: 1 });
        const r2 = new Counter(stack, 'Counter2', { Count: 1 });
        const r3 = new lib_1.CfnResource(stack, 'Resource3', { type: 'MyResourceType' });
        r2.node.addDependency(r1);
        r2.node.addDependency(r3);
        synthesis_1.synthesize(stack);
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Counter1: {
                    Type: 'My::Counter',
                    Properties: { Count: 1 },
                },
                Counter2: {
                    Type: 'My::Counter',
                    Properties: { Count: 1 },
                    DependsOn: [
                        'Counter1',
                        'Resource3',
                    ],
                },
                Resource3: { Type: 'MyResourceType' },
            },
        });
        test.done();
    },
    'if addDependency is called multiple times with the same resource, it will only appear once'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const r1 = new Counter(stack, 'Counter1', { Count: 1 });
        const dependent = new lib_1.CfnResource(stack, 'Dependent', { type: 'R' });
        // WHEN
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Counter1: {
                    Type: 'My::Counter',
                    Properties: {
                        Count: 1,
                    },
                },
                Dependent: {
                    Type: 'R',
                    DependsOn: [
                        'Counter1',
                    ],
                },
            },
        });
        test.done();
    },
    'conditions can be attached to a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        const cond = new lib_1.CfnCondition(stack, 'MyCondition', { expression: lib_1.Fn.conditionNot(lib_1.Fn.conditionEquals('a', 'b')) });
        r1.cfnOptions.condition = cond;
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: { Resource: { Type: 'Type', Condition: 'MyCondition' } },
            Conditions: { MyCondition: { 'Fn::Not': [{ 'Fn::Equals': ['a', 'b'] }] } },
        });
        test.done();
    },
    'creation/update/updateReplace/deletion policies can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.creationPolicy = { autoScalingCreationPolicy: { minSuccessfulInstancesPercent: 10 } };
        // eslint-disable-next-line max-len
        r1.cfnOptions.updatePolicy = {
            autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: false },
            autoScalingReplacingUpdate: { willReplace: true },
            codeDeployLambdaAliasUpdate: {
                applicationName: 'CodeDeployApplication',
                deploymentGroupName: 'CodeDeployDeploymentGroup',
                beforeAllowTrafficHook: 'lambda1',
            },
        };
        r1.cfnOptions.deletionPolicy = lib_1.CfnDeletionPolicy.RETAIN;
        r1.cfnOptions.updateReplacePolicy = lib_1.CfnDeletionPolicy.SNAPSHOT;
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    CreationPolicy: { AutoScalingCreationPolicy: { MinSuccessfulInstancesPercent: 10 } },
                    UpdatePolicy: {
                        AutoScalingScheduledAction: { IgnoreUnmodifiedGroupSizeProperties: false },
                        AutoScalingReplacingUpdate: { WillReplace: true },
                        CodeDeployLambdaAliasUpdate: {
                            ApplicationName: 'CodeDeployApplication',
                            DeploymentGroupName: 'CodeDeployDeploymentGroup',
                            BeforeAllowTrafficHook: 'lambda1',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Snapshot',
                },
            },
        });
        test.done();
    },
    'update policies UseOnlineResharding flag'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.updatePolicy = { useOnlineResharding: true };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    UpdatePolicy: {
                        UseOnlineResharding: true,
                    },
                },
            },
        });
        test.done();
    },
    'metadata can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.metadata = {
            MyKey: 10,
            MyValue: 99,
        };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    Metadata: {
                        MyKey: 10,
                        MyValue: 99,
                    },
                },
            },
        });
        test.done();
    },
    'the "type" property is required when creating a resource'(test) {
        const stack = new lib_1.Stack();
        test.throws(() => new lib_1.CfnResource(stack, 'Resource', { notypehere: true }));
        test.done();
    },
    'removal policy is a high level abstraction of deletion policy used by l2'(test) {
        const stack = new lib_1.Stack();
        const retain = new lib_1.CfnResource(stack, 'Retain', { type: 'T1' });
        const destroy = new lib_1.CfnResource(stack, 'Destroy', { type: 'T3' });
        const def = new lib_1.CfnResource(stack, 'Default1', { type: 'T4' });
        const def2 = new lib_1.CfnResource(stack, 'Default2', { type: 'T4' });
        retain.applyRemovalPolicy(lib_1.RemovalPolicy.RETAIN);
        destroy.applyRemovalPolicy(lib_1.RemovalPolicy.DESTROY);
        def.applyRemovalPolicy(undefined, { default: lib_1.RemovalPolicy.DESTROY });
        def2.applyRemovalPolicy(undefined);
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Retain: { Type: 'T1', DeletionPolicy: 'Retain', UpdateReplacePolicy: 'Retain' },
                Destroy: { Type: 'T3', DeletionPolicy: 'Delete', UpdateReplacePolicy: 'Delete' },
                Default1: { Type: 'T4', DeletionPolicy: 'Delete', UpdateReplacePolicy: 'Delete' },
                Default2: { Type: 'T4', DeletionPolicy: 'Retain', UpdateReplacePolicy: 'Retain' },
            },
        });
        test.done();
    },
    'addDependency adds all dependencyElements of dependent constructs'(test) {
        class C1 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                this.r1 = new lib_1.CfnResource(this, 'R1', { type: 'T1' });
                this.r2 = new lib_1.CfnResource(this, 'R2', { type: 'T2' });
            }
        }
        class C2 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                this.r3 = new lib_1.CfnResource(this, 'R3', { type: 'T3' });
            }
        }
        // C3 returns [ c2 ] for it's dependency elements
        // this should result in 'flattening' the list of elements.
        class C3 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new C2(this, 'C2');
            }
        }
        const stack = new lib_1.Stack();
        const c1 = new C1(stack, 'MyC1');
        const c2 = new C2(stack, 'MyC2');
        const c3 = new C3(stack, 'MyC3');
        const dependingResource = new lib_1.CfnResource(stack, 'MyResource', { type: 'R' });
        dependingResource.node.addDependency(c1, c2);
        dependingResource.node.addDependency(c3);
        synthesis_1.synthesize(stack);
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyC1R1FB2A562F: { Type: 'T1' },
                MyC1R2AE2B5066: { Type: 'T2' },
                MyC2R3809EEAD6: { Type: 'T3' },
                MyC3C2R38CE6F9F7: { Type: 'T3' },
                MyResource: {
                    Type: 'R',
                    DependsOn: ['MyC1R1FB2A562F',
                        'MyC1R2AE2B5066',
                        'MyC2R3809EEAD6',
                        'MyC3C2R38CE6F9F7'],
                },
            },
        });
        test.done();
    },
    'resource.ref returns the {Ref} token'(test) {
        const stack = new lib_1.Stack();
        const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'R' });
        test.deepEqual(stack.resolve(r.ref), { Ref: 'MyResource' });
        test.done();
    },
    overrides: {
        'addOverride(p, v) allows assigning arbitrary values to synthesized resource definitions'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'AWS::Resource::Type' });
            // WHEN
            r.addOverride('Type', 'YouCanEvenOverrideTheType');
            r.addOverride('Metadata', { Key: 12 });
            r.addOverride('Use.Dot.Notation', 'To create subtrees');
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    MyResource: {
                        Type: 'YouCanEvenOverrideTheType',
                        Use: { Dot: { Notation: 'To create subtrees' } },
                        Metadata: { Key: 12 },
                    },
                },
            });
            test.done();
        },
        'addOverride(p, null) will assign an "null" value'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                        },
                    },
                },
            });
            // WHEN
            r.addOverride('Properties.Hello.World.Value2', null);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    MyResource: {
                        Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello', Value2: null } } },
                    },
                },
            });
            test.done();
        },
        'addOverride(p, undefined) can be used to delete a value'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                        },
                    },
                },
            });
            // WHEN
            r.addOverride('Properties.Hello.World.Value2', undefined);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    MyResource: {
                        Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello' } } },
                    },
                },
            });
            test.done();
        },
        'addOverride(p, undefined) will not create empty trees'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'AWS::Resource::Type' });
            // WHEN
            r.addPropertyOverride('Tree.Exists', 42);
            r.addPropertyOverride('Tree.Does.Not.Exist', undefined);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    MyResource: {
                        Type: 'AWS::Resource::Type',
                        Properties: { Tree: { Exists: 42 } },
                    },
                },
            });
            test.done();
        },
        'addDeletionOverride(p) and addPropertyDeletionOverride(pp) are sugar `undefined`'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                            Value3: ['foo', 'bar'],
                        },
                    },
                },
            });
            // WHEN
            r.addDeletionOverride('Properties.Hello.World.Value2');
            r.addPropertyDeletionOverride('Hello.World.Value3');
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    MyResource: {
                        Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello' } } },
                    },
                },
            });
            test.done();
        },
        'addOverride(p, v) will overwrite any non-objects along the path'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: 42,
                    },
                },
            });
            // WHEN
            r.addOverride('Properties.Override1', ['Hello', 123]);
            r.addOverride('Properties.Override1.Override2', { Heyy: [1] });
            r.addOverride('Properties.Hello.World.Foo.Bar', 42);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    MyResource: {
                        Type: 'AWS::Resource::Type',
                        Properties: {
                            Hello: { World: { Foo: { Bar: 42 } } },
                            Override1: {
                                Override2: { Heyy: [1] },
                            },
                        },
                    },
                },
            });
            test.done();
        },
        'addPropertyOverride(pp, v) is a sugar for overriding properties'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: { Hello: { World: 42 } },
            });
            // WHEN
            r.addPropertyOverride('Hello.World', { Hey: 'Jude' });
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    MyResource: {
                        Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Hey: 'Jude' } } },
                    },
                },
            });
            test.done();
        },
        'overrides are applied after render'(test) {
            // GIVEN
            class MyResource extends lib_1.CfnResource {
                renderProperties() {
                    return { Fixed: 123 };
                }
            }
            const stack = new lib_1.Stack();
            const cfn = new MyResource(stack, 'rr', { type: 'AWS::Resource::Type' });
            // WHEN
            cfn.addPropertyOverride('Boom', 'Hi');
            cfn.addOverride('Properties.Foo.Bar', 'Bar');
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    rr: {
                        Type: 'AWS::Resource::Type',
                        Properties: {
                            Fixed: 123,
                            Boom: 'Hi',
                            Foo: {
                                Bar: 'Bar',
                            },
                        },
                    },
                },
            });
            test.done();
        },
        'using mutable properties': {
            'can be used by derived classes to specify overrides before render()'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource', {
                    prop1: 'foo',
                });
                r.prop2 = 'bar';
                test.deepEqual(util_1.toCloudFormation(stack), {
                    Resources: {
                        MyResource: {
                            Type: 'MyResourceType',
                            Properties: { PROP1: 'foo', PROP2: 'bar' },
                        },
                    },
                });
                test.done();
            },
            '"properties" is undefined'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource');
                r.prop3 = 'zoo';
                test.deepEqual(util_1.toCloudFormation(stack), {
                    Resources: {
                        MyResource: {
                            Type: 'MyResourceType',
                            Properties: { PROP3: 'zoo' },
                        },
                    },
                });
                test.done();
            },
            '"properties" is empty'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource', {});
                r.prop3 = 'zoo';
                r.prop2 = 'hey';
                test.deepEqual(util_1.toCloudFormation(stack), {
                    Resources: {
                        MyResource: {
                            Type: 'MyResourceType',
                            Properties: { PROP2: 'hey', PROP3: 'zoo' },
                        },
                    },
                });
                test.done();
            },
        },
    },
    '"aws:cdk:path" metadata is added if "aws:cdk:path-metadata" context is set to true'(test) {
        const stack = new lib_1.Stack();
        stack.node.setContext(cxapi.PATH_METADATA_ENABLE_CONTEXT, true);
        const parent = new lib_1.Construct(stack, 'Parent');
        new lib_1.CfnResource(parent, 'MyResource', {
            type: 'MyResourceType',
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                ParentMyResource4B1FDBCC: {
                    Type: 'MyResourceType',
                    Metadata: { [cxapi.PATH_METADATA_KEY]: 'Default/Parent/MyResource' },
                },
            },
        });
        test.done();
    },
    'cross-stack construct dependencies are not rendered but turned into stack dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stackA = new lib_1.Stack(app, 'StackA');
        const resA = new lib_1.CfnResource(stackA, 'Resource', { type: 'R' });
        const stackB = new lib_1.Stack(app, 'StackB');
        const resB = new lib_1.CfnResource(stackB, 'Resource', { type: 'R' });
        // WHEN
        resB.node.addDependency(resA);
        // THEN
        const assembly = app.synth();
        const templateB = assembly.getStackByName(stackB.stackName).template;
        test.deepEqual(templateB, {
            Resources: {
                Resource: {
                    Type: 'R',
                },
            },
        });
        test.deepEqual(stackB.dependencies.map(s => s.node.id), ['StackA']);
        test.done();
    },
    'enableVersionUpgrade can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.updatePolicy = {
            enableVersionUpgrade: true,
        };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    UpdatePolicy: {
                        EnableVersionUpgrade: true,
                    },
                },
            },
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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