"use strict";
const fc = require("fast-check");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
function asyncTest(cb) {
    return async (test) => {
        let error;
        try {
            await cb(test);
        }
        catch (e) {
            error = e;
        }
        finally {
            test.doesNotThrow(() => {
                if (error) {
                    throw error;
                }
            });
            test.done();
        }
    };
}
const nonEmptyString = fc.string(1, 16);
const tokenish = fc.array(nonEmptyString, 2, 2).map(arr => ({ [arr[0]]: arr[1] }));
const anyValue = fc.oneof(nonEmptyString, tokenish);
function stringListToken(o) {
    return lib_1.Token.asList(new intrinsic_1.Intrinsic(o));
}
function stringToken(o) {
    return lib_1.Token.asString(new intrinsic_1.Intrinsic(o));
}
module.exports = nodeunit.testCase({
    'eager resolution for non-tokens': {
        'Fn.select'(test) {
            test.deepEqual(lib_1.Fn.select(2, ['hello', 'you', 'dude']), 'dude');
            test.done();
        },
        'Fn.split'(test) {
            test.deepEqual(lib_1.Fn.split(':', 'hello:world:yeah'), ['hello', 'world', 'yeah']);
            test.done();
        },
    },
    'FnParseDomainName': {
        'parse domain name from resolved url'(test) {
            test.deepEqual(lib_1.Fn.parseDomainName('https://test.com/'), 'test.com');
            test.done();
        },
        'parse domain name on token'(test) {
            const stack = new lib_1.Stack();
            const url = lib_1.Fn.join('//', [
                'https:',
                lib_1.Fn.join('/', [
                    'test.com',
                    'graphql',
                ]),
            ]);
            test.deepEqual(lib_1.Fn.parseDomainName(stack.resolve(url)), 'test.com');
            test.done();
        },
    },
    'FnJoin': {
        'rejects empty list of arguments to join'(test) {
            test.throws(() => lib_1.Fn.join('.', []));
            test.done();
        },
        'collapse nested FnJoins even if they contain tokens'(test) {
            const stack = new lib_1.Stack();
            const obj = lib_1.Fn.join('', [
                'a',
                lib_1.Fn.join('', [lib_1.Fn.getAtt('a', 'bc').toString(), 'c']),
                'd',
            ]);
            test.deepEqual(stack.resolve(obj), {
                'Fn::Join': ['',
                    [
                        'a',
                        { 'Fn::GetAtt': ['a', 'bc'] },
                        'cd',
                    ]],
            });
            test.done();
        },
        'resolves to the value if only one value is joined': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), anyValue, (delimiter, value) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [value])), value)), { verbose: true });
        }),
        'pre-concatenates string literals': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 15), (delimiter, values) => stack.resolve(lib_1.Fn.join(delimiter, values)) === values.join(delimiter)), { verbose: true });
        }),
        'pre-concatenates around tokens': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 3), tokenish, fc.array(nonEmptyString, 1, 3), (delimiter, prefix, obj, suffix) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, stringToken(obj), ...suffix])), { 'Fn::Join': [delimiter, [prefix.join(delimiter), obj, suffix.join(delimiter)]] })), { verbose: true, seed: 1539874645005, path: '0:0:0:0:0:0:0:0:0' });
        }),
        'flattens joins nested under joins with same delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.array(anyValue), fc.array(anyValue, 1, 3), fc.array(anyValue), (delimiter, prefix, nested, suffix) => 
            // Gonna test
            _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, lib_1.Fn.join(delimiter, nested), ...suffix])), stack.resolve(lib_1.Fn.join(delimiter, [...prefix, ...nested, ...suffix])))), { verbose: true });
        }),
        'does not flatten joins nested under joins with different delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.string(), fc.array(anyValue, 1, 3), fc.array(tokenish, 2, 3), fc.array(anyValue, 3), (delimiter1, delimiter2, prefix, nested, suffix) => {
                fc.pre(delimiter1 !== delimiter2);
                const join = lib_1.Fn.join(delimiter1, [...prefix, lib_1.Fn.join(delimiter2, stringListToken(nested)), ...suffix]);
                const resolved = stack.resolve(join);
                return resolved['Fn::Join'][1].find((e) => typeof e === 'object'
                    && ('Fn::Join' in e)
                    && e['Fn::Join'][0] === delimiter2) != null;
            }), { verbose: true });
        }),
        'Fn::EachMemberIn': asyncTest(async (test) => {
            const stack = new lib_1.Stack();
            const eachMemberIn = lib_1.Fn.conditionEachMemberIn(lib_1.Fn.valueOfAll('AWS::EC2::Subnet::Id', 'VpcId'), lib_1.Fn.refAll('AWS::EC2::VPC::Id'));
            test.deepEqual(stack.resolve(eachMemberIn), {
                'Fn::EachMemberIn': [
                    { 'Fn::ValueOfAll': ['AWS::EC2::Subnet::Id', 'VpcId'] },
                    { 'Fn::RefAll': 'AWS::EC2::VPC::Id' },
                ],
            });
        }),
        'cross-stack FnJoin elements are properly resolved': asyncTest(async (test) => {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1');
            const stack2 = new lib_1.Stack(app, 'Stack2');
            // WHEN
            new lib_1.CfnOutput(stack2, 'Stack1Id', {
                value: lib_1.Fn.join(' = ', ['Stack1Id', stack1.stackId]),
            });
            // THEN
            const template = app.synth().getStackByName('Stack2').template;
            test.deepEqual(template, {
                Outputs: {
                    Stack1Id: {
                        Value: {
                            'Fn::Join': [' = ', [
                                    'Stack1Id',
                                    { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSStackIdB2DD5BAA' },
                                ]],
                        },
                    },
                },
            });
        }),
    },
    'Ref': {
        'returns a reference given a logical name'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(lib_1.Fn.ref('hello')), {
                Ref: 'hello',
            });
            test.done();
        },
    },
    'nested Fn::Join with list token'(test) {
        const stack = new lib_1.Stack();
        const inner = lib_1.Fn.join(',', lib_1.Token.asList({ NotReallyList: true }));
        const outer = lib_1.Fn.join(',', [inner, 'Foo']);
        test.deepEqual(stack.resolve(outer), {
            'Fn::Join': [
                ',',
                [
                    { 'Fn::Join': [',', { NotReallyList: true }] },
                    'Foo',
                ],
            ],
        });
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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