"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
function floatEqual(test, actual, expected) {
    test.ok(
    // Floats are subject to rounding errors up to Number.ESPILON
    actual >= expected - Number.EPSILON && actual <= expected + Number.EPSILON, `${actual} == ${expected}`);
}
module.exports = nodeunit.testCase({
    'negative amount'(test) {
        test.throws(() => lib_1.Duration.seconds(-1), /negative/);
        test.done();
    },
    'unresolved amount'(test) {
        const stack = new lib_1.Stack();
        const lazyDuration = lib_1.Duration.seconds(lib_1.Token.asNumber({ resolve: () => 1337 }));
        test.equals(stack.resolve(lazyDuration.toSeconds()), 1337);
        test.throws(() => stack.resolve(lazyDuration.toMinutes()), /Unable to perform time unit conversion on un-resolved token/);
        test.done();
    },
    'Duration in seconds'(test) {
        const duration = lib_1.Duration.seconds(300);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'300 seconds' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.seconds(60 * 60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in minutes'(test) {
        const duration = lib_1.Duration.minutes(5);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'5 minutes' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.minutes(60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in hours'(test) {
        const duration = lib_1.Duration.hours(5);
        test.equal(duration.toSeconds(), 18000);
        test.equal(duration.toMinutes(), 300);
        test.throws(() => duration.toDays(), /'5 hours' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 5 / 24);
        test.equal(lib_1.Duration.hours(24).toDays(), 1);
        test.done();
    },
    'seconds to milliseconds'(test) {
        const duration = lib_1.Duration.seconds(5);
        test.equal(duration.toMilliseconds(), 5000);
        test.done();
    },
    'Duration in days'(test) {
        const duration = lib_1.Duration.days(1);
        test.equal(duration.toSeconds(), 86400);
        test.equal(duration.toMinutes(), 1440);
        test.equal(duration.toDays(), 1);
        test.done();
    },
    'toISOString'(test) {
        test.equal(lib_1.Duration.millis(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.millis(5).toISOString(), 'PT0.005S');
        test.equal(lib_1.Duration.seconds(5).toISOString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toISOString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toISOString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toISOString(), 'P5D');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toISOString(), 'P1DT1H1M1S');
        test.done();
    },
    'toIsoString'(test) {
        test.equal(lib_1.Duration.millis(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.millis(5).toIsoString(), 'PT0.005S');
        test.equal(lib_1.Duration.seconds(5).toIsoString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toIsoString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toIsoString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toIsoString(), 'P5D');
        test.equal(lib_1.Duration.seconds(65).toIsoString(), 'PT1M5S');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toIsoString(), 'P1DT1H1M1S');
        test.done();
    },
    'parse'(test) {
        test.equal(lib_1.Duration.parse('PT0S').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0M').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0H').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('P0D').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT5S').toSeconds(), 5);
        test.equal(lib_1.Duration.parse('PT5M').toSeconds(), 300);
        test.equal(lib_1.Duration.parse('PT5H').toSeconds(), 18000);
        test.equal(lib_1.Duration.parse('P5D').toSeconds(), 432000);
        test.equal(lib_1.Duration.parse('P1DT1H1M1S').toSeconds(), 1 + 60 * (1 + 60 * (1 + 24)));
        test.done();
    },
    'reject illegal parses'(test) {
        const err = 'Not a valid ISO duration';
        test.throws(() => {
            lib_1.Duration.parse('PT1D');
        }, err);
        test.throws(() => {
            lib_1.Duration.parse('P5S');
        }, err);
        test.done();
    },
    'to human string'(test) {
        test.equal(lib_1.Duration.minutes(0).toHumanString(), '0 minutes');
        test.equal(lib_1.Duration.minutes(lib_1.Lazy.numberValue({ produce: () => 5 })).toHumanString(), '<token> minutes');
        test.equal(lib_1.Duration.minutes(10).toHumanString(), '10 minutes');
        test.equal(lib_1.Duration.minutes(1).toHumanString(), '1 minute');
        test.equal(lib_1.Duration.minutes(62).toHumanString(), '1 hour 2 minutes');
        test.equal(lib_1.Duration.seconds(3666).toHumanString(), '1 hour 1 minute');
        test.equal(lib_1.Duration.millis(3000).toHumanString(), '3 seconds');
        test.equal(lib_1.Duration.millis(3666).toHumanString(), '3 seconds 666 millis');
        test.equal(lib_1.Duration.millis(3.6).toHumanString(), '3.6 millis');
        test.done();
    },
    'add two durations'(test) {
        test.equal(lib_1.Duration.minutes(1).plus(lib_1.Duration.seconds(30)).toSeconds(), lib_1.Duration.seconds(90).toSeconds());
        test.equal(lib_1.Duration.minutes(1).plus(lib_1.Duration.seconds(30)).toMinutes({ integral: false }), lib_1.Duration.seconds(90).toMinutes({ integral: false }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5kdXJhdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuZHVyYXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLHFDQUFxQztBQUNyQyxnQ0FBc0Q7QUEyS3RELFNBQVMsVUFBVSxDQUFDLElBQW1CLEVBQUUsTUFBYyxFQUFFLFFBQWdCO0lBQ3ZFLElBQUksQ0FBQyxFQUFFO0lBQ0wsNkRBQTZEO0lBQzdELE1BQU0sSUFBSSxRQUFRLEdBQUcsTUFBTSxDQUFDLE9BQU8sSUFBSSxNQUFNLElBQUksUUFBUSxHQUFHLE1BQU0sQ0FBQyxPQUFPLEVBQzFFLEdBQUcsTUFBTSxPQUFPLFFBQVEsRUFBRSxDQUMzQixDQUFDO0FBQ0osQ0FBQztBQS9LRCxpQkFBUyxRQUFRLENBQUMsUUFBUSxDQUFDO0lBQ3pCLGlCQUFpQixDQUFDLElBQW1CO1FBQ25DLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsY0FBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRXBELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQkFBbUIsQ0FBQyxJQUFtQjtRQUNyQyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sWUFBWSxHQUFHLGNBQVEsQ0FBQyxPQUFPLENBQUMsV0FBSyxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDL0UsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUUsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQzNELElBQUksQ0FBQyxNQUFNLENBQ1QsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsU0FBUyxFQUFFLENBQUMsRUFDN0MsNkRBQTZELENBQzlELENBQUM7UUFFRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscUJBQXFCLENBQUMsSUFBbUI7UUFDdkMsTUFBTSxRQUFRLEdBQUcsY0FBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUV2QyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsRUFBRSwrREFBK0QsQ0FBQyxDQUFDO1FBQ3RHLFVBQVUsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsQ0FBQyxFQUFFLEdBQUcsR0FBRyxLQUFNLENBQUMsQ0FBQztRQUVyRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUV2RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscUJBQXFCLENBQUMsSUFBbUI7UUFDdkMsTUFBTSxRQUFRLEdBQUcsY0FBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVyQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsRUFBRSw2REFBNkQsQ0FBQyxDQUFDO1FBQ3BHLFVBQVUsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsQ0FBQyxFQUFFLEdBQUcsR0FBRyxLQUFNLENBQUMsQ0FBQztRQUVyRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQkFBbUIsQ0FBQyxJQUFtQjtRQUNyQyxNQUFNLFFBQVEsR0FBRyxjQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRW5DLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxFQUFFLEtBQU0sQ0FBQyxDQUFDO1FBQ3pDLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxFQUFFLDJEQUEyRCxDQUFDLENBQUM7UUFDbEcsVUFBVSxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxDQUFDLEVBQUUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBRS9ELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUUzQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUJBQXlCLENBQUMsSUFBbUI7UUFDM0MsTUFBTSxRQUFRLEdBQUcsY0FBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVyQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxjQUFjLEVBQUUsRUFBRSxJQUFLLENBQUMsQ0FBQztRQUU3QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0JBQWtCLENBQUMsSUFBbUI7UUFDcEMsTUFBTSxRQUFRLEdBQUcsY0FBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVsQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsRUFBRSxLQUFNLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsRUFBRSxJQUFLLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUVqQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsYUFBYSxDQUFDLElBQW1CO1FBQy9CLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNyRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNwRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFbkQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUVsRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBRXZGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxhQUFhLENBQUMsSUFBbUI7UUFDL0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3JELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUVuRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDekQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN6RCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBRXZGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxPQUFPLENBQUMsSUFBbUI7UUFDekIsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2xELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDbEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRWpELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLEtBQU0sQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxNQUFPLENBQUMsQ0FBQztRQUV2RCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQUUsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRW5GLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1QkFBdUIsQ0FBQyxJQUFtQjtRQUN6QyxNQUFNLEdBQUcsR0FBRywwQkFBMEIsQ0FBQztRQUN2QyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLGNBQVEsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDekIsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRVIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixjQUFRLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3hCLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUVSLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxJQUFtQjtRQUNuQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxFQUFFLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDN0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsT0FBTyxDQUFDLFVBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGFBQWEsRUFBRSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFFeEcsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLGFBQWEsRUFBRSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQy9ELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxhQUFhLEVBQUUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUU1RCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsYUFBYSxFQUFFLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUVyRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsYUFBYSxFQUFFLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztRQUV0RSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsYUFBYSxFQUFFLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLGFBQWEsRUFBRSxFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFFMUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLGFBQWEsRUFBRSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBRS9ELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQkFBbUIsQ0FBQyxJQUFtQjtRQUNyQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGNBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxjQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDekcsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxDQUFDLEVBQUUsY0FBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsRUFBRSxRQUFRLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRS9JLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBub2RldW5pdCBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgTGF6eSwgU3RhY2ssIFRva2VuIH0gZnJvbSAnLi4vbGliJztcblxuZXhwb3J0ID0gbm9kZXVuaXQudGVzdENhc2Uoe1xuICAnbmVnYXRpdmUgYW1vdW50Jyh0ZXN0OiBub2RldW5pdC5UZXN0KSB7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gRHVyYXRpb24uc2Vjb25kcygtMSksIC9uZWdhdGl2ZS8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3VucmVzb2x2ZWQgYW1vdW50Jyh0ZXN0OiBub2RldW5pdC5UZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBsYXp5RHVyYXRpb24gPSBEdXJhdGlvbi5zZWNvbmRzKFRva2VuLmFzTnVtYmVyKHsgcmVzb2x2ZTogKCkgPT4gMTMzNyB9KSk7XG4gICAgdGVzdC5lcXVhbHMoc3RhY2sucmVzb2x2ZShsYXp5RHVyYXRpb24udG9TZWNvbmRzKCkpLCAxMzM3KTtcbiAgICB0ZXN0LnRocm93cyhcbiAgICAgICgpID0+IHN0YWNrLnJlc29sdmUobGF6eUR1cmF0aW9uLnRvTWludXRlcygpKSxcbiAgICAgIC9VbmFibGUgdG8gcGVyZm9ybSB0aW1lIHVuaXQgY29udmVyc2lvbiBvbiB1bi1yZXNvbHZlZCB0b2tlbi8sXG4gICAgKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdEdXJhdGlvbiBpbiBzZWNvbmRzJyh0ZXN0OiBub2RldW5pdC5UZXN0KSB7XG4gICAgY29uc3QgZHVyYXRpb24gPSBEdXJhdGlvbi5zZWNvbmRzKDMwMCk7XG5cbiAgICB0ZXN0LmVxdWFsKGR1cmF0aW9uLnRvU2Vjb25kcygpLCAzMDApO1xuICAgIHRlc3QuZXF1YWwoZHVyYXRpb24udG9NaW51dGVzKCksIDUpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IGR1cmF0aW9uLnRvRGF5cygpLCAvJzMwMCBzZWNvbmRzJyBjYW5ub3QgYmUgY29udmVydGVkIGludG8gYSB3aG9sZSBudW1iZXIgb2YgZGF5cy8pO1xuICAgIGZsb2F0RXF1YWwodGVzdCwgZHVyYXRpb24udG9EYXlzKHsgaW50ZWdyYWw6IGZhbHNlIH0pLCAzMDAgLyA4Nl80MDApO1xuXG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5zZWNvbmRzKDYwICogNjAgKiAyNCkudG9EYXlzKCksIDEpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0R1cmF0aW9uIGluIG1pbnV0ZXMnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICBjb25zdCBkdXJhdGlvbiA9IER1cmF0aW9uLm1pbnV0ZXMoNSk7XG5cbiAgICB0ZXN0LmVxdWFsKGR1cmF0aW9uLnRvU2Vjb25kcygpLCAzMDApO1xuICAgIHRlc3QuZXF1YWwoZHVyYXRpb24udG9NaW51dGVzKCksIDUpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IGR1cmF0aW9uLnRvRGF5cygpLCAvJzUgbWludXRlcycgY2Fubm90IGJlIGNvbnZlcnRlZCBpbnRvIGEgd2hvbGUgbnVtYmVyIG9mIGRheXMvKTtcbiAgICBmbG9hdEVxdWFsKHRlc3QsIGR1cmF0aW9uLnRvRGF5cyh7IGludGVncmFsOiBmYWxzZSB9KSwgMzAwIC8gODZfNDAwKTtcblxuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWludXRlcyg2MCAqIDI0KS50b0RheXMoKSwgMSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnRHVyYXRpb24gaW4gaG91cnMnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICBjb25zdCBkdXJhdGlvbiA9IER1cmF0aW9uLmhvdXJzKDUpO1xuXG4gICAgdGVzdC5lcXVhbChkdXJhdGlvbi50b1NlY29uZHMoKSwgMThfMDAwKTtcbiAgICB0ZXN0LmVxdWFsKGR1cmF0aW9uLnRvTWludXRlcygpLCAzMDApO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IGR1cmF0aW9uLnRvRGF5cygpLCAvJzUgaG91cnMnIGNhbm5vdCBiZSBjb252ZXJ0ZWQgaW50byBhIHdob2xlIG51bWJlciBvZiBkYXlzLyk7XG4gICAgZmxvYXRFcXVhbCh0ZXN0LCBkdXJhdGlvbi50b0RheXMoeyBpbnRlZ3JhbDogZmFsc2UgfSksIDUgLyAyNCk7XG5cbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLmhvdXJzKDI0KS50b0RheXMoKSwgMSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc2Vjb25kcyB0byBtaWxsaXNlY29uZHMnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICBjb25zdCBkdXJhdGlvbiA9IER1cmF0aW9uLnNlY29uZHMoNSk7XG5cbiAgICB0ZXN0LmVxdWFsKGR1cmF0aW9uLnRvTWlsbGlzZWNvbmRzKCksIDVfMDAwKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdEdXJhdGlvbiBpbiBkYXlzJyh0ZXN0OiBub2RldW5pdC5UZXN0KSB7XG4gICAgY29uc3QgZHVyYXRpb24gPSBEdXJhdGlvbi5kYXlzKDEpO1xuXG4gICAgdGVzdC5lcXVhbChkdXJhdGlvbi50b1NlY29uZHMoKSwgODZfNDAwKTtcbiAgICB0ZXN0LmVxdWFsKGR1cmF0aW9uLnRvTWludXRlcygpLCAxXzQ0MCk7XG4gICAgdGVzdC5lcXVhbChkdXJhdGlvbi50b0RheXMoKSwgMSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndG9JU09TdHJpbmcnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLm1pbGxpcygwKS50b0lTT1N0cmluZygpLCAnUFQwUycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uc2Vjb25kcygwKS50b0lTT1N0cmluZygpLCAnUFQwUycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWludXRlcygwKS50b0lTT1N0cmluZygpLCAnUFQwUycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uaG91cnMoMCkudG9JU09TdHJpbmcoKSwgJ1BUMFMnKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLmRheXMoMCkudG9JU09TdHJpbmcoKSwgJ1BUMFMnKTtcblxuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWlsbGlzKDUpLnRvSVNPU3RyaW5nKCksICdQVDAuMDA1UycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uc2Vjb25kcyg1KS50b0lTT1N0cmluZygpLCAnUFQ1UycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWludXRlcyg1KS50b0lTT1N0cmluZygpLCAnUFQ1TScpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uaG91cnMoNSkudG9JU09TdHJpbmcoKSwgJ1BUNUgnKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLmRheXMoNSkudG9JU09TdHJpbmcoKSwgJ1A1RCcpO1xuXG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5zZWNvbmRzKDEgKyA2MCAqICgxICsgNjAgKiAoMSArIDI0KSkpLnRvSVNPU3RyaW5nKCksICdQMURUMUgxTTFTJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndG9Jc29TdHJpbmcnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLm1pbGxpcygwKS50b0lzb1N0cmluZygpLCAnUFQwUycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uc2Vjb25kcygwKS50b0lzb1N0cmluZygpLCAnUFQwUycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWludXRlcygwKS50b0lzb1N0cmluZygpLCAnUFQwUycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uaG91cnMoMCkudG9Jc29TdHJpbmcoKSwgJ1BUMFMnKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLmRheXMoMCkudG9Jc29TdHJpbmcoKSwgJ1BUMFMnKTtcblxuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWlsbGlzKDUpLnRvSXNvU3RyaW5nKCksICdQVDAuMDA1UycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uc2Vjb25kcyg1KS50b0lzb1N0cmluZygpLCAnUFQ1UycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWludXRlcyg1KS50b0lzb1N0cmluZygpLCAnUFQ1TScpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24uaG91cnMoNSkudG9Jc29TdHJpbmcoKSwgJ1BUNUgnKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLmRheXMoNSkudG9Jc29TdHJpbmcoKSwgJ1A1RCcpO1xuXG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5zZWNvbmRzKDY1KS50b0lzb1N0cmluZygpLCAnUFQxTTVTJyk7XG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5zZWNvbmRzKDEgKyA2MCAqICgxICsgNjAgKiAoMSArIDI0KSkpLnRvSXNvU3RyaW5nKCksICdQMURUMUgxTTFTJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAncGFyc2UnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLnBhcnNlKCdQVDBTJykudG9TZWNvbmRzKCksIDApO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ucGFyc2UoJ1BUME0nKS50b1NlY29uZHMoKSwgMCk7XG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5wYXJzZSgnUFQwSCcpLnRvU2Vjb25kcygpLCAwKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLnBhcnNlKCdQMEQnKS50b1NlY29uZHMoKSwgMCk7XG5cbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLnBhcnNlKCdQVDVTJykudG9TZWNvbmRzKCksIDUpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ucGFyc2UoJ1BUNU0nKS50b1NlY29uZHMoKSwgMzAwKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLnBhcnNlKCdQVDVIJykudG9TZWNvbmRzKCksIDE4XzAwMCk7XG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5wYXJzZSgnUDVEJykudG9TZWNvbmRzKCksIDQzMl8wMDApO1xuXG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5wYXJzZSgnUDFEVDFIMU0xUycpLnRvU2Vjb25kcygpLCAxICsgNjAgKiAoMSArIDYwICogKDEgKyAyNCkpKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdyZWplY3QgaWxsZWdhbCBwYXJzZXMnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICBjb25zdCBlcnIgPSAnTm90IGEgdmFsaWQgSVNPIGR1cmF0aW9uJztcbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICBEdXJhdGlvbi5wYXJzZSgnUFQxRCcpO1xuICAgIH0sIGVycik7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICBEdXJhdGlvbi5wYXJzZSgnUDVTJyk7XG4gICAgfSwgZXJyKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0byBodW1hbiBzdHJpbmcnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLm1pbnV0ZXMoMCkudG9IdW1hblN0cmluZygpLCAnMCBtaW51dGVzJyk7XG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5taW51dGVzKExhenkubnVtYmVyVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiA1IH0pKS50b0h1bWFuU3RyaW5nKCksICc8dG9rZW4+IG1pbnV0ZXMnKTtcblxuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWludXRlcygxMCkudG9IdW1hblN0cmluZygpLCAnMTAgbWludXRlcycpO1xuICAgIHRlc3QuZXF1YWwoRHVyYXRpb24ubWludXRlcygxKS50b0h1bWFuU3RyaW5nKCksICcxIG1pbnV0ZScpO1xuXG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5taW51dGVzKDYyKS50b0h1bWFuU3RyaW5nKCksICcxIGhvdXIgMiBtaW51dGVzJyk7XG5cbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLnNlY29uZHMoMzY2NikudG9IdW1hblN0cmluZygpLCAnMSBob3VyIDEgbWludXRlJyk7XG5cbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLm1pbGxpcygzMDAwKS50b0h1bWFuU3RyaW5nKCksICczIHNlY29uZHMnKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLm1pbGxpcygzNjY2KS50b0h1bWFuU3RyaW5nKCksICczIHNlY29uZHMgNjY2IG1pbGxpcycpO1xuXG4gICAgdGVzdC5lcXVhbChEdXJhdGlvbi5taWxsaXMoMy42KS50b0h1bWFuU3RyaW5nKCksICczLjYgbWlsbGlzJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkIHR3byBkdXJhdGlvbnMnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLm1pbnV0ZXMoMSkucGx1cyhEdXJhdGlvbi5zZWNvbmRzKDMwKSkudG9TZWNvbmRzKCksIER1cmF0aW9uLnNlY29uZHMoOTApLnRvU2Vjb25kcygpKTtcbiAgICB0ZXN0LmVxdWFsKER1cmF0aW9uLm1pbnV0ZXMoMSkucGx1cyhEdXJhdGlvbi5zZWNvbmRzKDMwKSkudG9NaW51dGVzKHsgaW50ZWdyYWw6IGZhbHNlIH0pLCBEdXJhdGlvbi5zZWNvbmRzKDkwKS50b01pbnV0ZXMoeyBpbnRlZ3JhbDogZmFsc2UgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59KTtcblxuZnVuY3Rpb24gZmxvYXRFcXVhbCh0ZXN0OiBub2RldW5pdC5UZXN0LCBhY3R1YWw6IG51bWJlciwgZXhwZWN0ZWQ6IG51bWJlcikge1xuICB0ZXN0Lm9rKFxuICAgIC8vIEZsb2F0cyBhcmUgc3ViamVjdCB0byByb3VuZGluZyBlcnJvcnMgdXAgdG8gTnVtYmVyLkVTUElMT05cbiAgICBhY3R1YWwgPj0gZXhwZWN0ZWQgLSBOdW1iZXIuRVBTSUxPTiAmJiBhY3R1YWwgPD0gZXhwZWN0ZWQgKyBOdW1iZXIuRVBTSUxPTixcbiAgICBgJHthY3R1YWx9ID09ICR7ZXhwZWN0ZWR9YCxcbiAgKTtcbn1cbiJdfQ==