"use strict";
const lib_1 = require("../lib");
const cfn_json_1 = require("../lib/cfn-json");
const index_1 = require("../lib/private/cfn-utils-provider/index");
module.exports = {
    'resolves to a fn::getatt'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'test');
        // WHEN
        const json = new cfn_json_1.CfnJson(stack, 'MyCfnJson', {
            value: {
                hello: 1234,
                world: { bar: 1234 },
            },
        });
        // THEN
        const template = app.synth().getStackArtifact(stack.artifactId).template;
        // input is stringified
        test.deepEqual(template.Resources.MyCfnJson248769BB.Properties.Value, '{"hello":1234,"world":{"bar":1234}}');
        // output is basically an Fn::GetAtt
        test.deepEqual(stack.resolve(json), { 'Fn::GetAtt': ['MyCfnJson248769BB', 'Value'] });
        test.done();
    },
    'tokens and intrinsics can be used freely in keys or values'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'test');
        const other = new lib_1.CfnResource(stack, 'Other', { type: 'MyResource' });
        // WHEN
        new cfn_json_1.CfnJson(stack, 'MyCfnJson', {
            value: {
                [other.ref]: 1234,
                world: {
                    bar: `this is a ${lib_1.Lazy.stringValue({ produce: () => 'I am lazy' })}`,
                },
            },
        });
        // THEN
        const template = app.synth().getStackArtifact(stack.artifactId).template;
        test.deepEqual(template.Resources.MyCfnJson248769BB.Properties.Value, {
            'Fn::Join': ['', ['{"', { Ref: 'Other' }, '":1234,"world":{"bar":"this is a I am lazy"}}']],
        });
        test.done();
    },
    'JSON.stringify() will return the CFN-stringified value to avoid circular references'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const res = new lib_1.CfnResource(stack, 'MyResource', { type: 'Foo' });
        const cfnjson = new cfn_json_1.CfnJson(stack, 'MyCfnJson', {
            value: {
                [`ref=${res.ref}`]: `this is a ${lib_1.Lazy.stringValue({ produce: () => 'I am lazy' })}`,
            },
        });
        // WHEN
        const str = JSON.stringify(cfnjson);
        // THEN
        test.ok(typeof (str) === 'string');
        test.deepEqual(stack.resolve(str), {
            'Fn::Join': ['', ['"{"ref=', { Ref: 'MyResource' }, '":"this is a I am lazy"}"']],
        });
        test.done();
    },
    async 'resource provider simply parses json and reflects back as an attribute'(test) {
        const input = { foo: 1234 };
        const response = await index_1.handler({
            ResourceType: "Custom::AWSCDKCfnJson" /* CFN_JSON */,
            ResourceProperties: {
                Value: JSON.stringify(input),
            },
        });
        test.deepEqual(input, response.Data.Value);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jZm4tanNvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuY2ZuLWpzb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUNBLGdDQUF1RDtBQUN2RCw4Q0FBMEM7QUFFMUMsbUVBQWtFO0FBRWxFLGlCQUFTO0lBRVAsMEJBQTBCLENBQUMsSUFBVTtRQUNuQyxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFckMsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksa0JBQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxLQUFLLEVBQUUsSUFBSTtnQkFDWCxLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsSUFBSSxFQUFFO2FBQ3JCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsUUFBUSxDQUFDO1FBRXpFLHVCQUF1QjtRQUN2QixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsaUJBQWlCLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxxQ0FBcUMsQ0FBQyxDQUFDO1FBRTdHLG9DQUFvQztRQUNwQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFdEYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDREQUE0RCxDQUFDLElBQVU7UUFDckUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3JDLE1BQU0sS0FBSyxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxDQUFDLENBQUM7UUFFdEUsT0FBTztRQUNQLElBQUksa0JBQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQzlCLEtBQUssRUFBRTtnQkFDTCxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsRUFBRSxJQUFJO2dCQUNqQixLQUFLLEVBQUU7b0JBQ0wsR0FBRyxFQUFFLGFBQWEsVUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxFQUFFO2lCQUNyRTthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsUUFBUSxDQUFDO1FBRXpFLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFO1lBQ3BFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLElBQUksRUFBRSxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsRUFBRSwrQ0FBK0MsQ0FBQyxDQUFDO1NBQzVGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxRkFBcUYsQ0FBQyxJQUFVO1FBQzlGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sR0FBRyxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDbEUsTUFBTSxPQUFPLEdBQUcsSUFBSSxrQkFBTyxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDOUMsS0FBSyxFQUFFO2dCQUNMLENBQUMsT0FBTyxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxhQUFhLFVBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsV0FBVyxFQUFFLENBQUMsRUFBRTthQUNwRjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRXBDLE9BQU87UUFDUCxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQU0sQ0FBQyxHQUFHLENBQUMsS0FBSyxRQUFRLENBQUMsQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDakMsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLFlBQVksRUFBRSxFQUFFLDJCQUEyQixDQUFDLENBQUM7U0FDbEYsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELEtBQUssQ0FBQyx3RUFBd0UsQ0FBQyxJQUFVO1FBQ3ZGLE1BQU0sS0FBSyxHQUFHLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxDQUFDO1FBQzVCLE1BQU0sUUFBUSxHQUFHLE1BQU0sZUFBTyxDQUFDO1lBQzdCLFlBQVksd0NBQStCO1lBQzNDLGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUM7YUFDN0I7U0FDSyxDQUFDLENBQUM7UUFDVixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCB7IEFwcCwgQ2ZuUmVzb3VyY2UsIExhenksIFN0YWNrIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IENmbkpzb24gfSBmcm9tICcuLi9saWIvY2ZuLWpzb24nO1xuaW1wb3J0IHsgQ2ZuVXRpbHNSZXNvdXJjZVR5cGUgfSBmcm9tICcuLi9saWIvcHJpdmF0ZS9jZm4tdXRpbHMtcHJvdmlkZXIvY29uc3RzJztcbmltcG9ydCB7IGhhbmRsZXIgfSBmcm9tICcuLi9saWIvcHJpdmF0ZS9jZm4tdXRpbHMtcHJvdmlkZXIvaW5kZXgnO1xuXG5leHBvcnQgPSB7XG5cbiAgJ3Jlc29sdmVzIHRvIGEgZm46OmdldGF0dCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ3Rlc3QnKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBqc29uID0gbmV3IENmbkpzb24oc3RhY2ssICdNeUNmbkpzb24nLCB7XG4gICAgICB2YWx1ZToge1xuICAgICAgICBoZWxsbzogMTIzNCxcbiAgICAgICAgd29ybGQ6IHsgYmFyOiAxMjM0IH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHRlbXBsYXRlID0gYXBwLnN5bnRoKCkuZ2V0U3RhY2tBcnRpZmFjdChzdGFjay5hcnRpZmFjdElkKS50ZW1wbGF0ZTtcblxuICAgIC8vIGlucHV0IGlzIHN0cmluZ2lmaWVkXG4gICAgdGVzdC5kZWVwRXF1YWwodGVtcGxhdGUuUmVzb3VyY2VzLk15Q2ZuSnNvbjI0ODc2OUJCLlByb3BlcnRpZXMuVmFsdWUsICd7XCJoZWxsb1wiOjEyMzQsXCJ3b3JsZFwiOntcImJhclwiOjEyMzR9fScpO1xuXG4gICAgLy8gb3V0cHV0IGlzIGJhc2ljYWxseSBhbiBGbjo6R2V0QXR0XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShqc29uKSwgeyAnRm46OkdldEF0dCc6IFsnTXlDZm5Kc29uMjQ4NzY5QkInLCAnVmFsdWUnXSB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0b2tlbnMgYW5kIGludHJpbnNpY3MgY2FuIGJlIHVzZWQgZnJlZWx5IGluIGtleXMgb3IgdmFsdWVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAndGVzdCcpO1xuICAgIGNvbnN0IG90aGVyID0gbmV3IENmblJlc291cmNlKHN0YWNrLCAnT3RoZXInLCB7IHR5cGU6ICdNeVJlc291cmNlJyB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2ZuSnNvbihzdGFjaywgJ015Q2ZuSnNvbicsIHtcbiAgICAgIHZhbHVlOiB7XG4gICAgICAgIFtvdGhlci5yZWZdOiAxMjM0LFxuICAgICAgICB3b3JsZDoge1xuICAgICAgICAgIGJhcjogYHRoaXMgaXMgYSAke0xhenkuc3RyaW5nVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiAnSSBhbSBsYXp5JyB9KX1gLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCB0ZW1wbGF0ZSA9IGFwcC5zeW50aCgpLmdldFN0YWNrQXJ0aWZhY3Qoc3RhY2suYXJ0aWZhY3RJZCkudGVtcGxhdGU7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbCh0ZW1wbGF0ZS5SZXNvdXJjZXMuTXlDZm5Kc29uMjQ4NzY5QkIuUHJvcGVydGllcy5WYWx1ZSwge1xuICAgICAgJ0ZuOjpKb2luJzogWycnLCBbJ3tcIicsIHsgUmVmOiAnT3RoZXInIH0sICdcIjoxMjM0LFwid29ybGRcIjp7XCJiYXJcIjpcInRoaXMgaXMgYSBJIGFtIGxhenlcIn19J11dLFxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdKU09OLnN0cmluZ2lmeSgpIHdpbGwgcmV0dXJuIHRoZSBDRk4tc3RyaW5naWZpZWQgdmFsdWUgdG8gYXZvaWQgY2lyY3VsYXIgcmVmZXJlbmNlcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdNeVJlc291cmNlJywgeyB0eXBlOiAnRm9vJyB9KTtcbiAgICBjb25zdCBjZm5qc29uID0gbmV3IENmbkpzb24oc3RhY2ssICdNeUNmbkpzb24nLCB7XG4gICAgICB2YWx1ZToge1xuICAgICAgICBbYHJlZj0ke3Jlcy5yZWZ9YF06IGB0aGlzIGlzIGEgJHtMYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT4gJ0kgYW0gbGF6eScgfSl9YCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RyID0gSlNPTi5zdHJpbmdpZnkoY2ZuanNvbik7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5vayh0eXBlb2Yoc3RyKSA9PT0gJ3N0cmluZycpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUoc3RyKSwge1xuICAgICAgJ0ZuOjpKb2luJzogWycnLCBbJ1wie1wicmVmPScsIHsgUmVmOiAnTXlSZXNvdXJjZScgfSwgJ1wiOlwidGhpcyBpcyBhIEkgYW0gbGF6eVwifVwiJ11dLFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgYXN5bmMgJ3Jlc291cmNlIHByb3ZpZGVyIHNpbXBseSBwYXJzZXMganNvbiBhbmQgcmVmbGVjdHMgYmFjayBhcyBhbiBhdHRyaWJ1dGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBpbnB1dCA9IHsgZm9vOiAxMjM0IH07XG4gICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBoYW5kbGVyKHtcbiAgICAgIFJlc291cmNlVHlwZTogQ2ZuVXRpbHNSZXNvdXJjZVR5cGUuQ0ZOX0pTT04sXG4gICAgICBSZXNvdXJjZVByb3BlcnRpZXM6IHtcbiAgICAgICAgVmFsdWU6IEpTT04uc3RyaW5naWZ5KGlucHV0KSxcbiAgICAgIH0sXG4gICAgfSBhcyBhbnkpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGlucHV0LCByZXNwb25zZS5EYXRhLlZhbHVlKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07Il19