"use strict";
const child_process = require("child_process");
const sinon = require("sinon");
const lib_1 = require("../lib");
module.exports = {
    'tearDown'(callback) {
        sinon.restore();
        callback();
    },
    'bundling with image from registry'(test) {
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        image._run({
            command: ['cool', 'command'],
            environment: {
                VAR1: 'value1',
                VAR2: 'value2',
            },
            volumes: [{ hostPath: '/host-path', containerPath: '/container-path' }],
            workingDirectory: '/working-directory',
            user: 'user:group',
        });
        test.ok(spawnSyncStub.calledWith('docker', [
            'run', '--rm',
            '-u', 'user:group',
            '-v', '/host-path:/container-path:delegated',
            '--env', 'VAR1=value1',
            '--env', 'VAR2=value2',
            '-w', '/working-directory',
            'alpine',
            'cool', 'command',
        ], { stdio: ['ignore', process.stderr, 'inherit'] }));
        test.done();
    },
    'bundling with image from asset'(test) {
        const imageId = 'abcdef123456';
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from(`sha256:${imageId}`),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const imageHash = '123456abcdef';
        const fingerprintStub = sinon.stub(lib_1.FileSystem, 'fingerprint');
        fingerprintStub.callsFake(() => imageHash);
        const image = lib_1.BundlingDockerImage.fromAsset('docker-path', {
            buildArgs: {
                TEST_ARG: 'cdk-test',
            },
        });
        image._run();
        test.ok(spawnSyncStub.firstCall.calledWith('docker', [
            'build', '-q',
            '--build-arg', 'TEST_ARG=cdk-test',
            'docker-path',
        ]));
        test.ok(spawnSyncStub.secondCall.calledWith('docker', [
            'run', '--rm',
            imageId,
        ]));
        test.done();
    },
    'throws if image id cannot be extracted from build output'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        test.throws(() => lib_1.BundlingDockerImage.fromAsset('docker-path'), /Failed to extract image ID from Docker build output/);
        test.done();
    },
    'throws in case of spawnSync error'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
            error: new Error('UnknownError'),
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /UnknownError/);
        test.done();
    },
    'throws if status is not 0'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: -1,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /\[Status -1\]/);
        test.done();
    },
    'BundlerDockerImage json is the bundler image name by default'(test) {
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.equals(image.toJSON(), 'alpine');
        test.done();
    },
    'BundlerDockerImage json is the bundler image if building an image'(test) {
        const imageId = 'abcdef123456';
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from(`sha256:${imageId}`),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const imageHash = '123456abcdef';
        const fingerprintStub = sinon.stub(lib_1.FileSystem, 'fingerprint');
        fingerprintStub.callsFake(() => imageHash);
        const image = lib_1.BundlingDockerImage.fromAsset('docker-path');
        test.equals(image.image, imageId);
        test.equals(image.toJSON(), imageHash);
        test.ok(fingerprintStub.calledWith('docker-path', sinon.match({ extraHash: JSON.stringify({}) })));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5idW5kbGluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuYnVuZGxpbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLCtDQUErQztBQUUvQywrQkFBK0I7QUFDL0IsZ0NBQXlEO0FBRXpELGlCQUFTO0lBQ1AsVUFBVSxDQUFDLFFBQWE7UUFDdEIsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ2hCLFFBQVEsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVELG1DQUFtQyxDQUFDLElBQVU7UUFDNUMsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ25FLE1BQU0sRUFBRSxDQUFDO1lBQ1QsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixHQUFHLEVBQUUsR0FBRztZQUNSLE1BQU0sRUFBRSxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUM7WUFDNUIsTUFBTSxFQUFFLElBQUk7U0FDYixDQUFDLENBQUM7UUFFSCxNQUFNLEtBQUssR0FBRyx5QkFBbUIsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDekQsS0FBSyxDQUFDLElBQUksQ0FBQztZQUNULE9BQU8sRUFBRSxDQUFDLE1BQU0sRUFBRSxTQUFTLENBQUM7WUFDNUIsV0FBVyxFQUFFO2dCQUNYLElBQUksRUFBRSxRQUFRO2dCQUNkLElBQUksRUFBRSxRQUFRO2FBQ2Y7WUFDRCxPQUFPLEVBQUUsQ0FBQyxFQUFFLFFBQVEsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLGlCQUFpQixFQUFFLENBQUM7WUFDdkUsZ0JBQWdCLEVBQUUsb0JBQW9CO1lBQ3RDLElBQUksRUFBRSxZQUFZO1NBQ25CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUU7WUFDekMsS0FBSyxFQUFFLE1BQU07WUFDYixJQUFJLEVBQUUsWUFBWTtZQUNsQixJQUFJLEVBQUUsc0NBQXNDO1lBQzVDLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLElBQUksRUFBRSxvQkFBb0I7WUFDMUIsUUFBUTtZQUNSLE1BQU0sRUFBRSxTQUFTO1NBQ2xCLEVBQUUsRUFBRSxLQUFLLEVBQUUsQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLE1BQU0sRUFBRSxTQUFTLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsZ0NBQWdDLENBQUMsSUFBVTtRQUN6QyxNQUFNLE9BQU8sR0FBRyxjQUFjLENBQUM7UUFDL0IsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ25FLE1BQU0sRUFBRSxDQUFDO1lBQ1QsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsT0FBTyxFQUFFLENBQUM7WUFDeEMsR0FBRyxFQUFFLEdBQUc7WUFDUixNQUFNLEVBQUUsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDO1lBQzVCLE1BQU0sRUFBRSxJQUFJO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDO1FBQ2pDLE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsZ0JBQVUsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUM5RCxlQUFlLENBQUMsU0FBUyxDQUFDLEdBQUcsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRTNDLE1BQU0sS0FBSyxHQUFHLHlCQUFtQixDQUFDLFNBQVMsQ0FBQyxhQUFhLEVBQUU7WUFDekQsU0FBUyxFQUFFO2dCQUNULFFBQVEsRUFBRSxVQUFVO2FBQ3JCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLElBQUksRUFBRSxDQUFDO1FBRWIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUU7WUFDbkQsT0FBTyxFQUFFLElBQUk7WUFDYixhQUFhLEVBQUUsbUJBQW1CO1lBQ2xDLGFBQWE7U0FDZCxDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFO1lBQ3BELEtBQUssRUFBRSxNQUFNO1lBQ2IsT0FBTztTQUNSLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBEQUEwRCxDQUFDLElBQVU7UUFDbkUsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQzdDLE1BQU0sRUFBRSxDQUFDO1lBQ1QsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixHQUFHLEVBQUUsR0FBRztZQUNSLE1BQU0sRUFBRSxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUM7WUFDNUIsTUFBTSxFQUFFLElBQUk7U0FDYixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLHlCQUFtQixDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsRUFBRSxxREFBcUQsQ0FBQyxDQUFDO1FBQ3ZILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUM3QyxNQUFNLEVBQUUsQ0FBQztZQUNULE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDN0IsR0FBRyxFQUFFLEdBQUc7WUFDUixNQUFNLEVBQUUsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDO1lBQzVCLE1BQU0sRUFBRSxJQUFJO1lBQ1osS0FBSyxFQUFFLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQztTQUNqQyxDQUFDLENBQUM7UUFFSCxNQUFNLEtBQUssR0FBRyx5QkFBbUIsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDekQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDaEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJCQUEyQixDQUFDLElBQVU7UUFDcEMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQzdDLE1BQU0sRUFBRSxDQUFDLENBQUM7WUFDVixNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDN0IsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLEdBQUcsRUFBRSxHQUFHO1lBQ1IsTUFBTSxFQUFFLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQztZQUM1QixNQUFNLEVBQUUsSUFBSTtTQUNiLENBQUMsQ0FBQztRQUVILE1BQU0sS0FBSyxHQUFHLHlCQUFtQixDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN6RCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOERBQThELENBQUMsSUFBVTtRQUN2RSxNQUFNLEtBQUssR0FBRyx5QkFBbUIsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFekQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1FQUFtRSxDQUFDLElBQVU7UUFDNUUsTUFBTSxPQUFPLEdBQUcsY0FBYyxDQUFDO1FBQy9CLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUM3QyxNQUFNLEVBQUUsQ0FBQztZQUNULE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLE9BQU8sRUFBRSxDQUFDO1lBQ3hDLEdBQUcsRUFBRSxHQUFHO1lBQ1IsTUFBTSxFQUFFLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQztZQUM1QixNQUFNLEVBQUUsSUFBSTtTQUNiLENBQUMsQ0FBQztRQUNILE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQztRQUNqQyxNQUFNLGVBQWUsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLGdCQUFVLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDOUQsZUFBZSxDQUFDLFNBQVMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUUzQyxNQUFNLEtBQUssR0FBRyx5QkFBbUIsQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFM0QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLEtBQUssQ0FBQyxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbkcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjaGlsZF9wcm9jZXNzIGZyb20gJ2NoaWxkX3Byb2Nlc3MnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCAqIGFzIHNpbm9uIGZyb20gJ3Npbm9uJztcbmltcG9ydCB7IEJ1bmRsaW5nRG9ja2VySW1hZ2UsIEZpbGVTeXN0ZW0gfSBmcm9tICcuLi9saWInO1xuXG5leHBvcnQgPSB7XG4gICd0ZWFyRG93bicoY2FsbGJhY2s6IGFueSkge1xuICAgIHNpbm9uLnJlc3RvcmUoKTtcbiAgICBjYWxsYmFjaygpO1xuICB9LFxuXG4gICdidW5kbGluZyB3aXRoIGltYWdlIGZyb20gcmVnaXN0cnknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzcGF3blN5bmNTdHViID0gc2lub24uc3R1YihjaGlsZF9wcm9jZXNzLCAnc3Bhd25TeW5jJykucmV0dXJucyh7XG4gICAgICBzdGF0dXM6IDAsXG4gICAgICBzdGRlcnI6IEJ1ZmZlci5mcm9tKCdzdGRlcnInKSxcbiAgICAgIHN0ZG91dDogQnVmZmVyLmZyb20oJ3N0ZG91dCcpLFxuICAgICAgcGlkOiAxMjMsXG4gICAgICBvdXRwdXQ6IFsnc3Rkb3V0JywgJ3N0ZGVyciddLFxuICAgICAgc2lnbmFsOiBudWxsLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaW1hZ2UgPSBCdW5kbGluZ0RvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnYWxwaW5lJyk7XG4gICAgaW1hZ2UuX3J1bih7XG4gICAgICBjb21tYW5kOiBbJ2Nvb2wnLCAnY29tbWFuZCddLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgVkFSMTogJ3ZhbHVlMScsXG4gICAgICAgIFZBUjI6ICd2YWx1ZTInLFxuICAgICAgfSxcbiAgICAgIHZvbHVtZXM6IFt7IGhvc3RQYXRoOiAnL2hvc3QtcGF0aCcsIGNvbnRhaW5lclBhdGg6ICcvY29udGFpbmVyLXBhdGgnIH1dLFxuICAgICAgd29ya2luZ0RpcmVjdG9yeTogJy93b3JraW5nLWRpcmVjdG9yeScsXG4gICAgICB1c2VyOiAndXNlcjpncm91cCcsXG4gICAgfSk7XG5cbiAgICB0ZXN0Lm9rKHNwYXduU3luY1N0dWIuY2FsbGVkV2l0aCgnZG9ja2VyJywgW1xuICAgICAgJ3J1bicsICctLXJtJyxcbiAgICAgICctdScsICd1c2VyOmdyb3VwJyxcbiAgICAgICctdicsICcvaG9zdC1wYXRoOi9jb250YWluZXItcGF0aDpkZWxlZ2F0ZWQnLFxuICAgICAgJy0tZW52JywgJ1ZBUjE9dmFsdWUxJyxcbiAgICAgICctLWVudicsICdWQVIyPXZhbHVlMicsXG4gICAgICAnLXcnLCAnL3dvcmtpbmctZGlyZWN0b3J5JyxcbiAgICAgICdhbHBpbmUnLFxuICAgICAgJ2Nvb2wnLCAnY29tbWFuZCcsXG4gICAgXSwgeyBzdGRpbzogWydpZ25vcmUnLCBwcm9jZXNzLnN0ZGVyciwgJ2luaGVyaXQnXSB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2J1bmRsaW5nIHdpdGggaW1hZ2UgZnJvbSBhc3NldCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IGltYWdlSWQgPSAnYWJjZGVmMTIzNDU2JztcbiAgICBjb25zdCBzcGF3blN5bmNTdHViID0gc2lub24uc3R1YihjaGlsZF9wcm9jZXNzLCAnc3Bhd25TeW5jJykucmV0dXJucyh7XG4gICAgICBzdGF0dXM6IDAsXG4gICAgICBzdGRlcnI6IEJ1ZmZlci5mcm9tKCdzdGRlcnInKSxcbiAgICAgIHN0ZG91dDogQnVmZmVyLmZyb20oYHNoYTI1Njoke2ltYWdlSWR9YCksXG4gICAgICBwaWQ6IDEyMyxcbiAgICAgIG91dHB1dDogWydzdGRvdXQnLCAnc3RkZXJyJ10sXG4gICAgICBzaWduYWw6IG51bGwsXG4gICAgfSk7XG5cbiAgICBjb25zdCBpbWFnZUhhc2ggPSAnMTIzNDU2YWJjZGVmJztcbiAgICBjb25zdCBmaW5nZXJwcmludFN0dWIgPSBzaW5vbi5zdHViKEZpbGVTeXN0ZW0sICdmaW5nZXJwcmludCcpO1xuICAgIGZpbmdlcnByaW50U3R1Yi5jYWxsc0Zha2UoKCkgPT4gaW1hZ2VIYXNoKTtcblxuICAgIGNvbnN0IGltYWdlID0gQnVuZGxpbmdEb2NrZXJJbWFnZS5mcm9tQXNzZXQoJ2RvY2tlci1wYXRoJywge1xuICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgIFRFU1RfQVJHOiAnY2RrLXRlc3QnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBpbWFnZS5fcnVuKCk7XG5cbiAgICB0ZXN0Lm9rKHNwYXduU3luY1N0dWIuZmlyc3RDYWxsLmNhbGxlZFdpdGgoJ2RvY2tlcicsIFtcbiAgICAgICdidWlsZCcsICctcScsXG4gICAgICAnLS1idWlsZC1hcmcnLCAnVEVTVF9BUkc9Y2RrLXRlc3QnLFxuICAgICAgJ2RvY2tlci1wYXRoJyxcbiAgICBdKSk7XG5cbiAgICB0ZXN0Lm9rKHNwYXduU3luY1N0dWIuc2Vjb25kQ2FsbC5jYWxsZWRXaXRoKCdkb2NrZXInLCBbXG4gICAgICAncnVuJywgJy0tcm0nLFxuICAgICAgaW1hZ2VJZCxcbiAgICBdKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rocm93cyBpZiBpbWFnZSBpZCBjYW5ub3QgYmUgZXh0cmFjdGVkIGZyb20gYnVpbGQgb3V0cHV0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgc2lub24uc3R1YihjaGlsZF9wcm9jZXNzLCAnc3Bhd25TeW5jJykucmV0dXJucyh7XG4gICAgICBzdGF0dXM6IDAsXG4gICAgICBzdGRlcnI6IEJ1ZmZlci5mcm9tKCdzdGRlcnInKSxcbiAgICAgIHN0ZG91dDogQnVmZmVyLmZyb20oJ3N0ZG91dCcpLFxuICAgICAgcGlkOiAxMjMsXG4gICAgICBvdXRwdXQ6IFsnc3Rkb3V0JywgJ3N0ZGVyciddLFxuICAgICAgc2lnbmFsOiBudWxsLFxuICAgIH0pO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gQnVuZGxpbmdEb2NrZXJJbWFnZS5mcm9tQXNzZXQoJ2RvY2tlci1wYXRoJyksIC9GYWlsZWQgdG8gZXh0cmFjdCBpbWFnZSBJRCBmcm9tIERvY2tlciBidWlsZCBvdXRwdXQvKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndGhyb3dzIGluIGNhc2Ugb2Ygc3Bhd25TeW5jIGVycm9yJyh0ZXN0OiBUZXN0KSB7XG4gICAgc2lub24uc3R1YihjaGlsZF9wcm9jZXNzLCAnc3Bhd25TeW5jJykucmV0dXJucyh7XG4gICAgICBzdGF0dXM6IDAsXG4gICAgICBzdGRlcnI6IEJ1ZmZlci5mcm9tKCdzdGRlcnInKSxcbiAgICAgIHN0ZG91dDogQnVmZmVyLmZyb20oJ3N0ZG91dCcpLFxuICAgICAgcGlkOiAxMjMsXG4gICAgICBvdXRwdXQ6IFsnc3Rkb3V0JywgJ3N0ZGVyciddLFxuICAgICAgc2lnbmFsOiBudWxsLFxuICAgICAgZXJyb3I6IG5ldyBFcnJvcignVW5rbm93bkVycm9yJyksXG4gICAgfSk7XG5cbiAgICBjb25zdCBpbWFnZSA9IEJ1bmRsaW5nRG9ja2VySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdhbHBpbmUnKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBpbWFnZS5fcnVuKCksIC9Vbmtub3duRXJyb3IvKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndGhyb3dzIGlmIHN0YXR1cyBpcyBub3QgMCcodGVzdDogVGVzdCkge1xuICAgIHNpbm9uLnN0dWIoY2hpbGRfcHJvY2VzcywgJ3NwYXduU3luYycpLnJldHVybnMoe1xuICAgICAgc3RhdHVzOiAtMSxcbiAgICAgIHN0ZGVycjogQnVmZmVyLmZyb20oJ3N0ZGVycicpLFxuICAgICAgc3Rkb3V0OiBCdWZmZXIuZnJvbSgnc3Rkb3V0JyksXG4gICAgICBwaWQ6IDEyMyxcbiAgICAgIG91dHB1dDogWydzdGRvdXQnLCAnc3RkZXJyJ10sXG4gICAgICBzaWduYWw6IG51bGwsXG4gICAgfSk7XG5cbiAgICBjb25zdCBpbWFnZSA9IEJ1bmRsaW5nRG9ja2VySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdhbHBpbmUnKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBpbWFnZS5fcnVuKCksIC9cXFtTdGF0dXMgLTFcXF0vKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQnVuZGxlckRvY2tlckltYWdlIGpzb24gaXMgdGhlIGJ1bmRsZXIgaW1hZ2UgbmFtZSBieSBkZWZhdWx0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgaW1hZ2UgPSBCdW5kbGluZ0RvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnYWxwaW5lJyk7XG5cbiAgICB0ZXN0LmVxdWFscyhpbWFnZS50b0pTT04oKSwgJ2FscGluZScpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdCdW5kbGVyRG9ja2VySW1hZ2UganNvbiBpcyB0aGUgYnVuZGxlciBpbWFnZSBpZiBidWlsZGluZyBhbiBpbWFnZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IGltYWdlSWQgPSAnYWJjZGVmMTIzNDU2JztcbiAgICBzaW5vbi5zdHViKGNoaWxkX3Byb2Nlc3MsICdzcGF3blN5bmMnKS5yZXR1cm5zKHtcbiAgICAgIHN0YXR1czogMCxcbiAgICAgIHN0ZGVycjogQnVmZmVyLmZyb20oJ3N0ZGVycicpLFxuICAgICAgc3Rkb3V0OiBCdWZmZXIuZnJvbShgc2hhMjU2OiR7aW1hZ2VJZH1gKSxcbiAgICAgIHBpZDogMTIzLFxuICAgICAgb3V0cHV0OiBbJ3N0ZG91dCcsICdzdGRlcnInXSxcbiAgICAgIHNpZ25hbDogbnVsbCxcbiAgICB9KTtcbiAgICBjb25zdCBpbWFnZUhhc2ggPSAnMTIzNDU2YWJjZGVmJztcbiAgICBjb25zdCBmaW5nZXJwcmludFN0dWIgPSBzaW5vbi5zdHViKEZpbGVTeXN0ZW0sICdmaW5nZXJwcmludCcpO1xuICAgIGZpbmdlcnByaW50U3R1Yi5jYWxsc0Zha2UoKCkgPT4gaW1hZ2VIYXNoKTtcblxuICAgIGNvbnN0IGltYWdlID0gQnVuZGxpbmdEb2NrZXJJbWFnZS5mcm9tQXNzZXQoJ2RvY2tlci1wYXRoJyk7XG5cbiAgICB0ZXN0LmVxdWFscyhpbWFnZS5pbWFnZSwgaW1hZ2VJZCk7XG4gICAgdGVzdC5lcXVhbHMoaW1hZ2UudG9KU09OKCksIGltYWdlSGFzaCk7XG4gICAgdGVzdC5vayhmaW5nZXJwcmludFN0dWIuY2FsbGVkV2l0aCgnZG9ja2VyLXBhdGgnLCBzaW5vbi5tYXRjaCh7IGV4dHJhSGFzaDogSlNPTi5zdHJpbmdpZnkoe30pIH0pKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59O1xuIl19