"use strict";
const lib_1 = require("../lib");
const annotations_1 = require("../lib/annotations");
const restore = process.env.CDK_BLOCK_DEPRECATIONS;
function getWarnings(casm) {
    var _a, _b;
    const result = new Array();
    for (const stack of Object.values((_a = casm.manifest.artifacts) !== null && _a !== void 0 ? _a : {})) {
        for (const [path, md] of Object.entries((_b = stack.metadata) !== null && _b !== void 0 ? _b : {})) {
            for (const x of md) {
                if (x.type === 'aws:cdk:warning') {
                    result.push({ path, message: x.data });
                }
            }
        }
    }
    return result;
}
module.exports = {
    'tearDown'(cb) {
        process.env.CDK_BLOCK_DEPRECATIONS = restore; // restore to the original value
        cb();
    },
    'addDeprecation() annotates the usage of a deprecated API'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'MyStack');
        const c1 = new lib_1.Construct(stack, 'Hello');
        // WHEN
        delete process.env.CDK_BLOCK_DEPRECATIONS;
        annotations_1.Annotations.of(c1).addDeprecation('@aws-cdk/core.Construct.node', 'use @aws-cdk.Construct.construct instead');
        // THEN
        test.deepEqual(getWarnings(app.synth()), [
            {
                path: '/MyStack/Hello',
                message: 'The API @aws-cdk/core.Construct.node is deprecated: use @aws-cdk.Construct.construct instead. This API will be removed in the next major release',
            },
        ]);
        test.done();
    },
    'deduplicated per node based on "api"'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'MyStack1');
        const stack2 = new lib_1.Stack(app, 'MyStack2');
        const c1 = new lib_1.Construct(stack1, 'Hello');
        const c2 = new lib_1.Construct(stack1, 'World');
        const c3 = new lib_1.Construct(stack2, 'FooBar');
        // WHEN
        delete process.env.CDK_BLOCK_DEPRECATIONS;
        annotations_1.Annotations.of(c1).addDeprecation('@aws-cdk/core.Construct.node', 'use @aws-cdk.Construct.construct instead');
        annotations_1.Annotations.of(c2).addDeprecation('@aws-cdk/core.Construct.node', 'use @aws-cdk.Construct.construct instead');
        annotations_1.Annotations.of(c1).addDeprecation('@aws-cdk/core.Construct.node', 'use @aws-cdk.Construct.construct instead');
        annotations_1.Annotations.of(c3).addDeprecation('@aws-cdk/core.Construct.node', 'use @aws-cdk.Construct.construct instead');
        annotations_1.Annotations.of(c1).addDeprecation('@aws-cdk/core.Construct.node', 'use @aws-cdk.Construct.construct instead');
        annotations_1.Annotations.of(c1).addDeprecation('@aws-cdk/core.Construct.node', 'use @aws-cdk.Construct.construct instead');
        // THEN
        test.deepEqual(getWarnings(app.synth()), [
            {
                path: '/MyStack1/Hello',
                message: 'The API @aws-cdk/core.Construct.node is deprecated: use @aws-cdk.Construct.construct instead. This API will be removed in the next major release',
            },
            {
                path: '/MyStack1/World',
                message: 'The API @aws-cdk/core.Construct.node is deprecated: use @aws-cdk.Construct.construct instead. This API will be removed in the next major release',
            },
            {
                path: '/MyStack2/FooBar',
                message: 'The API @aws-cdk/core.Construct.node is deprecated: use @aws-cdk.Construct.construct instead. This API will be removed in the next major release',
            },
        ]);
        test.done();
    },
    'CDK_BLOCK_DEPRECATIONS will throw if a deprecated API is used'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'MyStack');
        const c1 = new lib_1.Construct(stack, 'Hello');
        // THEN
        process.env.CDK_BLOCK_DEPRECATIONS = '1';
        test.throws(() => annotations_1.Annotations.of(c1).addDeprecation('foo', 'bar'), /MyStack\/Hello: The API foo is deprecated: bar\. This API will be removed in the next major release/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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