"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const fs_1 = require("../../lib/fs");
function tree(dir, depth = '') {
    const lines = new Array();
    for (const file of fs.readdirSync(dir).sort()) {
        const filePath = path.join(dir, file);
        const stat = fs.lstatSync(filePath);
        if (stat.isSymbolicLink()) {
            const linkDest = fs.readlinkSync(filePath);
            lines.push(depth + file + ' => ' + linkDest);
        }
        else if (stat.isDirectory()) {
            lines.push(depth + file + ' (D)');
            lines.push(...tree(filePath, depth + '    '));
        }
        else {
            lines.push(depth + file);
        }
    }
    return lines;
}
module.exports = {
    'Default: copies all files and subdirectories, with default follow mode is "External"'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        fs_1.FileSystem.copyDirectory(path.join(__dirname, 'fixtures', 'test1'), outdir);
        // THEN
        test.deepEqual(tree(outdir), [
            'external-link.txt',
            'file1.txt',
            'local-link.txt => file1.txt',
            'subdir (D)',
            '    file2.txt',
            'subdir2 (D)',
            '    empty-subdir (D)',
            '        .hidden',
            '    subdir3 (D)',
            '        file3.txt',
        ]);
        test.done();
    },
    'Always: follow all symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        fs_1.FileSystem.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: fs_1.SymlinkFollowMode.ALWAYS,
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link (D)',
            '    file2.txt',
            'external-link.txt',
            'indirect-external-link.txt',
            'local-dir-link (D)',
            '    file-in-subdir.txt',
            'local-link.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt',
        ]);
        test.done();
    },
    'Never: do not follow all symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        fs_1.FileSystem.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: fs_1.SymlinkFollowMode.NEVER,
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link => ../test1/subdir',
            'external-link.txt => ../test1/subdir2/subdir3/file3.txt',
            'indirect-external-link.txt => external-link.txt',
            'local-dir-link => normal-dir',
            'local-link.txt => normal-file.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt',
        ]);
        test.done();
    },
    'External: follow only external symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        fs_1.FileSystem.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: fs_1.SymlinkFollowMode.EXTERNAL,
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link (D)',
            '    file2.txt',
            'external-link.txt',
            'indirect-external-link.txt => external-link.txt',
            'local-dir-link => normal-dir',
            'local-link.txt => normal-file.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt',
        ]);
        test.done();
    },
    'exclude'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        fs_1.FileSystem.copyDirectory(path.join(__dirname, 'fixtures', 'test1'), outdir, {
            exclude: [
                '*',
                '!subdir2',
                '!subdir2/**/*',
                '.*',
            ],
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'subdir2 (D)',
            '    empty-subdir (D)',
            '    subdir3 (D)',
            '        file3.txt',
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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