"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetadataResource = void 0;
const cxapi = require("@aws-cdk/cx-api");
const region_info_1 = require("@aws-cdk/region-info");
const cfn_condition_1 = require("../cfn-condition");
const cfn_fn_1 = require("../cfn-fn");
const cfn_pseudo_1 = require("../cfn-pseudo");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const lazy_1 = require("../lazy");
const token_1 = require("../token");
const runtime_info_1 = require("./runtime-info");
/**
 * Construct that will render the metadata resource
 */
class MetadataResource extends construct_compat_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        const metadataServiceExists = token_1.Token.isUnresolved(scope.region) || region_info_1.RegionInfo.get(scope.region).cdkMetadataResourceAvailable;
        if (metadataServiceExists) {
            const resource = new cfn_resource_1.CfnResource(this, 'Default', {
                type: 'AWS::CDK::Metadata',
                properties: {
                    Modules: lazy_1.Lazy.stringValue({ produce: () => MetadataResource.modulesProperty() }),
                },
            });
            // In case we don't actually know the region, add a condition to determine it at deploy time
            if (token_1.Token.isUnresolved(scope.region)) {
                const condition = new cfn_condition_1.CfnCondition(this, 'Condition', {
                    expression: makeCdkMetadataAvailableCondition(),
                });
                // To not cause undue template changes
                condition.overrideLogicalId('CDKMetadataAvailable');
                resource.cfnOptions.condition = condition;
            }
        }
    }
    /**
     * Clear the modules cache
     *
     * The next time the MetadataResource is rendered, it will do a lookup of the
     * modules from the NodeJS module cache again.
     *
     * Used only for unit tests.
     */
    static clearModulesCache() {
        this._modulesPropertyCache = undefined;
    }
    /**
     * Calculate the modules property
     */
    static modulesProperty() {
        if (this._modulesPropertyCache === undefined) {
            this._modulesPropertyCache = formatModules(runtime_info_1.collectRuntimeInformation());
        }
        return this._modulesPropertyCache;
    }
}
exports.MetadataResource = MetadataResource;
function makeCdkMetadataAvailableCondition() {
    return cfn_fn_1.Fn.conditionOr(...region_info_1.RegionInfo.regions
        .filter(ri => ri.cdkMetadataResourceAvailable)
        .map(ri => cfn_fn_1.Fn.conditionEquals(cfn_pseudo_1.Aws.REGION, ri.name)));
}
function formatModules(runtime) {
    const modules = new Array();
    // inject toolkit version to list of modules
    const cliVersion = process.env[cxapi.CLI_VERSION_ENV];
    if (cliVersion) {
        modules.push(`aws-cdk=${cliVersion}`);
    }
    for (const key of Object.keys(runtime.libraries).sort()) {
        modules.push(`${key}=${runtime.libraries[key]}`);
    }
    return modules.join(',');
}
//# sourceMappingURL=data:application/json;base64,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