"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResource = void 0;
const cfn_resource_1 = require("./cfn-resource");
const removal_policy_1 = require("./removal-policy");
const resource_1 = require("./resource");
const token_1 = require("./token");
/**
 * Custom resource that is implemented using a Lambda
 *
 * As a custom resource author, you should be publishing a subclass of this class
 * that hides the choice of provider, and accepts a strongly-typed properties
 * object with the properties your provider accepts.
 *
 * @resource AWS::CloudFormation::CustomResource
 */
class CustomResource extends resource_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const type = renderResourceType(props.resourceType);
        const pascalCaseProperties = (_a = props.pascalCaseProperties) !== null && _a !== void 0 ? _a : false;
        const properties = pascalCaseProperties ? uppercaseProperties(props.properties || {}) : (props.properties || {});
        this.resource = new cfn_resource_1.CfnResource(this, 'Default', {
            type,
            properties: {
                ServiceToken: props.serviceToken,
                ...properties,
            },
        });
        this.resource.applyRemovalPolicy(props.removalPolicy, {
            default: removal_policy_1.RemovalPolicy.DESTROY,
        });
    }
    /**
     * The physical name of this custom resource.
     */
    get ref() {
        return this.resource.ref;
    }
    /**
     * Returns the value of an attribute of the custom resource of an arbitrary
     * type. Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     */
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    /**
     * Returns the value of an attribute of the custom resource of type string.
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt` encoded as a string.
     */
    getAttString(attributeName) {
        return token_1.Token.asString(this.getAtt(attributeName));
    }
}
exports.CustomResource = CustomResource;
/**
 * Uppercase the first letter of every property name
 *
 * It's customary for CloudFormation properties to start with capitals, and our
 * properties to start with lowercase, so this function translates from one
 * to the other
 */
function uppercaseProperties(props) {
    const ret = {};
    Object.keys(props).forEach(key => {
        const upper = key.substr(0, 1).toUpperCase() + key.substr(1);
        ret[upper] = props[key];
    });
    return ret;
}
function renderResourceType(resourceType) {
    if (!resourceType) {
        return 'AWS::CloudFormation::CustomResource';
    }
    if (!resourceType.startsWith('Custom::')) {
        throw new Error(`Custom resource type must begin with "Custom::" (${resourceType})`);
    }
    const typeName = resourceType.substr(resourceType.indexOf('::') + 2);
    if (typeName.length > 60) {
        throw new Error(`Custom resource type length > 60 (${resourceType})`);
    }
    if (!/^[a-z0-9_@-]+$/i.test(typeName)) {
        throw new Error(`Custom resource type name can only include alphanumeric characters and _@- (${typeName})`);
    }
    return resourceType;
}
//# sourceMappingURL=data:application/json;base64,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