import { Construct } from '../construct-compat';
import { Duration } from '../duration';
import { Size } from '../size';
/**
 * Initialization properties for `CustomResourceProvider`.
 *
 * @experimental
 */
export interface CustomResourceProviderProps {
    /**
     * A local file system directory with the provider's code. The code will be
     * bundled into a zip asset and wired to the provider's AWS Lambda function.
     */
    readonly codeDirectory: string;
    /**
     * The AWS Lambda runtime and version to use for the provider.
     */
    readonly runtime: CustomResourceProviderRuntime;
    /**
     * A set of IAM policy statements to include in the inline policy of the
     * provider's lambda function.
     *
     * @default - no additional inline policy
     *
     * @example
     *
     *   policyStatements: [ { Effect: 'Allow', Action: 's3:PutObject*', Resource: '*' } ]
     *
     */
    readonly policyStatements?: any[];
    /**
     * AWS Lambda timeout for the provider.
     *
     * @default Duration.minutes(15)
     */
    readonly timeout?: Duration;
    /**
     * The amount of memory that your function has access to. Increasing the
     * function's memory also increases its CPU allocation.
     *
     * @default Size.mebibytes(128)
     */
    readonly memorySize?: Size;
}
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 * @experimental
 */
export declare enum CustomResourceProviderRuntime {
    /**
     * Node.js 12.x
     */
    NODEJS_12 = "nodejs12"
}
/**
 * An AWS-Lambda backed custom resource provider.
 *
 * @experimental
 */
export declare class CustomResourceProvider extends Construct {
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource
     * provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreate(scope: Construct, uniqueid: string, props: CustomResourceProviderProps): string;
    /**
     * The ARN of the provider's AWS Lambda function which should be used as the
     * `serviceToken` when defining a custom resource.
     *
     * @example
     *
     *   new CustomResource(this, 'MyCustomResource', {
     *     // ...
     *     serviceToken: provider.serviceToken // <--- here
     *   })
     *
     */
    readonly serviceToken: string;
    protected constructor(scope: Construct, id: string, props: CustomResourceProviderProps);
}
