"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnCodeDeployBlueGreenHook = exports.CfnTrafficRoutingType = void 0;
const cfn_hook_1 = require("./cfn-hook");
/**
 * The possible types of traffic shifting for the blue-green deployment configuration.
 * The type of the {@link CfnTrafficRoutingConfig.type} property.
 */
var CfnTrafficRoutingType;
(function (CfnTrafficRoutingType) {
    /**
     * Switch from blue to green at once.
     */
    CfnTrafficRoutingType["ALL_AT_ONCE"] = "AllAtOnce";
    /**
     * Specifies a configuration that shifts traffic from blue to green in two increments.
     */
    CfnTrafficRoutingType["TIME_BASED_CANARY"] = "TimeBasedCanary";
    /**
     * Specifies a configuration that shifts traffic from blue to green in equal increments,
     * with an equal number of minutes between each increment.
     */
    CfnTrafficRoutingType["TIME_BASED_LINEAR"] = "TimeBasedLinear";
})(CfnTrafficRoutingType = exports.CfnTrafficRoutingType || (exports.CfnTrafficRoutingType = {}));
/**
 * A CloudFormation Hook for CodeDeploy blue-green ECS deployments.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/blue-green.html#blue-green-template-reference
 */
class CfnCodeDeployBlueGreenHook extends cfn_hook_1.CfnHook {
    /**
     * Creates a new CodeDeploy blue-green ECS Hook.
     *
     * @param scope the scope to create the hook in (usually the containing Stack object)
     * @param id the identifier of the construct - will be used to generate the logical ID of the Hook
     * @param props the properties of the Hook
     */
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'AWS::CodeDeploy::BlueGreen',
        });
        this._serviceRole = props.serviceRole;
        this._applications = props.applications;
        this._trafficRoutingConfig = props.trafficRoutingConfig;
        this._additionalOptions = props.additionalOptions;
        this._lifecycleEventHooks = props.lifecycleEventHooks;
    }
    /**
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope, id, hookAttributes, options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        hookAttributes = hookAttributes || {};
        const hookProperties = options.parser.parseValue(hookAttributes.Properties);
        return new CfnCodeDeployBlueGreenHook(scope, id, {
            serviceRole: hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.ServiceRole,
            applications: (_a = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.Applications) === null || _a === void 0 ? void 0 : _a.map(applicationFromCloudFormation),
            trafficRoutingConfig: {
                type: (_b = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.Type,
                timeBasedCanary: {
                    stepPercentage: (_d = (_c = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _c === void 0 ? void 0 : _c.TimeBasedCanary) === null || _d === void 0 ? void 0 : _d.StepPercentage,
                    bakeTimeMins: (_f = (_e = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _e === void 0 ? void 0 : _e.TimeBasedCanary) === null || _f === void 0 ? void 0 : _f.BakeTimeMins,
                },
                timeBasedLinear: {
                    stepPercentage: (_h = (_g = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _g === void 0 ? void 0 : _g.TimeBasedLinear) === null || _h === void 0 ? void 0 : _h.StepPercentage,
                    bakeTimeMins: (_k = (_j = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _j === void 0 ? void 0 : _j.TimeBasedLinear) === null || _k === void 0 ? void 0 : _k.BakeTimeMins,
                },
            },
            additionalOptions: {
                terminationWaitTimeInMinutes: (_l = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.AdditionalOptions) === null || _l === void 0 ? void 0 : _l.TerminationWaitTimeInMinutes,
            },
            lifecycleEventHooks: {
                beforeInstall: (_m = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.BeforeInstall,
                afterInstall: (_o = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.AfterInstall,
                afterAllowTestTraffic: (_p = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.AfterAllowTestTraffic,
                beforeAllowTraffic: (_q = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.BeforeAllowTraffic,
                afterAllowTraffic: (_r = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _r === void 0 ? void 0 : _r.AfterAllowTraffic,
            },
        });
        function applicationFromCloudFormation(app) {
            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w;
            const target = findResource((_a = app === null || app === void 0 ? void 0 : app.Target) === null || _a === void 0 ? void 0 : _a.LogicalID);
            const taskDefinitions = (_c = (_b = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _b === void 0 ? void 0 : _b.TaskDefinitions) === null || _c === void 0 ? void 0 : _c.map((td) => findResource(td));
            const taskSets = (_e = (_d = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _d === void 0 ? void 0 : _d.TaskSets) === null || _e === void 0 ? void 0 : _e.map((ts) => findResource(ts));
            const prodTrafficRoute = findResource((_h = (_g = (_f = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _f === void 0 ? void 0 : _f.TrafficRouting) === null || _g === void 0 ? void 0 : _g.ProdTrafficRoute) === null || _h === void 0 ? void 0 : _h.LogicalID);
            const testTrafficRoute = findResource((_l = (_k = (_j = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _j === void 0 ? void 0 : _j.TrafficRouting) === null || _k === void 0 ? void 0 : _k.TestTrafficRoute) === null || _l === void 0 ? void 0 : _l.LogicalID);
            const targetGroups = (_p = (_o = (_m = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _m === void 0 ? void 0 : _m.TrafficRouting) === null || _o === void 0 ? void 0 : _o.TargetGroups) === null || _p === void 0 ? void 0 : _p.map((tg) => findResource(tg));
            return {
                target: {
                    type: (_q = app === null || app === void 0 ? void 0 : app.Target) === null || _q === void 0 ? void 0 : _q.Type,
                    logicalId: target === null || target === void 0 ? void 0 : target.logicalId,
                },
                ecsAttributes: {
                    taskDefinitions: taskDefinitions === null || taskDefinitions === void 0 ? void 0 : taskDefinitions.map(td => td === null || td === void 0 ? void 0 : td.logicalId),
                    taskSets: taskSets === null || taskSets === void 0 ? void 0 : taskSets.map(ts => ts === null || ts === void 0 ? void 0 : ts.logicalId),
                    trafficRouting: {
                        prodTrafficRoute: {
                            type: (_t = (_s = (_r = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _r === void 0 ? void 0 : _r.TrafficRouting) === null || _s === void 0 ? void 0 : _s.ProdTrafficRoute) === null || _t === void 0 ? void 0 : _t.Type,
                            logicalId: prodTrafficRoute === null || prodTrafficRoute === void 0 ? void 0 : prodTrafficRoute.logicalId,
                        },
                        testTrafficRoute: {
                            type: (_w = (_v = (_u = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _u === void 0 ? void 0 : _u.TrafficRouting) === null || _v === void 0 ? void 0 : _v.TestTrafficRoute) === null || _w === void 0 ? void 0 : _w.Type,
                            logicalId: testTrafficRoute === null || testTrafficRoute === void 0 ? void 0 : testTrafficRoute.logicalId,
                        },
                        targetGroups: targetGroups === null || targetGroups === void 0 ? void 0 : targetGroups.map((tg) => tg === null || tg === void 0 ? void 0 : tg.logicalId),
                    },
                },
            };
        }
        function findResource(logicalId) {
            if (logicalId == null) {
                return undefined;
            }
            const ret = options.parser.finder.findResource(logicalId);
            if (!ret) {
                throw new Error(`Hook '${id}' references resource '${logicalId}' that was not found in the template`);
            }
            return ret;
        }
    }
    /**
     * The IAM Role for CloudFormation to use to perform blue-green deployments.
     */
    get serviceRole() {
        return this._serviceRole;
    }
    set serviceRole(serviceRole) {
        this._serviceRole = serviceRole;
    }
    /**
     * Properties of the Amazon ECS applications being deployed.
     */
    get applications() {
        return this._applications;
    }
    set applications(value) {
        this._applications = value;
    }
    /**
     * Traffic routing configuration settings.
     *
     * @default - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time
     */
    get trafficRoutingConfig() {
        return this._trafficRoutingConfig;
    }
    set trafficRoutingConfig(value) {
        this._trafficRoutingConfig = value;
    }
    /**
     * Additional options for the blue/green deployment.
     *
     * @default - no additional options
     */
    get additionalOptions() {
        return this._additionalOptions;
    }
    set additionalOptions(value) {
        this._additionalOptions = value;
    }
    /**
     * Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment.
     * You can use the same function or a different one for deployment lifecycle events.
     * Following completion of the validation tests,
     * the Lambda {@link CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic}
     * function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'.
     *
     * @default - no lifecycle event hooks
     */
    get lifecycleEventHooks() {
        return this._lifecycleEventHooks;
    }
    set lifecycleEventHooks(value) {
        this._lifecycleEventHooks = value;
    }
    renderProperties(_props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        return {
            ServiceRole: this.serviceRole,
            Applications: this.applications.map((app) => ({
                Target: {
                    Type: app.target.type,
                    LogicalID: app.target.logicalId,
                },
                ECSAttributes: {
                    TaskDefinitions: app.ecsAttributes.taskDefinitions,
                    TaskSets: app.ecsAttributes.taskSets,
                    TrafficRouting: {
                        ProdTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.prodTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.prodTrafficRoute.logicalId,
                        },
                        TestTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.testTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.testTrafficRoute.logicalId,
                        },
                        TargetGroups: app.ecsAttributes.trafficRouting.targetGroups,
                    },
                },
            })),
            TrafficRoutingConfig: {
                Type: (_a = this.trafficRoutingConfig) === null || _a === void 0 ? void 0 : _a.type,
                TimeBasedCanary: {
                    StepPercentage: (_c = (_b = this.trafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.timeBasedCanary) === null || _c === void 0 ? void 0 : _c.stepPercentage,
                    BakeTimeMins: (_e = (_d = this.trafficRoutingConfig) === null || _d === void 0 ? void 0 : _d.timeBasedCanary) === null || _e === void 0 ? void 0 : _e.bakeTimeMins,
                },
                TimeBasedLinear: {
                    StepPercentage: (_g = (_f = this.trafficRoutingConfig) === null || _f === void 0 ? void 0 : _f.timeBasedLinear) === null || _g === void 0 ? void 0 : _g.stepPercentage,
                    BakeTimeMins: (_j = (_h = this.trafficRoutingConfig) === null || _h === void 0 ? void 0 : _h.timeBasedLinear) === null || _j === void 0 ? void 0 : _j.bakeTimeMins,
                },
            },
            AdditionalOptions: {
                TerminationWaitTimeInMinutes: (_k = this.additionalOptions) === null || _k === void 0 ? void 0 : _k.terminationWaitTimeInMinutes,
            },
            LifecycleEventHooks: {
                BeforeInstall: (_l = this.lifecycleEventHooks) === null || _l === void 0 ? void 0 : _l.beforeInstall,
                AfterInstall: (_m = this.lifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.afterInstall,
                AfterAllowTestTraffic: (_o = this.lifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.afterAllowTestTraffic,
                BeforeAllowTraffic: (_p = this.lifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.beforeAllowTraffic,
                AfterAllowTraffic: (_q = this.lifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.afterAllowTraffic,
            },
        };
    }
}
exports.CfnCodeDeployBlueGreenHook = CfnCodeDeployBlueGreenHook;
//# sourceMappingURL=data:application/json;base64,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