"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerVolumeConsistency = exports.BundlingDockerImage = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("./fs");
/**
 * A Docker image used for asset bundling
 */
class BundlingDockerImage {
    /** @param image The Docker image */
    constructor(image, _imageHash) {
        this.image = image;
        this._imageHash = _imageHash;
    }
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name
     */
    static fromRegistry(image) {
        return new BundlingDockerImage(image);
    }
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file
     * @param options Docker build options
     */
    static fromAsset(path, options = {}) {
        const buildArgs = options.buildArgs || {};
        const dockerArgs = [
            'build', '-q',
            ...flatten(Object.entries(buildArgs).map(([k, v]) => ['--build-arg', `${k}=${v}`])),
            path,
        ];
        const docker = dockerExec(dockerArgs);
        const match = docker.stdout.toString().match(/sha256:([a-z0-9]+)/);
        if (!match) {
            throw new Error('Failed to extract image ID from Docker build output');
        }
        // Fingerprints the directory containing the Dockerfile we're building and
        // differentiates the fingerprint based on build arguments. We do this so
        // we can provide a stable image hash. Otherwise, the image ID will be
        // different every time the Docker layer cache is cleared, due primarily to
        // timestamps.
        const hash = fs_1.FileSystem.fingerprint(path, { extraHash: JSON.stringify(options) });
        return new BundlingDockerImage(match[1], hash);
    }
    /**
     * Provides a stable representation of this image for JSON serialization.
     *
     * @return The overridden image name if set or image hash name in that order
     */
    toJSON() {
        var _a;
        return (_a = this._imageHash) !== null && _a !== void 0 ? _a : this.image;
    }
    /**
     * Runs a Docker image
     *
     * @internal
     */
    _run(options = {}) {
        const volumes = options.volumes || [];
        const environment = options.environment || {};
        const command = options.command || [];
        const dockerArgs = [
            'run', '--rm',
            ...options.user
                ? ['-u', options.user]
                : [],
            ...flatten(volumes.map(v => { var _a; return ['-v', `${v.hostPath}:${v.containerPath}:${(_a = v.consistency) !== null && _a !== void 0 ? _a : DockerVolumeConsistency.DELEGATED}`]; })),
            ...flatten(Object.entries(environment).map(([k, v]) => ['--env', `${k}=${v}`])),
            ...options.workingDirectory
                ? ['-w', options.workingDirectory]
                : [],
            this.image,
            ...command,
        ];
        dockerExec(dockerArgs, {
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
        });
    }
}
exports.BundlingDockerImage = BundlingDockerImage;
/**
 * Supported Docker volume consistency types. Only valid on macOS due to the way file storage works on Mac
 */
var DockerVolumeConsistency;
(function (DockerVolumeConsistency) {
    /**
     * Read/write operations inside the Docker container are applied immediately on the mounted host machine volumes
     */
    DockerVolumeConsistency["CONSISTENT"] = "consistent";
    /**
     * Read/write operations on mounted Docker volumes are first written inside the container and then synchronized to the host machine
     */
    DockerVolumeConsistency["DELEGATED"] = "delegated";
    /**
     * Read/write operations on mounted Docker volumes are first applied on the host machine and then synchronized to the container
     */
    DockerVolumeConsistency["CACHED"] = "cached";
})(DockerVolumeConsistency = exports.DockerVolumeConsistency || (exports.DockerVolumeConsistency = {}));
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
function dockerExec(args, options) {
    var _a, _b, _c;
    const prog = (_a = process.env.CDK_DOCKER) !== null && _a !== void 0 ? _a : 'docker';
    const proc = child_process_1.spawnSync(prog, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_b = proc.stdout) === null || _b === void 0 ? void 0 : _b.toString().trim()}\n\n\nstderr: ${(_c = proc.stderr) === null || _c === void 0 ? void 0 : _c.toString().trim()}`);
        }
        throw new Error(`${prog} exited with status ${proc.status}`);
    }
    return proc;
}
//# sourceMappingURL=data:application/json;base64,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