"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
const cxapi = require("@aws-cdk/cx-api");
const tree_metadata_1 = require("./private/tree-metadata");
const stage_1 = require("./stage");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends stage_1.Stage {
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        var _a, _b;
        super(undefined, '', {
            outdir: (_a = props.outdir) !== null && _a !== void 0 ? _a : process.env[cxapi.OUTDIR_ENV],
        });
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        const analyticsReporting = (_b = props.analyticsReporting) !== null && _b !== void 0 ? _b : props.runtimeInfo;
        if (analyticsReporting !== undefined) {
            this.node.setContext(cxapi.ANALYTICS_REPORTING_ENABLED_CONTEXT, analyticsReporting);
        }
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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