# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytorch_forecasting',
 'pytorch_forecasting.data',
 'pytorch_forecasting.models',
 'pytorch_forecasting.models.basic_rnn',
 'pytorch_forecasting.models.deepar',
 'pytorch_forecasting.models.mlp',
 'pytorch_forecasting.models.nbeats',
 'pytorch_forecasting.models.nn',
 'pytorch_forecasting.models.rnn',
 'pytorch_forecasting.models.temporal_fusion_transformer']

package_data = \
{'': ['*']}

install_requires = \
['matplotlib',
 'optuna>=2.3.0,<3.0.0',
 'pandas>=1.1.0,<2.0.0',
 'pytorch-lightning>=1.2.4,<2.0.0',
 'scikit-learn>=0.23,<0.25',
 'scipy',
 'statsmodels',
 'torch>=1.7,<2.0']

extras_require = \
{'github-actions': ['pytest-github-actions-annotate-failures']}

setup_kwargs = {
    'name': 'pytorch-forecasting',
    'version': '0.9.0',
    'description': 'Forecasting timeseries with PyTorch - dataloaders, normalizers, metrics and models',
    'long_description': '![PyTorch Forecasting](./docs/source/_static/logo.svg)\n\n[pypi-image]: https://badge.fury.io/py/pytorch-forecasting.svg\n[pypi-url]: https://pypi.python.org/pypi/pytorch-forecasting\n[conda-image]: https://img.shields.io/conda/v/conda-forge/pytorch-forecasting\n[conda-url]: https://anaconda.org/conda-forge/pytorch-forecasting\n[build-image]: https://github.com/jdb78/pytorch-forecasting/actions/workflows/test.yml/badge.svg?branch=master\n[build-url]: https://github.com/jdb78/pytorch-forecasting/actions/workflows/test.yml?query=branch%3Amaster\n[linter-image]: https://github.com/jdb78/pytorch-forecasting/actions/workflows/code_quality.yml/badge.svg?branch=master\n[linter-url]: https://github.com/jdb78/pytorch-forecasting/actions/workflows/code_quality.yml?query=branch%3Amaster\n[docs-image]: https://readthedocs.org/projects/pytorch-forecasting/badge/?version=latest\n[docs-url]: https://pytorch-forecasting.readthedocs.io\n[coverage-image]: https://codecov.io/gh/jdb78/pytorch-forecasting/branch/master/graph/badge.svg\n[coverage-url]: https://codecov.io/github/jdb78/pytorch-forecasting?branch=master\n\n[![PyPI Version][pypi-image]][pypi-url] [![Conda Version][conda-image]][conda-url] [![Docs Status][docs-image]][docs-url] [![Linter Status][linter-image]][linter-url] [![Build Status][build-image]][build-url] [![Code Coverage][coverage-image]][coverage-url]\n\n**[Documentation](https://pytorch-forecasting.readthedocs.io)** | **[Tutorials](https://pytorch-forecasting.readthedocs.io/en/latest/tutorials.html)** | **[Release Notes](https://pytorch-forecasting.readthedocs.io/en/latest/CHANGELOG.html)**\n\n_Pytorch Forecasting_ is a PyTorch-based package for forecasting time series with state-of-the-art network architectures. It provides a high-level API for training networks on pandas data frames and leverages\n[PyTorch Lightning](https://pytorch-lightning.readthedocs.io/) for scalable training on (multiple) GPUs, CPUs and for automatic logging.\n\n---\n\nOur article on [Towards Data Science](https://towardsdatascience.com/introducing-pytorch-forecasting-64de99b9ef46) introduces the package and provides background information.\n\nPytorch Forecasting aims to ease state-of-the-art timeseries forecasting with neural networks for real-world cases and research alike. The goal is to provide a high-level API with maximum flexibility for professionals and reasonable defaults for beginners.\nSpecifically, the package provides\n\n- A timeseries dataset class which abstracts handling variable transformations, missing values,\n  randomized subsampling, multiple history lengths, etc.\n- A base model class which provides basic training of timeseries models along with logging in tensorboard\n  and generic visualizations such actual vs predictions and dependency plots\n- Multiple neural network architectures for timeseries forecasting that have been enhanced\n  for real-world deployment and come with in-built interpretation capabilities\n- Multi-horizon timeseries metrics\n- Ranger optimizer for faster model training\n- Hyperparameter tuning with [optuna](https://optuna.readthedocs.io/)\n\nThe package is built on [pytorch-lightning](https://pytorch-lightning.readthedocs.io/) to allow training on CPUs, single and multiple GPUs out-of-the-box.\n\n# Installation\n\nIf you are working on windows, you need to first install PyTorch with\n\n`pip install torch -f https://download.pytorch.org/whl/torch_stable.html`.\n\nOtherwise, you can proceed with\n\n`pip install pytorch-forecasting`\n\nAlternatively, you can install the package via conda\n\n`conda install pytorch-forecasting pytorch -c pytorch>=1.7 -c conda-forge`\n\nPyTorch Forecasting is now installed from the conda-forge channel while PyTorch is install from the pytorch channel.\n\n# Documentation\n\nVisit [https://pytorch-forecasting.readthedocs.io](https://pytorch-forecasting.readthedocs.io) to read the\ndocumentation with detailed tutorials.\n\n# Available models\n\nThe documentation provides a [comparison of available models](https://pytorch-forecasting.readthedocs.io/en/latest/models.html).\n\n- [Temporal Fusion Transformers for Interpretable Multi-horizon Time Series Forecasting](https://arxiv.org/pdf/1912.09363.pdf)\n  which outperforms DeepAR by Amazon by 36-69% in benchmarks\n- [N-BEATS: Neural basis expansion analysis for interpretable time series forecasting](http://arxiv.org/abs/1905.10437)\n  which has (if used as ensemble) outperformed all other methods including ensembles of traditional statical\n  methods in the M4 competition. The M4 competition is arguably the most important benchmark for univariate time series forecasting.\n- [DeepAR: Probabilistic forecasting with autoregressive recurrent networks](https://www.sciencedirect.com/science/article/pii/S0169207019301888)\n  which is the one of the most popular forecasting algorithms and is often used as a baseline\n- Simple standard networks for baselining: LSTM and GRU networks as well as a MLP on the decoder\n- A baseline model that always predicts the latest known value\n\nTo implement new models or other custom components, see the [How to implement new models tutorial](https://pytorch-forecasting.readthedocs.io/en/latest/tutorials/building.html). It covers basic as well as advanced architectures.\n\n# Usage example\n\nNetworks can be trained with the [PyTorch Lighning Trainer](https://pytorch-lightning.readthedocs.io/en/latest/common/trainer.html) on [pandas Dataframes](https://pandas.pydata.org/pandas-docs/stable/user_guide/dsintro.html#dataframe) which are first converted to a [TimeSeriesDataSet](https://pytorch-forecasting.readthedocs.io/en/latest/data.html).\n\n```python\n# imports for training\nimport pytorch_lightning as pl\nfrom pytorch_lightning.loggers import TensorBoardLogger\nfrom pytorch_lightning.callbacks import EarlyStopping, LearningRateMonitor\n# import dataset, network to train and metric to optimize\nfrom pytorch_forecasting import TimeSeriesDataSet, TemporalFusionTransformer, QuantileLoss\n\n# load data: this is pandas dataframe with at least a column for\n# * the target (what you want to predict)\n# * the timeseries ID (which should be a unique string to identify each timeseries)\n# * the time of the observation (which should be a monotonically increasing integer)\ndata = ...\n\n# define the dataset, i.e. add metadata to pandas dataframe for the model to understand it\nmax_encoder_length = 36\nmax_prediction_length = 6\ntraining_cutoff = "YYYY-MM-DD"  # day for cutoff\n\ntraining = TimeSeriesDataSet(\n    data[lambda x: x.date <= training_cutoff],\n    time_idx= ...,  # column name of time of observation\n    target= ...,  # column name of target to predict\n    group_ids=[ ... ],  # column name(s) for timeseries IDs\n    max_encoder_length=max_encoder_length,  # how much history to use\n    max_prediction_length=max_prediction_length,  # how far to predict into future\n    # covariates static for a timeseries ID\n    static_categoricals=[ ... ],\n    static_reals=[ ... ],\n    # covariates known and unknown in the future to inform prediction\n    time_varying_known_categoricals=[ ... ],\n    time_varying_known_reals=[ ... ],\n    time_varying_unknown_categoricals=[ ... ],\n    time_varying_unknown_reals=[ ... ],\n)\n\n# create validation dataset using the same normalization techniques as for the training dataset\nvalidation = TimeSeriesDataSet.from_dataset(training, data, min_prediction_idx=training.index.time.max() + 1, stop_randomization=True)\n\n# convert datasets to dataloaders for training\nbatch_size = 128\ntrain_dataloader = training.to_dataloader(train=True, batch_size=batch_size, num_workers=2)\nval_dataloader = validation.to_dataloader(train=False, batch_size=batch_size, num_workers=2)\n\n# create PyTorch Lighning Trainer with early stopping\nearly_stop_callback = EarlyStopping(monitor="val_loss", min_delta=1e-4, patience=1, verbose=False, mode="min")\nlr_logger = LearningRateMonitor()\ntrainer = pl.Trainer(\n    max_epochs=100,\n    gpus=0,  # run on CPU, if on multiple GPUs, use accelerator="ddp"\n    gradient_clip_val=0.1,\n    limit_train_batches=30,  # 30 batches per epoch\n    callbacks=[lr_logger, early_stop_callback],\n    logger=TensorBoardLogger("lightning_logs")\n)\n\n# define network to train - the architecture is mostly inferred from the dataset, so that only a few hyperparameters have to be set by the user\ntft = TemporalFusionTransformer.from_dataset(\n    # dataset\n    training,\n    # architecture hyperparameters\n    hidden_size=32,\n    attention_head_size=1,\n    dropout=0.1,\n    hidden_continuous_size=16,\n    # loss metric to optimize\n    loss=QuantileLoss(),\n    # logging frequency\n    log_interval=2,\n    # optimizer parameters\n    learning_rate=0.03,\n    reduce_on_plateau_patience=4\n)\nprint(f"Number of parameters in network: {tft.size()/1e3:.1f}k")\n\n# find the optimal learning rate\nres = trainer.lr_find(\n    tft, train_dataloader=train_dataloader, val_dataloaders=val_dataloader, early_stop_threshold=1000.0, max_lr=0.3,\n)\n# and plot the result - always visually confirm that the suggested learning rate makes sense\nprint(f"suggested learning rate: {res.suggestion()}")\nfig = res.plot(show=True, suggest=True)\nfig.show()\n\n# fit the model on the data - redefine the model with the correct learning rate if necessary\ntrainer.fit(\n    tft, train_dataloader=train_dataloader, val_dataloaders=val_dataloader,\n)\n```\n',
    'author': 'Jan Beitner',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://pytorch-forecasting.readthedocs.io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
