/**
 * Determines how symlinks are followed.
 *
 * @stability stable
 */
export declare enum SymlinkFollowMode {
    /**
     * Never follow symlinks.
     *
     * @stability stable
     */
    NEVER = "never",
    /**
     * Materialize all symlinks, whether they are internal or external to the source directory.
     *
     * @stability stable
     */
    ALWAYS = "always",
    /**
     * Only follows symlinks that are external to the source directory.
     *
     * @stability stable
     */
    EXTERNAL = "external",
    /**
     * Forbids source from having any symlinks pointing outside of the source tree.
     *
     * This is the safest mode of operation as it ensures that copy operations
     * won't materialize files from the user's file system. Internal symlinks are
     * not followed.
     *
     * If the copy operation runs into an external symlink, it will fail.
     *
     * @stability stable
     */
    BLOCK_EXTERNAL = "internal-only"
}
/**
 * Determines the ignore behavior to use.
 *
 * @stability stable
 */
export declare enum IgnoreMode {
    /**
     * Ignores file paths based on simple glob patterns.
     *
     * This is the default for file assets.
     *
     * It is also the default for Docker image assets, unless the '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport'
     * context flag is set.
     *
     * @stability stable
     */
    GLOB = "glob",
    /**
     * Ignores file paths based on the [`.gitignore specification`](https://git-scm.com/docs/gitignore).
     *
     * @stability stable
     */
    GIT = "git",
    /**
     * Ignores file paths based on the [`.dockerignore specification`](https://docs.docker.com/engine/reference/builder/#dockerignore-file).
     *
     * This is the default for Docker image assets if the '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport'
     * context flag is set.
     *
     * @stability stable
     */
    DOCKER = "docker"
}
/**
 * Obtains applied when copying directories into the staging location.
 *
 * @stability stable
 */
export interface CopyOptions {
    /**
     * A strategy for how to handle symlinks.
     *
     * @default SymlinkFollowMode.NEVER
     * @stability stable
     */
    readonly follow?: SymlinkFollowMode;
    /**
     * Glob patterns to exclude from the copy.
     *
     * @default - nothing is excluded
     * @stability stable
     */
    readonly exclude?: string[];
    /**
     * The ignore behavior to use for exclude patterns.
     *
     * @default IgnoreMode.GLOB
     * @stability stable
     */
    readonly ignoreMode?: IgnoreMode;
}
/**
 * Options related to calculating source hash.
 *
 * @stability stable
 */
export interface FingerprintOptions extends CopyOptions {
    /**
     * Extra information to encode into the fingerprint (e.g. build instructions and other inputs).
     *
     * @default - hash is only based on source content
     * @stability stable
     */
    readonly extraHash?: string;
}
