"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenMap = void 0;
const token_1 = require("../token");
const encoding_1 = require("./encoding");
const glob = global;
const STRING_SYMBOL = Symbol.for('@aws-cdk/core.TokenMap.STRING');
const LIST_SYMBOL = Symbol.for('@aws-cdk/core.TokenMap.LIST');
const NUMBER_SYMBOL = Symbol.for('@aws-cdk/core.TokenMap.NUMBER');
/**
 * Central place where we keep a mapping from Tokens to their String representation
 *
 * The string representation is used to embed token into strings,
 * and stored to be able to reverse that mapping.
 *
 * All instances of TokenStringMap share the same storage, so that this process
 * works even when different copies of the library are loaded.
 */
class TokenMap {
    constructor() {
        this.stringTokenMap = new Map();
        this.numberTokenMap = new Map();
        /**
         * Counter to assign unique IDs to tokens
         *
         * Start at a random number to prevent people from accidentally taking
         * dependencies on token values between runs.
         *
         * This is most prominent in tests, where people will write:
         *
         * ```ts
         * sha256(JSON.stringify({ ...some structure that can contain tokens ... }))
         * ```
         *
         * This should have been:
         *
         * ```ts
         * sha256(JSON.stringify(stack.resolve({ ...some structure that can contain tokens ... })))
         * ```
         *
         * The hash value is hard to inspect for correctness. It will LOOK consistent
         * during testing, but will break as soon as someone stringifies another
         * token before the run.
         *
         * By changing the starting number for tokens, we ensure that the hash is almost
         * guaranteed to be different during a few test runs, so the hashing of unresolved
         * tokens can be detected.
         */
        this.tokenCounter = Math.floor(Math.random() * 10);
    }
    /**
     * Singleton instance of the token string map
     */
    static instance() {
        if (!glob.__cdkTokenMap) {
            glob.__cdkTokenMap = new TokenMap();
        }
        return glob.__cdkTokenMap;
    }
    /**
     * Generate a unique string for this Token, returning a key
     *
     * Every call for the same Token will produce a new unique string, no
     * attempt is made to deduplicate. Token objects should cache the
     * value themselves, if required.
     *
     * The token can choose (part of) its own representation string with a
     * hint. This may be used to produce aesthetically pleasing and
     * recognizable token representations for humans.
     */
    registerString(token, displayHint) {
        return cachedValue(token, STRING_SYMBOL, () => {
            const key = this.registerStringKey(token, displayHint);
            return `${encoding_1.BEGIN_STRING_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`;
        });
    }
    /**
     * Generate a unique string for this Token, returning a key
     */
    registerList(token, displayHint) {
        return cachedValue(token, LIST_SYMBOL, () => {
            const key = this.registerStringKey(token, displayHint);
            return [`${encoding_1.BEGIN_LIST_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`];
        });
    }
    /**
     * Create a unique number representation for this Token and return it
     */
    registerNumber(token) {
        return cachedValue(token, NUMBER_SYMBOL, () => {
            return this.registerNumberKey(token);
        });
    }
    /**
     * Lookup a token from an encoded value
     */
    tokenFromEncoding(x) {
        if (typeof x === 'string') {
            return this.lookupString(x);
        }
        if (Array.isArray(x)) {
            return this.lookupList(x);
        }
        if (token_1.Token.isUnresolved(x)) {
            return x;
        }
        return undefined;
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupString(s) {
        const fragments = this.splitString(s);
        if (fragments.tokens.length > 0 && fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupList(xs) {
        if (xs.length !== 1) {
            return undefined;
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const fragments = str.split(this.lookupToken.bind(this));
        if (fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Split a string into literals and Tokens
     */
    splitString(s) {
        const str = encoding_1.TokenString.forString(s);
        return str.split(this.lookupToken.bind(this));
    }
    /**
     * Reverse a number encoding into a Token, or undefined if the number wasn't a Token
     */
    lookupNumberToken(x) {
        const tokenIndex = encoding_1.extractTokenDouble(x);
        if (tokenIndex === undefined) {
            return undefined;
        }
        const t = this.numberTokenMap.get(tokenIndex);
        if (t === undefined) {
            throw new Error('Encoded representation of unknown number Token found');
        }
        return t;
    }
    /**
     * Find a Token by key.
     *
     * This excludes the token markers.
     */
    lookupToken(key) {
        const token = this.stringTokenMap.get(key);
        if (!token) {
            throw new Error(`Unrecognized token key: ${key}`);
        }
        return token;
    }
    registerStringKey(token, displayHint) {
        const counter = this.tokenCounter++;
        const representation = (displayHint || 'TOKEN').replace(new RegExp(`[^${encoding_1.VALID_KEY_CHARS}]`, 'g'), '.');
        const key = `${representation}.${counter}`;
        this.stringTokenMap.set(key, token);
        return key;
    }
    registerNumberKey(token) {
        const counter = this.tokenCounter++;
        this.numberTokenMap.set(counter, token);
        return encoding_1.createTokenDouble(counter);
    }
}
exports.TokenMap = TokenMap;
/**
 * Get a cached value for an object, storing it on the object in a symbol
 */
function cachedValue(x, sym, prod) {
    let cached = x[sym];
    if (cached === undefined) {
        cached = prod();
        Object.defineProperty(x, sym, { value: cached });
    }
    return cached;
}
//# sourceMappingURL=data:application/json;base64,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