"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const ALLOWED_SCOPES = ['@aws-cdk', '@aws-cdk-containers', '@aws-solutions-konstruk', '@aws-solutions-constructs', '@amzn'];
// list of NPM packages included in version reporting
const ALLOWED_PACKAGES = ['aws-rfdk', 'aws-cdk-lib', 'monocdk'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the allowlistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of ALLOWED_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        foundMatch = foundMatch || ALLOWED_PACKAGES.includes(name);
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!(mod === null || mod === void 0 ? void 0 : mod.paths)) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod === null || mod === void 0 ? void 0 : mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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