# FastAPI Extras

This library is a collection of utilities for running FastAPI applications at Explosion AI.


## HttpizeErrorsAPIRouter
This custom router's main functionality is to handle errors per route instead of through a global exception handler by adding the `httpize_errors` keyword argument to the FastAPI route declaration. This allows each route to return normal informative Python errors instead of the FastAPI `HTTPException` class to get valid responses.

It also times each request and sets the `X-Response-Time` header on the Response

**An example route**

```python
@router.get("/testing", httpize_errors={ValueError: 400})
def test_route(i: int):
    if i < 1:
        raise ValueError("Bad Input Data")
    return {"i": i}
```

If the ValueError is raised, this custom router knows to return a Response with a status code of 400 (Bad Request) and the message provided to the `ValueError` 


**Usage**

FastAPI doesn't have built-in support for overriding the app Router, however this is required since we add a new keyword argument to the route declaration. FastAPI doesn't pass `**kwargs` forward, it only passes explict named keyword arguments. 

To get around this, we need to overwrite the app router manually and refresh the routes after all of them have been included in the main app. This looks like:

```python
from fastapi import FastAPI
from fastapi_extras import HttpizeErrorsAPIRouter, init_app
import uvicorn


# API Router (could be in another module)
api_router = HttpizeErrorsAPIRouter(tags=["tests"])


@api_router.get("/testing", httpize_errors={ValueError: 400})
def test_route(i: int):
    if i < 1:
        raise ValueError("Bad Input Data")
    return {"i": i}


# Main app definition
app = FastAPI()

# Overwrite App Router to use the custom HttpizeErrorsAPIRouter
app.router = HttpizeErrorsAPIRouter.from_app(app)

# Include API Router from above
app.include_router(api_router)

# Refresh the App (this rebuilds the Starlette Middleware Stack)
init_app(app)

uvicorn.run(app)
```