"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
function deploy(stack, props = {}) {
    return new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, 'test-stream-firehose-s3', props);
}
test('test kinesisFirehose override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
test('test kinesisFirehose.deliveryStreamType override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisFirehoseProps: {
            deliveryStreamType: 'DirectPut'
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        DeliveryStreamType: 'KinesisStreamAsSource'
    });
});
test('test kinesisFirehose.kinesisStreamSourceConfiguration override ', () => {
    const stack = new cdk.Stack();
    const kinesisStream = defaults.buildKinesisStream(stack, {
        existingStreamObj: undefined,
        kinesisStreamProps: undefined
    });
    deploy(stack, {
        kinesisFirehoseProps: {
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: kinesisStream.streamArn,
                roleArn: new aws_iam_1.Role(stack, 'myRole', {
                    assumedBy: new aws_iam_1.ServicePrincipal('firehose.amazonaws.com')
                }).roleArn
            }
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        KinesisStreamSourceConfiguration: {
            KinesisStreamARN: {
                "Fn::GetAtt": [
                    "teststreamfirehoses3KinesisStream3165E68E",
                    "Arn"
                ]
            },
            RoleARN: {
                "Fn::GetAtt": [
                    "KinesisStreamsRole2BFD39A5",
                    "Arn"
                ]
            }
        }
    });
});
test('test kinesisStreamProps override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisStreamProps: {
            shardCount: 3
        }
    });
    expect(stack).toHaveResourceLike("AWS::Kinesis::Stream", {
        ShardCount: 3
    });
});
test('Test All properties', () => {
    const stack = new cdk.Stack();
    const construct = deploy(stack);
    expect(construct.cloudwatchAlarms).not.toEqual(undefined);
    expect(construct.kinesisFirehose).not.toEqual(undefined);
    expect(construct.kinesisFirehoseRole).not.toEqual(undefined);
    expect(construct.kinesisFirehoseLogGroup).not.toEqual(undefined);
    expect(construct.kinesisStream).not.toEqual(undefined);
    expect(construct.kinesisStreamRole).not.toEqual(undefined);
    expect(construct.s3Bucket).not.toEqual(undefined);
    expect(construct.s3LoggingBucket).not.toEqual(undefined);
});
test('Test properties with no CW Alarms', () => {
    const stack = new cdk.Stack();
    const construct = deploy(stack, {
        createCloudWatchAlarms: false
    });
    expect(construct.cloudwatchAlarms).toEqual(undefined);
});
test('Test properties with existing S3 bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = defaults.CreateScrapBucket(stack, {});
    const mybucket = s3.Bucket.fromBucketName(stack, 'mybucket', existingBucket.bucketName);
    const construct = deploy(stack, {
        existingBucketObj: mybucket
    });
    expect(construct.s3Bucket).toEqual(undefined);
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
test('Test properties with existing logging S3 bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = defaults.CreateScrapBucket(stack, {});
    const myLoggingBucket = s3.Bucket.fromBucketName(stack, 'myLoggingBucket', existingBucket.bucketName);
    const construct = deploy(stack, {
        existingLoggingBucketObj: myLoggingBucket
    });
    expect(construct.s3Bucket).not.toEqual(undefined);
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new cdk.Stack();
    new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, 'kinsisfirehose-s3', {
        kinesisFirehoseProps: {
            deliveryStreamType: 'KinesisStreamAsSource'
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        },
        loggingBucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "kinsisfirehoses3KinesisFirehoseToS3S3LoggingBucket1CC9C6B7"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with one content bucket and no logging bucket', () => {
    const stack = new cdk.Stack();
    new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, 'kinsisfirehose-s3', {
        kinesisFirehoseProps: {
            deliveryStreamType: 'KinesisStreamAsSource'
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    expect(stack).toCountResources("AWS::S3::Bucket", 1);
});
//# sourceMappingURL=data:application/json;base64,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