"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleMetadata = exports.PackName = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const cdk_nag_1 = require("cdk-nag");
/**
 * Pack name
 *
 */
exports.PackName = "AwsPrototyping";
/**
 * Rule metadata
 *
 */
exports.RuleMetadata = [
    {
        info: "The REST API stage is not associated with AWS WAFv2 web ACL.",
        explanation: "AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAssociatedWithWAF,
    },
    {
        info: "The API does not implement authorization.",
        explanation: "In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.\n\nExample threat: An actor with a network path to an API gateway stage end-point can interact with the API method in question without authorization, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAuthorization,
    },
    {
        info: "The REST API does not have request validation enabled.",
        explanation: "The API should have basic request validation enabled. If the API is integrated with a custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWRequestValidation,
    },
    {
        info: "The Athena workgroup does not encrypt query results.",
        explanation: "Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.athena.AthenaWorkgroupEncryptedQueryResults,
    },
    {
        info: "The Auto Scaling launch configuration does not have public IP addresses disabled.",
        explanation: "If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.autoscaling.AutoScalingLaunchConfigPublicIpDisabled,
    },
    {
        info: "The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.",
        explanation: "SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.\n\nExample threat: A global internet-based actor can discover Cloud9 EC2 instances that have public IP addresses and that are exposing SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloud9.Cloud9InstanceNoIngressSystemsManager,
    },
    {
        info: "The prototypes CloudFront distribution has not been configured with geographic restrictions (GeoRestriction)",
        explanation: "Geo restriction should be enabled for the distribution in order to limit the surface area exposed to expected geographies\n\nExample threat: A global internet-based actor can discover prototype web assets that are exposed via CloudFront distributions, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionGeoRestrictions,
    },
    {
        info: "The CloudFront distribution allows for SSLv3 or TLSv1 for HTTPS viewer connections.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Help protect viewer connections by specifying a viewer certificate that enforces a minimum of TLSv1.1 or TLSv1.2 in the security policy. Distributions that use that use the default CloudFront viewer certificate or use 'vip' for the SslSupportMethod are non-compliant with this rule, as the minimum security policy is set to TLSv1 regardless of the specified MinimumProtocolVersion",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionHttpsViewerNoOutdatedSSL,
    },
    {
        info: "The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionNoOutdatedSSL,
    },
    {
        info: "The CloudFront distribution does not use an origin access identity an S3 origin.",
        explanation: "Origin access identities help with security by restricting any direct access to objects through S3 URLs.\n\nExample threat: A global internet-based actor can bypass the CloudFront distribution and associated controls (e.g. geo blocking), which may lead to direct access to static assets hosted on the S3 origin possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 origin for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionS3OriginAccessIdentity,
    },
    {
        info: "The CloudFront distribution may require integration with AWS WAF.",
        explanation: "The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionWAFIntegration,
    },
    {
        info: "The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables.",
        explanation: "Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.\n\nExample threat: An actor who can view the CodeBuild environment variables can obtain the AWS Access Key and Secret Access Key, which may lead to the actor being able to do anything the AWS keys are authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectEnvVarAwsCred,
    },
    {
        info: "The CodeBuild project does not use an AWS KMS key for encryption.",
        explanation: "Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectKMSEncryptedArtifacts,
    },
    {
        info: "The CodeBuild project does not use images provided by the CodeBuild service or have a cdk-nag suppression rule explaining the need for a custom image.",
        explanation: "Explaining differences/edits to Docker images helps operators better understand system dependencies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectManagedImages,
    },
    {
        info: "The CodeBuild project has privileged mode enabled.",
        explanation: "Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectPrivilegedModeDisabled,
    },
    {
        info: "The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.",
        explanation: "Advanced security features enable the system to detect and act upon malicious sign-in attempts.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAdvancedSecurityModeEnforced,
    },
    {
        info: "The API Gateway method does not use a Cognito user pool authorizer.",
        explanation: "API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAPIGWAuthorizer,
    },
    {
        info: "The Cognito user pool is not configured to require MFA.",
        explanation: "Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolMFA,
    },
    {
        info: "The Cognito identity pool allows for unauthenticated logins and does not have a cdk-nag rule suppression with a reason.",
        explanation: "Applications do not warrant unauthenticated guest access in many cases. Metadata explaining the use case allows for transparency to operators.\n\nExample threat: A global internet-based actor who has discovered a prototype endpoint with Cognito unauthenticated logins can does not need to provide credentials to interact with the endpoint, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolNoUnauthenticatedLogins,
    },
    {
        info: "The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.",
        explanation: "Strong password policies increase system security by encouraging users to create reliable and secure passwords.\n\nExample threat: An actor who has discovered a prototype endpoint with Cognito authenticated logins can perform a dictionary or brute force attack to authenticate as an authorized Cognito user, which may lead to the actor being able to do anything the associated Cognito user is authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolStrongPasswordPolicy,
    },
    {
        info: "The DMS replication instance is public.",
        explanation: "DMS replication instances can contain sensitive information and access control is required for such accounts.\n\nExample threat: A global internet-based actor can discover DMS instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.dms.DMSReplicationNotPublic,
    },
    {
        info: "The Document DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterEncryptionAtRest,
    },
    {
        info: "The Document DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterNonDefaultPort,
    },
    {
        info: "The Document DB cluster does not have the username and password stored in Secrets Manager.",
        explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.\n\nExample threat: An actor who can view the DocumentDB configuration can obtain the username and password for the DocumentDB cluster, which may lead to the actor being able to access anything the associated DocumentDB user is authorised to do possibly impacting the confidentiality, integrity and availability of the data assets hosted on the DocumentDB cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.documentdb.DocumentDBCredentialsInSecretsManager,
    },
    {
        info: "The DAX cluster does not have server-side encryption enabled.",
        explanation: "Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.dynamodb.DAXEncrypted,
    },
    {
        info: "The EBS volume has encryption disabled.",
        explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2EBSVolumeEncrypted,
    },
    {
        info: "The EC2 instance is associated with a public IP address.",
        explanation: "Amazon EC2 instances can contain sensitive information and access control is required for such resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceNoPublicIp,
    },
    {
        info: "The EC2 instance does not have an instance profile attached.",
        explanation: "EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceProfileAttached,
    },
    {
        info: "The EC2 instance is not within a VPC.",
        explanation: "Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2InstancesInVPC,
    },
    {
        info: "The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).",
        explanation: "Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.\n\nExample threat: A global internet-based actor can discover exposed services (e.g. Telnet, SSH, RDS, MySQL) using their common port numbers, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedCommonPorts,
    },
    {
        info: "The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.",
        explanation: "Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public EC2 instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedInbound,
    },
    {
        info: "The Security Group allows unrestricted SSH access.",
        explanation: "Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address to SSH or move laterally to non-public EC2 instances via SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedSSH,
    },
    {
        info: "The Security Group does not have a description.",
        explanation: "Descriptions help simplify operations and remove any opportunities for operator errors.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2SecurityGroupDescription,
    },
    {
        info: "The ECR Repository allows open access.",
        explanation: "Removing * principals in an ECR Repository helps protect against unauthorized access.\n\nExample threat: A global internet-based actor who has discovered a ECR repository can access the container images hosted within the repository, which may lead to information disclosure that aids in the intrusion activities being successful against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ecr.ECROpenAccess,
    },
    {
        info: "One or more containers in the ECS Task Definition do not have container logging enabled.",
        explanation: "Container logging allows operators to view and aggregate the logs from the container. Containers should use the 'awslogs' driver at a minimum.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionContainerLogging,
    },
    {
        info: "The ECS Task Definition includes a container definition that directly specifies environment variables.",
        explanation: "Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionNoEnvironmentVariables,
    },
    {
        info: "The EFS does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.efs.EFSEncrypted,
    },
    {
        info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
        explanation: "EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format LogExport::<log> for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.eks.EKSClusterControlPlaneLogs,
    },
    {
        info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
        explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).\n\nExample threat: A global internet-based actor who has discovered a EKS cluster Kubernetes API server endpoint can perform reconnaissance and intrusion activities against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eks.EKSClusterNoEndpointPublicAccess,
    },
    {
        info: "The ElastiCache cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover the ElastiCache cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterInVPC,
    },
    {
        info: "The ElastiCache cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterNonDefaultPort,
    },
    {
        info: "The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.",
        explanation: "Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterEncryption,
    },
    {
        info: "The ElastiCache Redis cluster does not use Redis AUTH for user authentication.",
        explanation: "Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterRedisAuth,
    },
    {
        info: "The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.",
        explanation: "Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkEC2InstanceLogsToS3,
    },
    {
        info: "The Elastic Beanstalk environment does not have managed updates enabled.",
        explanation: "Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.\n\nExample threat: An actor with a network path to the Elastic Beanstalk environment can attempt to take advantage of a known vulnerability in a platform component used by Elastic Beanstalk, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkManagedUpdatesEnabled,
    },
    {
        info: "The Elastic Beanstalk environment is not configured to use a specific VPC.",
        explanation: "Use a non-default VPC in order to separate your environment from default resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkVPCSpecified,
    },
    {
        info: "The ALB's HTTP listeners are not configured to redirect to HTTPS.",
        explanation: "To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBHttpToHttpsRedirection,
    },
    {
        info: "The CLB does not restrict its listeners to only the SSL and HTTPS protocols.",
        explanation: "Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBTlsHttpsListenersOnly,
    },
    {
        info: "The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.",
        explanation: "SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRAuthEC2KeyPairOrKerberos,
    },
    {
        info: "The EMR cluster does not use a security configuration with encryption in transit enabled and configured.",
        explanation: "EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMREncryptionInTransit,
    },
    {
        info: "The EMR cluster does not use a security configuration with local disk encryption enabled.",
        explanation: "Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRLocalDiskEncryption,
    },
    {
        info: "The event bus policy allows for open access.",
        explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.\n\nExample threat: A global internet-based actor who has discovered the Event Bridge event bus (e.g. Endpoint ID) can put arbitrary events onto the bus, which may lead to which could be processed by the prototype possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype',
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eventbridge.EventBusOpenAccess,
    },
    {
        info: "The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.",
        explanation: "Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueEncryptedCloudWatchLogs,
    },
    {
        info: "The Glue job does not have use a security configuration with job bookmark encryption enabled.",
        explanation: "Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueJobBookmarkEncrypted,
    },
    {
        info: "The IAM user, role, or group uses AWS managed policies.",
        explanation: "An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Policy::<policy> for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoManagedPolicies,
    },
    {
        info: "The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.",
        explanation: "Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Action::<action> for policy actions and Resource::<resource> for resources. Example: appliesTo: ['Action::s3:*'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoWildcardPermissions,
    },
    {
        info: "The IAM policy grants admin access - meaning the policy allows a principal to perform unlimited actions on any service",
        explanation: "AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.\n\nExample threat: A global internet-based actor who has successfully obtained valid keys or a session associated of the IAM Principal associated with the IAM policy can perform unlimited AWS actions on any AWS service which are exposed via the AWS API/Management Console/CLI, which may lead to broad and unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithAdminAccess,
    },
    {
        info: "The IAM policy grants full access - meaning the policy allows unlimited actions for a given AWS service",
        explanation: "Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithFullAccess,
    },
    {
        info: "The Kinesis Data Firehose delivery stream does not have server-side encryption enabled.",
        explanation: "Enabling encryption allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataFirehoseSSE,
    },
    {
        info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
        explanation: "Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamDefaultKeyWhenSSE,
    },
    {
        info: "The Kinesis Data Stream does not have server-side encryption enabled.",
        explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamSSE,
    },
    {
        info: "The KMS Symmetric key does not have automatic key rotation enabled.",
        explanation: "KMS key rotation allow a system to set a rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kms.KMSBackingKeyRotationEnabled,
    },
    {
        info: "The Lambda function permission grants public access",
        explanation: "Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.\n\nExample threat: A global internet-based actor who has discovered the Lambda function name or ARN can invoke/delete/modify the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionPublicAccessProhibited,
    },
    {
        info: "The Lambda Function URL allows for public, unauthenticated access.",
        explanation: "AWS Lambda Function URLs allow you to invoke your function via a HTTPS end-point, setting the authentication to NONE allows anyone on the internet to invoke your function.\n\nExample threat: A global internet-based actor who has discovered the Lambda Function URL can invoke the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionUrlAuth,
    },
    {
        info: "The non-container Lambda function is not configured to use the latest runtime version.",
        explanation: "Use the latest available runtime for the targeted language to avoid technical debt. Runtimes specific to a language or framework version are deprecated when the version reaches end of life. This rule only applies to non-container Lambda functions.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.lambda.LambdaLatestVersion,
    },
    {
        info: "The MediaStore container does not define a CORS policy.",
        explanation: "Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerCORSPolicy,
    },
    {
        info: "The MediaStore container does not define a container policy.",
        explanation: "Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerHasContainerPolicy,
    },
    {
        info: "The MediaStore container does not require requests to use SSL.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerSSLRequestsOnly,
    },
    {
        info: "The MSK cluster uses plaintext communication between brokers.",
        explanation: "TLS communication secures data-in-transit by encrypting the connection between brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKBrokerToBrokerTLS,
    },
    {
        info: "The MSK cluster uses plaintext communication between clients and brokers.",
        explanation: "TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKClientToBrokerTLS,
    },
    {
        info: "The Neptune DB instance does have Auto Minor Version Upgrade enabled.",
        explanation: "The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.\n\nExample threat: An actor with a network path to the Neptune cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by Neptune, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterAutomaticMinorVersionUpgrade,
    },
    {
        info: "The Neptune DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterEncryptionAtRest,
    },
    {
        info: "The Neptune DB cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterIAMAuth,
    },
    {
        info: "The OpenSearch Service domain does not only grant access via allowlisted IP addresses.",
        explanation: "Using allowlisted IP addresses helps protect the domain against unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchAllowlistedIPs,
    },
    {
        info: "The OpenSearch Service domain does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchEncryptedAtRest,
    },
    {
        info: "The OpenSearch Service domain is not provisioned inside a VPC.",
        explanation: "Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.\n\nExample threat: A global internet-based actor can discover the OpenSearch Service domain that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchInVPCOnly,
    },
    {
        info: "The OpenSearch Service domain does not have node-to-node encryption enabled.",
        explanation: "Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNodeToNodeEncryption,
    },
    {
        info: "The OpenSearch Service domain does not allow for unsigned requests or anonymous access.",
        explanation: "Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.\n\nExample threat: An actor with a network path to the OpenSearch Service domain can directly access the domain without authentication, which may lead to allowing access to data hosted within the domain possibly impacting the confidentiality, integrity and availability of the data assets hosted on the OpenSearch Service domain for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNoUnsignedOrAnonymousAccess,
    },
    {
        info: "The Quicksight data sources connection is not configured to use SSL.",
        explanation: "SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.quicksight.QuicksightSSLConnections,
    },
    {
        info: "The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.AuroraMySQLPostgresIAMAuth,
    },
    {
        info: "The RDS DB instance does not have automatic minor version upgrades enabled.",
        explanation: "Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.\n\nExample threat: An actor with a network path to the RDS cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by RDS, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSAutomaticMinorVersionUpgradeEnabled,
    },
    {
        info: "The RDS DB instance allows public access.",
        explanation: "Amazon RDS database instances can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the RDS DB instance endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSInstancePublicAccess,
    },
    {
        info: "The RDS instance or Aurora DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSNonDefaultPort,
    },
    {
        info: "The RDS DB Security Group allows for 0.0.0.0/0 inbound access.",
        explanation: "RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.\n\nExample threat: A global internet-based actor can discover RDS DB instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public RDS DB instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSRestrictedInbound,
    },
    {
        info: "The RDS DB instance or Aurora DB cluster does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon RDS DB instances, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSStorageEncrypted,
    },
    {
        info: "The Redshift cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterEncryptionAtRest,
    },
    {
        info: "The Redshift cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover a RedShift cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterInVPC,
    },
    {
        info: "The Redshift cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultPort,
    },
    {
        info: 'The Redshift cluster uses the default "awsuser" username.',
        explanation: 'Using a custom user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultUsername,
    },
    {
        info: "The Redshift cluster allows public access.",
        explanation: "Amazon Redshift clusters can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the Redshift cluster endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterPublicAccess,
    },
    {
        info: "The Redshift cluster does not have version upgrade enabled.",
        explanation: "Version Upgrade must enabled on the cluster in order to automatically receive upgrades during the maintenance window.\n\nExample threat: An actor with a network path to the Redshift cluster can attempt to take advantage of a known vulnerability in a component exposed by Redshift, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterVersionUpgrade,
    },
    {
        info: "The Redshift cluster does not require TLS/SSL encryption.",
        explanation: "Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftRequireTlsSSL,
    },
    {
        info: "The S3 bucket does not prohibit public access through bucket level settings.",
        explanation: "Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read or write can read or write data to or from the S3 bucket, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketLevelPublicAccessProhibited,
    },
    {
        info: "The S3 Bucket does not have server access logs enabled.",
        explanation: "Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketLoggingEnabled,
    },
    {
        info: "The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read can read data from the S3 bucket, which may lead to possibly impacting the confidentiality of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicReadProhibited,
    },
    {
        info: "The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public write can write data to, or overwrite data within the S3 bucket, which may lead to possibly impacting the integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicWriteProhibited,
    },
    {
        info: "The S3 Bucket does not have default server-side encryption enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketServerSideEncryptionEnabled,
    },
    {
        info: "The S3 Bucket or bucket policy does not require requests to use SSL/TLS.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketSSLRequestsOnly,
    },
    {
        info: "The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.",
        explanation: "An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.\n\nExample threat: A global internet-based actor who has discovered a S3 hosted website can discover prototype web assets that are hosted on the website, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3WebBucketOAIAccess,
    },
    {
        info: "The SageMaker notebook instance does not have an encrypted storage volume.",
        explanation: "Encrypting storage volumes helps protect SageMaker data-at-rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInstanceKMSKeyConfigured,
    },
    {
        info: "The SageMaker notebook instance is not provisioned inside a VPC.",
        explanation: "Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInVPC,
    },
    {
        info: "The SageMaker notebook does not disable direct internet access.",
        explanation: "By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookNoDirectInternetAccess,
    },
    {
        info: "The secret does not have automatic rotation scheduled.",
        explanation: "Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.secretsmanager.SecretsManagerRotationEnabled,
    },
    {
        info: "The SNS topic does not have KMS encryption enabled.",
        explanation: "To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS). Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSEncryptedKMS,
    },
    {
        info: "The SNS Topic does not require publishers to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns:Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSTopicSSLPublishOnly,
    },
    {
        info: "The SQS Queue does not have server-side encryption enabled.",
        explanation: "Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSE,
    },
    {
        info: "The SQS queue does not require requests to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSLRequestsOnly,
    },
    {
        info: "The Timestream database does not use a Customer Managed KMS Key for at rest encryption.",
        explanation: "All Timestream tables in a database are encrypted at rest by default using an AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an periodic basis.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.timestream.TimestreamDatabaseCustomerManagedKey,
    },
    {
        info: "The VPC's default security group allows inbound or outbound traffic.",
        explanation: "When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) that have public IP addresses and allow ingress to all internet address or move laterally to non-public VPC-attached instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCDefaultSecurityGroupClosed,
    },
    {
        info: "A Network ACL or Network ACL entry has been implemented.",
        explanation: "Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.vpc.VPCNoNACLs,
    },
    {
        info: "The subnet auto-assigns public IP addresses.",
        explanation: "Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) within the subnet in question that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCSubnetAutoAssignPublicIpDisabled,
    },
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzLXByb3RvdHlwaW5nLXJ1bGVzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL3BhY2tzL2F3cy1wcm90b3R5cGluZy1ydWxlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQTtzQ0FDc0M7QUFDdEMscUNBQWlEO0FBRWpEOzs7R0FHRztBQUNRLFFBQUEsUUFBUSxHQUFHLGdCQUFnQixDQUFDO0FBRXZDOzs7R0FHRztBQUNRLFFBQUEsWUFBWSxHQUFHO0lBQ3hCO1FBQ0UsSUFBSSxFQUFFLDhEQUE4RDtRQUNwRSxXQUFXLEVBQ1QsOE9BQThPO1FBQ2hQLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxLQUFLLENBQUMsc0JBQXNCO0tBQ3pDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMkNBQTJDO1FBQ2pELFdBQVcsRUFDVCw2akJBQTZqQjtRQUMvakIsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEtBQUssQ0FBQyxrQkFBa0I7S0FDckM7SUFDRDtRQUNFLElBQUksRUFBRSx3REFBd0Q7UUFDOUQsV0FBVyxFQUNULDJNQUEyTTtRQUM3TSxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsS0FBSyxDQUFDLHNCQUFzQjtLQUN6QztJQUNEO1FBQ0UsSUFBSSxFQUFFLHNEQUFzRDtRQUM1RCxXQUFXLEVBQ1Qsc0hBQXNIO1FBQ3hILEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxNQUFNLENBQUMsb0NBQW9DO0tBQ3hEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsbUZBQW1GO1FBQ3pGLFdBQVcsRUFDVCw4bUJBQThtQjtRQUNobkIsS0FBSyxFQUFFLHlCQUFlLENBQUMsS0FBSztRQUM1QixJQUFJLEVBQUUsZUFBSyxDQUFDLFdBQVcsQ0FBQyx1Q0FBdUM7S0FDaEU7SUFDRDtRQUNFLElBQUksRUFBRSxzRkFBc0Y7UUFDNUYsV0FBVyxFQUNULG1oQkFBbWhCO1FBQ3JoQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsTUFBTSxDQUFDLHFDQUFxQztLQUN6RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLDhHQUE4RztRQUNwSCxXQUFXLEVBQ1Qsa2VBQWtlO1FBQ3BlLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxVQUFVLENBQUMscUNBQXFDO0tBQzdEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUscUZBQXFGO1FBQzNGLFdBQVcsRUFDVCwrZEFBK2Q7UUFDamUsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFVBQVUsQ0FBQyw4Q0FBOEM7S0FDdEU7SUFDRDtRQUNFLElBQUksRUFBRSxtRkFBbUY7UUFDekYsV0FBVyxFQUNULHVPQUF1TztRQUN6TyxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsVUFBVSxDQUFDLG1DQUFtQztLQUMzRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLGtGQUFrRjtRQUN4RixXQUFXLEVBQ1Qsd2JBQXdiO1FBQzFiLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxVQUFVLENBQUMsNENBQTRDO0tBQ3BFO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsbUVBQW1FO1FBQ3pFLFdBQVcsRUFDVCxtS0FBbUs7UUFDckssS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFVBQVUsQ0FBQyxvQ0FBb0M7S0FDNUQ7SUFDRDtRQUNFLElBQUksRUFBRSxxSkFBcUo7UUFDM0osV0FBVyxFQUNULHllQUF5ZTtRQUMzZSxLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsU0FBUyxDQUFDLDZCQUE2QjtLQUNwRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLG1FQUFtRTtRQUN6RSxXQUFXLEVBQ1QsaUlBQWlJO1FBQ25JLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxTQUFTLENBQUMscUNBQXFDO0tBQzVEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsd0pBQXdKO1FBQzlKLFdBQVcsRUFDVCxzR0FBc0c7UUFDeEcsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFNBQVMsQ0FBQyw2QkFBNkI7S0FDcEQ7SUFDRDtRQUNFLElBQUksRUFBRSxvREFBb0Q7UUFDMUQsV0FBVyxFQUNULG9RQUFvUTtRQUN0USxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsU0FBUyxDQUFDLHNDQUFzQztLQUM3RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLDJFQUEyRTtRQUNqRixXQUFXLEVBQ1QsaUdBQWlHO1FBQ25HLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxPQUFPLENBQUMsMkNBQTJDO0tBQ2hFO0lBQ0Q7UUFDRSxJQUFJLEVBQUUscUVBQXFFO1FBQzNFLFdBQVcsRUFDVCxpTEFBaUw7UUFDbkwsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLE9BQU8sQ0FBQyw4QkFBOEI7S0FDbkQ7SUFDRDtRQUNFLElBQUksRUFBRSx5REFBeUQ7UUFDL0QsV0FBVyxFQUNULHFLQUFxSztRQUN2SyxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsT0FBTyxDQUFDLGtCQUFrQjtLQUN2QztJQUNEO1FBQ0UsSUFBSSxFQUFFLHlIQUF5SDtRQUMvSCxXQUFXLEVBQ1QsbWtCQUFta0I7UUFDcmtCLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxPQUFPLENBQUMsc0NBQXNDO0tBQzNEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMkxBQTJMO1FBQ2pNLFdBQVcsRUFDVCw0aEJBQTRoQjtRQUM5aEIsS0FBSyxFQUFFLHlCQUFlLENBQUMsS0FBSztRQUM1QixJQUFJLEVBQUUsZUFBSyxDQUFDLE9BQU8sQ0FBQyxtQ0FBbUM7S0FDeEQ7SUFDRDtRQUNFLElBQUksRUFBRSx5Q0FBeUM7UUFDL0MsV0FBVyxFQUNULHdjQUF3YztRQUMxYyxLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLHVCQUF1QjtLQUN4QztJQUNEO1FBQ0UsSUFBSSxFQUFFLG1FQUFtRTtRQUN6RSxXQUFXLEVBQ1QsNkhBQTZIO1FBQy9ILEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxVQUFVLENBQUMsaUNBQWlDO0tBQ3pEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUseURBQXlEO1FBQy9ELFdBQVcsRUFDVCxrSkFBa0o7UUFDcEosS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFVBQVUsQ0FBQywrQkFBK0I7S0FDdkQ7SUFDRDtRQUNFLElBQUksRUFBRSw0RkFBNEY7UUFDbEcsV0FBVyxFQUNULHE2QkFBcTZCO1FBQ3Y2QixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsVUFBVSxDQUFDLHFDQUFxQztLQUM3RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLCtEQUErRDtRQUNyRSxXQUFXLEVBQ1QsMEtBQTBLO1FBQzVLLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxRQUFRLENBQUMsWUFBWTtLQUNsQztJQUNEO1FBQ0UsSUFBSSxFQUFFLHlDQUF5QztRQUMvQyxXQUFXLEVBQ1QsOE5BQThOO1FBQ2hPLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMscUJBQXFCO0tBQ3RDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMERBQTBEO1FBQ2hFLFdBQVcsRUFDVCwyR0FBMkc7UUFDN0csS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyxxQkFBcUI7S0FDdEM7SUFDRDtRQUNFLElBQUksRUFBRSw4REFBOEQ7UUFDcEUsV0FBVyxFQUNULHdLQUF3SztRQUMxSyxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLDBCQUEwQjtLQUMzQztJQUNEO1FBQ0UsSUFBSSxFQUFFLHVDQUF1QztRQUM3QyxXQUFXLEVBQ1QsMmZBQTJmO1FBQzdmLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsaUJBQWlCO0tBQ2xDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsNEpBQTRKO1FBQ2xLLFdBQVcsRUFDVCxnbkJBQWduQjtRQUNsbkIsS0FBSyxFQUFFLHlCQUFlLENBQUMsS0FBSztRQUM1QixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyx3QkFBd0I7S0FDekM7SUFDRDtRQUNFLElBQUksRUFBRSxpRUFBaUU7UUFDdkUsV0FBVyxFQUNULHEzQkFBcTNCO1FBQ3YzQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLG9CQUFvQjtLQUNyQztJQUNEO1FBQ0UsSUFBSSxFQUFFLG9EQUFvRDtRQUMxRCxXQUFXLEVBQ1QsK2pCQUErakI7UUFDamtCLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsZ0JBQWdCO0tBQ2pDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsaURBQWlEO1FBQ3ZELFdBQVcsRUFDVCx5RkFBeUY7UUFDM0YsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQywyQkFBMkI7S0FDNUM7SUFDRDtRQUNFLElBQUksRUFBRSx3Q0FBd0M7UUFDOUMsV0FBVyxFQUNULDhlQUE4ZTtRQUNoZixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLGFBQWE7S0FDOUI7SUFDRDtRQUNFLElBQUksRUFBRSwwRkFBMEY7UUFDaEcsV0FBVyxFQUNULGdKQUFnSjtRQUNsSixLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLGlDQUFpQztLQUNsRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLHdHQUF3RztRQUM5RyxXQUFXLEVBQ1Qsb1RBQW9UO1FBQ3RULEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsdUNBQXVDO0tBQ3hEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsbURBQW1EO1FBQ3pELFdBQVcsRUFDVCw0SUFBNEk7UUFDOUksS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyxZQUFZO0tBQzdCO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMkhBQTJIO1FBQ2pJLFdBQVcsRUFDVCxrYUFBa2E7UUFDcGEsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQywwQkFBMEI7S0FDM0M7SUFDRDtRQUNFLElBQUksRUFBRSw2RUFBNkU7UUFDbkYsV0FBVyxFQUNULHFtQkFBcW1CO1FBQ3ZtQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLGdDQUFnQztLQUNqRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLHNEQUFzRDtRQUM1RCxXQUFXLEVBQ1QsdWZBQXVmO1FBQ3pmLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxXQUFXLENBQUMsdUJBQXVCO0tBQ2hEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUseURBQXlEO1FBQy9ELFdBQVcsRUFDVCx3S0FBd0s7UUFDMUssS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFdBQVcsQ0FBQyxnQ0FBZ0M7S0FDekQ7SUFDRDtRQUNFLElBQUksRUFBRSw2RkFBNkY7UUFDbkcsV0FBVyxFQUNULDJJQUEySTtRQUM3SSxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsV0FBVyxDQUFDLGlDQUFpQztLQUMxRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLGdGQUFnRjtRQUN0RixXQUFXLEVBQ1Qsc0pBQXNKO1FBQ3hKLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxXQUFXLENBQUMsZ0NBQWdDO0tBQ3pEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsNEVBQTRFO1FBQ2xGLFdBQVcsRUFDVCw2T0FBNk87UUFDL08sS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLGdCQUFnQixDQUFDLG1DQUFtQztLQUNqRTtJQUNEO1FBQ0UsSUFBSSxFQUFFLDBFQUEwRTtRQUNoRixXQUFXLEVBQ1QsZ2lCQUFnaUI7UUFDbGlCLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxxQ0FBcUM7S0FDbkU7SUFDRDtRQUNFLElBQUksRUFBRSw0RUFBNEU7UUFDbEYsV0FBVyxFQUNULHFGQUFxRjtRQUN2RixLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsZ0JBQWdCLENBQUMsNEJBQTRCO0tBQzFEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsbUVBQW1FO1FBQ3pFLFdBQVcsRUFDVCxtT0FBbU87UUFDck8sS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyx5QkFBeUI7S0FDMUM7SUFDRDtRQUNFLElBQUksRUFBRSw4RUFBOEU7UUFDcEYsV0FBVyxFQUNULHNMQUFzTDtRQUN4TCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLHdCQUF3QjtLQUN6QztJQUNEO1FBQ0UsSUFBSSxFQUFFLG9GQUFvRjtRQUMxRixXQUFXLEVBQ1QsbU9BQW1PO1FBQ3JPLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsMkJBQTJCO0tBQzVDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMEdBQTBHO1FBQ2hILFdBQVcsRUFDVCx3R0FBd0c7UUFDMUcsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0I7S0FDdkM7SUFDRDtRQUNFLElBQUksRUFBRSwyRkFBMkY7UUFDakcsV0FBVyxFQUNULHFIQUFxSDtRQUN2SCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLHNCQUFzQjtLQUN2QztJQUNEO1FBQ0UsSUFBSSxFQUFFLDhDQUE4QztRQUNwRCxXQUFXLEVBQ1QsaWtCQUFpa0I7UUFDbmtCLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxXQUFXLENBQUMsa0JBQWtCO0tBQzNDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsdUdBQXVHO1FBQzdHLFdBQVcsRUFDVCxvSUFBb0k7UUFDdEksS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLElBQUksQ0FBQywyQkFBMkI7S0FDN0M7SUFDRDtRQUNFLElBQUksRUFBRSwrRkFBK0Y7UUFDckcsV0FBVyxFQUNULDRGQUE0RjtRQUM5RixLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsSUFBSSxDQUFDLHdCQUF3QjtLQUMxQztJQUNEO1FBQ0UsSUFBSSxFQUFFLHlEQUF5RDtRQUMvRCxXQUFXLEVBQ1QsK2RBQStkO1FBQ2plLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsb0JBQW9CO0tBQ3JDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsK0hBQStIO1FBQ3JJLFdBQVcsRUFDVCw4V0FBOFc7UUFDaFgsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyx3QkFBd0I7S0FDekM7SUFDRDtRQUNFLElBQUksRUFBRSx3SEFBd0g7UUFDOUgsV0FBVyxFQUNULGl5QkFBaXlCO1FBQ255QixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLG9DQUFvQztLQUNyRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLHlHQUF5RztRQUMvRyxXQUFXLEVBQ1QsdU5BQXVOO1FBQ3pOLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsbUNBQW1DO0tBQ3BEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUseUZBQXlGO1FBQy9GLFdBQVcsRUFDVCx1SEFBdUg7UUFDekgsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLE9BQU8sQ0FBQyxzQkFBc0I7S0FDM0M7SUFDRDtRQUNFLElBQUksRUFBRSxrR0FBa0c7UUFDeEcsV0FBVyxFQUNULGtRQUFrUTtRQUNwUSxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsT0FBTyxDQUFDLGtDQUFrQztLQUN2RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLHVFQUF1RTtRQUM3RSxXQUFXLEVBQ1Qsb09BQW9PO1FBQ3RPLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxPQUFPLENBQUMsb0JBQW9CO0tBQ3pDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUscUVBQXFFO1FBQzNFLFdBQVcsRUFDVCw0T0FBNE87UUFDOU8sS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyw0QkFBNEI7S0FDN0M7SUFDRDtRQUNFLElBQUksRUFBRSxxREFBcUQ7UUFDM0QsV0FBVyxFQUNULCtpQkFBK2lCO1FBQ2pqQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsTUFBTSxDQUFDLG9DQUFvQztLQUN4RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLG9FQUFvRTtRQUMxRSxXQUFXLEVBQ1Qsa2pCQUFrakI7UUFDcGpCLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxNQUFNLENBQUMscUJBQXFCO0tBQ3pDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsd0ZBQXdGO1FBQzlGLFdBQVcsRUFDVCx5UEFBeVA7UUFDM1AsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLE1BQU0sQ0FBQyxtQkFBbUI7S0FDdkM7SUFDRDtRQUNFLElBQUksRUFBRSx5REFBeUQ7UUFDL0QsV0FBVyxFQUNULCtLQUErSztRQUNqTCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsVUFBVSxDQUFDLDZCQUE2QjtLQUNyRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLDhEQUE4RDtRQUNwRSxXQUFXLEVBQ1Qsb0xBQW9MO1FBQ3RMLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxVQUFVLENBQUMscUNBQXFDO0tBQzdEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsZ0VBQWdFO1FBQ3RFLFdBQVcsRUFDVCx5U0FBeVM7UUFDM1MsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFVBQVUsQ0FBQyxrQ0FBa0M7S0FDMUQ7SUFDRDtRQUNFLElBQUksRUFBRSwrREFBK0Q7UUFDckUsV0FBVyxFQUNULHlGQUF5RjtRQUMzRixLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLG9CQUFvQjtLQUNyQztJQUNEO1FBQ0UsSUFBSSxFQUFFLDJFQUEyRTtRQUNqRixXQUFXLEVBQ1QsOEdBQThHO1FBQ2hILEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsb0JBQW9CO0tBQ3JDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsdUVBQXVFO1FBQzdFLFdBQVcsRUFDVCwwZkFBMGY7UUFDNWYsS0FBSyxFQUFFLHlCQUFlLENBQUMsS0FBSztRQUM1QixJQUFJLEVBQUUsZUFBSyxDQUFDLE9BQU8sQ0FBQywwQ0FBMEM7S0FDL0Q7SUFDRDtRQUNFLElBQUksRUFBRSxrRUFBa0U7UUFDeEUsV0FBVyxFQUNULDZIQUE2SDtRQUMvSCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsT0FBTyxDQUFDLDhCQUE4QjtLQUNuRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLDJFQUEyRTtRQUNqRixXQUFXLEVBQ1QscUhBQXFIO1FBQ3ZILEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxPQUFPLENBQUMscUJBQXFCO0tBQzFDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsd0ZBQXdGO1FBQzlGLFdBQVcsRUFDVCxzRkFBc0Y7UUFDeEYsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFVBQVUsQ0FBQyx3QkFBd0I7S0FDaEQ7SUFDRDtRQUNFLElBQUksRUFBRSx5RUFBeUU7UUFDL0UsV0FBVyxFQUNULGtLQUFrSztRQUNwSyxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsVUFBVSxDQUFDLHlCQUF5QjtLQUNqRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLGdFQUFnRTtRQUN0RSxXQUFXLEVBQ1QsZ21CQUFnbUI7UUFDbG1CLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxVQUFVLENBQUMsbUJBQW1CO0tBQzNDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsOEVBQThFO1FBQ3BGLFdBQVcsRUFDVCw4SkFBOEo7UUFDaEssS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFVBQVUsQ0FBQyw4QkFBOEI7S0FDdEQ7SUFDRDtRQUNFLElBQUksRUFBRSx5RkFBeUY7UUFDL0YsV0FBVyxFQUNULHdkQUF3ZDtRQUMxZCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsVUFBVSxDQUFDLHFDQUFxQztLQUM3RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLHNFQUFzRTtRQUM1RSxXQUFXLEVBQ1Qsc01BQXNNO1FBQ3hNLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxVQUFVLENBQUMsd0JBQXdCO0tBQ2hEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsNkZBQTZGO1FBQ25HLFdBQVcsRUFDVCwwTEFBMEw7UUFDNUwsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQywwQkFBMEI7S0FDM0M7SUFDRDtRQUNFLElBQUksRUFBRSw2RUFBNkU7UUFDbkYsV0FBVyxFQUNULDBsQkFBMGxCO1FBQzVsQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLHNDQUFzQztLQUN2RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLDJDQUEyQztRQUNqRCxXQUFXLEVBQ1Qsc2pCQUFzakI7UUFDeGpCLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsdUJBQXVCO0tBQ3hDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsdUVBQXVFO1FBQzdFLFdBQVcsRUFDVCx1TUFBdU07UUFDek0sS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUI7S0FDbEM7SUFDRDtRQUNFLElBQUksRUFBRSxnRUFBZ0U7UUFDdEUsV0FBVyxFQUNULHFvQkFBcW9CO1FBQ3ZvQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLG9CQUFvQjtLQUNyQztJQUNEO1FBQ0UsSUFBSSxFQUFFLG9GQUFvRjtRQUMxRixXQUFXLEVBQ1QsMkhBQTJIO1FBQzdILEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxHQUFHLENBQUMsbUJBQW1CO0tBQ3BDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsZ0VBQWdFO1FBQ3RFLFdBQVcsRUFBRSx3REFBd0Q7UUFDckUsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFFBQVEsQ0FBQywrQkFBK0I7S0FDckQ7SUFDRDtRQUNFLElBQUksRUFBRSxtREFBbUQ7UUFDekQsV0FBVyxFQUNULDBqQkFBMGpCO1FBQzVqQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsUUFBUSxDQUFDLG9CQUFvQjtLQUMxQztJQUNEO1FBQ0UsSUFBSSxFQUFFLHNEQUFzRDtRQUM1RCxXQUFXLEVBQ1Qsa0pBQWtKO1FBQ3BKLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxRQUFRLENBQUMsNkJBQTZCO0tBQ25EO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMkRBQTJEO1FBQ2pFLFdBQVcsRUFDVCx5SkFBeUo7UUFDM0osS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFFBQVEsQ0FBQyxpQ0FBaUM7S0FDdkQ7SUFDRDtRQUNFLElBQUksRUFBRSw0Q0FBNEM7UUFDbEQsV0FBVyxFQUNULDJpQkFBMmlCO1FBQzdpQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsUUFBUSxDQUFDLDJCQUEyQjtLQUNqRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLDZEQUE2RDtRQUNuRSxXQUFXLEVBQ1QsbWNBQW1jO1FBQ3JjLEtBQUssRUFBRSx5QkFBZSxDQUFDLEtBQUs7UUFDNUIsSUFBSSxFQUFFLGVBQUssQ0FBQyxRQUFRLENBQUMsNkJBQTZCO0tBQ25EO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMkRBQTJEO1FBQ2pFLFdBQVcsRUFDVCwyTEFBMkw7UUFDN0wsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFFBQVEsQ0FBQyxxQkFBcUI7S0FDM0M7SUFDRDtRQUNFLElBQUksRUFBRSw4RUFBOEU7UUFDcEYsV0FBVyxFQUNULGlhQUFpYTtRQUNuYSxLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsRUFBRSxDQUFDLG1DQUFtQztLQUNuRDtJQUNEO1FBQ0UsSUFBSSxFQUFFLHlEQUF5RDtRQUMvRCxXQUFXLEVBQ1Qsa2NBQWtjO1FBQ3BjLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxFQUFFLENBQUMsc0JBQXNCO0tBQ3RDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsb0hBQW9IO1FBQzFILFdBQVcsRUFDVCx1VkFBdVY7UUFDelYsS0FBSyxFQUFFLHlCQUFlLENBQUMsS0FBSztRQUM1QixJQUFJLEVBQUUsZUFBSyxDQUFDLEVBQUUsQ0FBQyw0QkFBNEI7S0FDNUM7SUFDRDtRQUNFLElBQUksRUFBRSxxSEFBcUg7UUFDM0gsV0FBVyxFQUNULDRYQUE0WDtRQUM5WCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsRUFBRSxDQUFDLDZCQUE2QjtLQUM3QztJQUNEO1FBQ0UsSUFBSSxFQUFFLHFFQUFxRTtRQUMzRSxXQUFXLEVBQ1QsNkdBQTZHO1FBQy9HLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxFQUFFLENBQUMsbUNBQW1DO0tBQ25EO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsMEVBQTBFO1FBQ2hGLFdBQVcsRUFDVCxxU0FBcVM7UUFDdlMsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEVBQUUsQ0FBQyx1QkFBdUI7S0FDdkM7SUFDRDtRQUNFLElBQUksRUFBRSx3TUFBd007UUFDOU0sV0FBVyxFQUNULDZtQkFBNm1CO1FBQy9tQixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsRUFBRSxDQUFDLG9CQUFvQjtLQUNwQztJQUNEO1FBQ0UsSUFBSSxFQUFFLDRFQUE0RTtRQUNsRixXQUFXLEVBQ1Qsa0VBQWtFO1FBQ3BFLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxTQUFTLENBQUMseUNBQXlDO0tBQ2hFO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsa0VBQWtFO1FBQ3hFLFdBQVcsRUFDVCw4SEFBOEg7UUFDaEksS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLFNBQVMsQ0FBQyxzQkFBc0I7S0FDN0M7SUFDRDtRQUNFLElBQUksRUFBRSxpRUFBaUU7UUFDdkUsV0FBVyxFQUNULDhHQUE4RztRQUNoSCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsU0FBUyxDQUFDLHVDQUF1QztLQUM5RDtJQUNEO1FBQ0UsSUFBSSxFQUFFLHdEQUF3RDtRQUM5RCxXQUFXLEVBQ1QsNEpBQTRKO1FBQzlKLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxjQUFjLENBQUMsNkJBQTZCO0tBQ3pEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUscURBQXFEO1FBQzNELFdBQVcsRUFDVCwyUkFBMlI7UUFDN1IsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyxlQUFlO0tBQ2hDO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsdURBQXVEO1FBQzdELFdBQVcsRUFDVCxzWEFBc1g7UUFDeFgsS0FBSyxFQUFFLHlCQUFlLENBQUMsSUFBSTtRQUMzQixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0I7S0FDdkM7SUFDRDtRQUNFLElBQUksRUFBRSw2REFBNkQ7UUFDbkUsV0FBVyxFQUNULDJHQUEyRztRQUM3RyxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLFdBQVc7S0FDNUI7SUFDRDtRQUNFLElBQUksRUFBRSxxREFBcUQ7UUFDM0QsV0FBVyxFQUNULHlSQUF5UjtRQUMzUixLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLHVCQUF1QjtLQUN4QztJQUNEO1FBQ0UsSUFBSSxFQUFFLHlGQUF5RjtRQUMvRixXQUFXLEVBQ1QsZ1ZBQWdWO1FBQ2xWLEtBQUssRUFBRSx5QkFBZSxDQUFDLElBQUk7UUFDM0IsSUFBSSxFQUFFLGVBQUssQ0FBQyxVQUFVLENBQUMsb0NBQW9DO0tBQzVEO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsc0VBQXNFO1FBQzVFLFdBQVcsRUFDVCw4eEJBQTh4QjtRQUNoeUIsS0FBSyxFQUFFLHlCQUFlLENBQUMsS0FBSztRQUM1QixJQUFJLEVBQUUsZUFBSyxDQUFDLEdBQUcsQ0FBQyw2QkFBNkI7S0FDOUM7SUFDRDtRQUNFLElBQUksRUFBRSwwREFBMEQ7UUFDaEUsV0FBVyxFQUNULHNUQUFzVDtRQUN4VCxLQUFLLEVBQUUseUJBQWUsQ0FBQyxJQUFJO1FBQzNCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLFVBQVU7S0FDM0I7SUFDRDtRQUNFLElBQUksRUFBRSw4Q0FBOEM7UUFDcEQsV0FBVyxFQUNULG10QkFBbXRCO1FBQ3J0QixLQUFLLEVBQUUseUJBQWUsQ0FBQyxLQUFLO1FBQzVCLElBQUksRUFBRSxlQUFLLENBQUMsR0FBRyxDQUFDLG1DQUFtQztLQUNwRDtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCB7IE5hZ01lc3NhZ2VMZXZlbCwgcnVsZXMgfSBmcm9tIFwiY2RrLW5hZ1wiO1xuXG4vKipcbiAqIFBhY2sgbmFtZVxuICpcbiAqL1xuZXhwb3J0IGxldCBQYWNrTmFtZSA9IFwiQXdzUHJvdG90eXBpbmdcIjtcblxuLyoqXG4gKiBSdWxlIG1ldGFkYXRhXG4gKlxuICovXG5leHBvcnQgbGV0IFJ1bGVNZXRhZGF0YSA9IFtcbiAge1xuICAgIGluZm86IFwiVGhlIFJFU1QgQVBJIHN0YWdlIGlzIG5vdCBhc3NvY2lhdGVkIHdpdGggQVdTIFdBRnYyIHdlYiBBQ0wuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkFXUyBXQUZ2MiBpcyBhIHdlYiBhcHBsaWNhdGlvbiBmaXJld2FsbCB0aGF0IGhlbHBzIHByb3RlY3Qgd2ViIGFwcGxpY2F0aW9ucyBhbmQgQVBJcyBmcm9tIGF0dGFja3MgYnkgYWxsb3dpbmcgY29uZmlndXJlZCBydWxlcyB0byBhbGxvdywgYmxvY2ssIG9yIG1vbml0b3IgKGNvdW50KSB3ZWIgcmVxdWVzdHMgYmFzZWQgb24gY3VzdG9taXphYmxlIHJ1bGVzIGFuZCBjb25kaXRpb25zIHRoYXQgYXJlIGRlZmluZWQuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmFwaWd3LkFQSUdXQXNzb2NpYXRlZFdpdGhXQUYsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBBUEkgZG9lcyBub3QgaW1wbGVtZW50IGF1dGhvcml6YXRpb24uXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkluIG1vc3QgY2FzZXMgYW4gQVBJIG5lZWRzIHRvIGhhdmUgYW4gYXV0aGVudGljYXRpb24gYW5kIGF1dGhvcml6YXRpb24gaW1wbGVtZW50YXRpb24gc3RyYXRlZ3kuIFRoaXMgaW5jbHVkZXMgdXNpbmcgc3VjaCBhcHByb2FjaGVzIGFzIElBTSwgQ29nbml0byBVc2VyIFBvb2xzLCBDdXN0b20gYXV0aG9yaXplciwgZXRjLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBbiBhY3RvciB3aXRoIGEgbmV0d29yayBwYXRoIHRvIGFuIEFQSSBnYXRld2F5IHN0YWdlIGVuZC1wb2ludCBjYW4gaW50ZXJhY3Qgd2l0aCB0aGUgQVBJIG1ldGhvZCBpbiBxdWVzdGlvbiB3aXRob3V0IGF1dGhvcml6YXRpb24sIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5hcGlndy5BUElHV0F1dGhvcml6YXRpb24sXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBSRVNUIEFQSSBkb2VzIG5vdCBoYXZlIHJlcXVlc3QgdmFsaWRhdGlvbiBlbmFibGVkLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJUaGUgQVBJIHNob3VsZCBoYXZlIGJhc2ljIHJlcXVlc3QgdmFsaWRhdGlvbiBlbmFibGVkLiBJZiB0aGUgQVBJIGlzIGludGVncmF0ZWQgd2l0aCBhIGN1c3RvbSBzb3VyY2UgKExhbWJkYSwgRUNTLCBldGMuLikgaW4gdGhlIGJhY2tlbmQsIGRlZXBlciBpbnB1dCB2YWxpZGF0aW9uIHNob3VsZCBiZSBjb25zaWRlcmVkIGZvciBpbXBsZW1lbnRhdGlvbi5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuYXBpZ3cuQVBJR1dSZXF1ZXN0VmFsaWRhdGlvbixcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEF0aGVuYSB3b3JrZ3JvdXAgZG9lcyBub3QgZW5jcnlwdCBxdWVyeSByZXN1bHRzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJFbmNyeXB0aW5nIHF1ZXJ5IHJlc3VsdHMgc3RvcmVkIGluIFMzIGhlbHBzIHNlY3VyZSBkYXRhIHRvIG1lZXQgY29tcGxpYW5jZSByZXF1aXJlbWVudHMgZm9yIGRhdGEtYXQtcmVzdCBlbmNyeXB0aW9uLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5hdGhlbmEuQXRoZW5hV29ya2dyb3VwRW5jcnlwdGVkUXVlcnlSZXN1bHRzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQXV0byBTY2FsaW5nIGxhdW5jaCBjb25maWd1cmF0aW9uIGRvZXMgbm90IGhhdmUgcHVibGljIElQIGFkZHJlc3NlcyBkaXNhYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiSWYgeW91IGNvbmZpZ3VyZSB5b3VyIE5ldHdvcmsgSW50ZXJmYWNlcyB3aXRoIGEgcHVibGljIElQIGFkZHJlc3MsIHRoZW4gdGhlIGFzc29jaWF0ZWQgcmVzb3VyY2VzIHRvIHRob3NlIE5ldHdvcmsgSW50ZXJmYWNlcyBhcmUgcmVhY2hhYmxlIGZyb20gdGhlIGludGVybmV0LiBFQzIgcmVzb3VyY2VzIHNob3VsZCBub3QgYmUgcHVibGljbHkgYWNjZXNzaWJsZSwgYXMgdGhpcyBtYXkgYWxsb3cgdW5pbnRlbmRlZCBhY2Nlc3MgdG8geW91ciBhcHBsaWNhdGlvbnMgb3Igc2VydmVycy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3IgY2FuIGRpc2NvdmVyIEVDMiBpbnN0YW5jZXMgdGhhdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuYXV0b3NjYWxpbmcuQXV0b1NjYWxpbmdMYXVuY2hDb25maWdQdWJsaWNJcERpc2FibGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQ2xvdWQ5IGluc3RhbmNlIGRvZXMgbm90IHVzZSBhIG5vLWluZ3Jlc3MgRUMyIGluc3RhbmNlIHdpdGggQVdTIFN5c3RlbXMgTWFuYWdlci5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiU1NNIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBwcm90ZWN0aW9uIGFzIGl0IGFsbG93cyBvcGVyYXRvcnMgdG8gY29udHJvbCBhY2Nlc3MgdGhyb3VnaCBJQU0gcGVybWlzc2lvbnMgYW5kIGRvZXMgbm90IHJlcXVpcmUgb3BlbmluZyBpbmJvdW5kIHBvcnRzLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciBjYW4gZGlzY292ZXIgQ2xvdWQ5IEVDMiBpbnN0YW5jZXMgdGhhdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMgYW5kIHRoYXQgYXJlIGV4cG9zaW5nIFNTSCwgd2hpY2ggbWF5IGxlYWQgdG8gcmVjb25uYWlzc2FuY2UgYW5kIGludHJ1c2lvbiBhY3Rpdml0aWVzIGJlaW5nIHBlcmZvcm1lZCBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhbmQgcmVzb3VyY2UgYXNzZXRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5jbG91ZDkuQ2xvdWQ5SW5zdGFuY2VOb0luZ3Jlc3NTeXN0ZW1zTWFuYWdlcixcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIHByb3RvdHlwZXMgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24gaGFzIG5vdCBiZWVuIGNvbmZpZ3VyZWQgd2l0aCBnZW9ncmFwaGljIHJlc3RyaWN0aW9ucyAoR2VvUmVzdHJpY3Rpb24pXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkdlbyByZXN0cmljdGlvbiBzaG91bGQgYmUgZW5hYmxlZCBmb3IgdGhlIGRpc3RyaWJ1dGlvbiBpbiBvcmRlciB0byBsaW1pdCB0aGUgc3VyZmFjZSBhcmVhIGV4cG9zZWQgdG8gZXhwZWN0ZWQgZ2VvZ3JhcGhpZXNcXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3IgY2FuIGRpc2NvdmVyIHByb3RvdHlwZSB3ZWIgYXNzZXRzIHRoYXQgYXJlIGV4cG9zZWQgdmlhIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9ucywgd2hpY2ggbWF5IGxlYWQgdG8gcmVjb24gYW5kIGludHJ1c2lvbiBhY3Rpdml0aWVzIGJlaW5nIHBlcmZvcm1lZCBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhbmQgcmVzb3VyY2UgYXNzZXRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5jbG91ZGZyb250LkNsb3VkRnJvbnREaXN0cmlidXRpb25HZW9SZXN0cmljdGlvbnMsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiBhbGxvd3MgZm9yIFNTTHYzIG9yIFRMU3YxIGZvciBIVFRQUyB2aWV3ZXIgY29ubmVjdGlvbnMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlZ1bG5lcmFiaWxpdGllcyBoYXZlIGJlZW4gYW5kIGNvbnRpbnVlIHRvIGJlIGRpc2NvdmVyZWQgaW4gdGhlIGRlcHJlY2F0ZWQgU1NMIGFuZCBUTFMgcHJvdG9jb2xzLiBIZWxwIHByb3RlY3Qgdmlld2VyIGNvbm5lY3Rpb25zIGJ5IHNwZWNpZnlpbmcgYSB2aWV3ZXIgY2VydGlmaWNhdGUgdGhhdCBlbmZvcmNlcyBhIG1pbmltdW0gb2YgVExTdjEuMSBvciBUTFN2MS4yIGluIHRoZSBzZWN1cml0eSBwb2xpY3kuIERpc3RyaWJ1dGlvbnMgdGhhdCB1c2UgdGhhdCB1c2UgdGhlIGRlZmF1bHQgQ2xvdWRGcm9udCB2aWV3ZXIgY2VydGlmaWNhdGUgb3IgdXNlICd2aXAnIGZvciB0aGUgU3NsU3VwcG9ydE1ldGhvZCBhcmUgbm9uLWNvbXBsaWFudCB3aXRoIHRoaXMgcnVsZSwgYXMgdGhlIG1pbmltdW0gc2VjdXJpdHkgcG9saWN5IGlzIHNldCB0byBUTFN2MSByZWdhcmRsZXNzIG9mIHRoZSBzcGVjaWZpZWQgTWluaW11bVByb3RvY29sVmVyc2lvblwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5jbG91ZGZyb250LkNsb3VkRnJvbnREaXN0cmlidXRpb25IdHRwc1ZpZXdlck5vT3V0ZGF0ZWRTU0wsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbnMgdXNlcyBTU0x2MyBvciBUTFN2MSBmb3IgY29tbXVuaWNhdGlvbiB0byB0aGUgb3JpZ2luLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJWdWxuZXJhYmlsaXRpZXMgaGF2ZSBiZWVuIGFuZCBjb250aW51ZSB0byBiZSBkaXNjb3ZlcmVkIGluIHRoZSBkZXByZWNhdGVkIFNTTCBhbmQgVExTIHByb3RvY29scy4gVXNpbmcgYSBzZWN1cml0eSBwb2xpY3kgd2l0aCBtaW5pbXVtIFRMU3YxLjEgb3IgVExTdjEuMiBhbmQgYXBwcm9wcmlhdGUgc2VjdXJpdHkgY2lwaGVycyBmb3IgSFRUUFMgaGVscHMgcHJvdGVjdCB2aWV3ZXIgY29ubmVjdGlvbnMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmNsb3VkZnJvbnQuQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk5vT3V0ZGF0ZWRTU0wsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiBkb2VzIG5vdCB1c2UgYW4gb3JpZ2luIGFjY2VzcyBpZGVudGl0eSBhbiBTMyBvcmlnaW4uXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIk9yaWdpbiBhY2Nlc3MgaWRlbnRpdGllcyBoZWxwIHdpdGggc2VjdXJpdHkgYnkgcmVzdHJpY3RpbmcgYW55IGRpcmVjdCBhY2Nlc3MgdG8gb2JqZWN0cyB0aHJvdWdoIFMzIFVSTHMuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIGNhbiBieXBhc3MgdGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uIGFuZCBhc3NvY2lhdGVkIGNvbnRyb2xzIChlLmcuIGdlbyBibG9ja2luZyksIHdoaWNoIG1heSBsZWFkIHRvIGRpcmVjdCBhY2Nlc3MgdG8gc3RhdGljIGFzc2V0cyBob3N0ZWQgb24gdGhlIFMzIG9yaWdpbiBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYXNzZXRzIGhvc3RlZCBvbiB0aGUgUzMgb3JpZ2luIGZvciB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5jbG91ZGZyb250LkNsb3VkRnJvbnREaXN0cmlidXRpb25TM09yaWdpbkFjY2Vzc0lkZW50aXR5LFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24gbWF5IHJlcXVpcmUgaW50ZWdyYXRpb24gd2l0aCBBV1MgV0FGLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJUaGUgV2ViIEFwcGxpY2F0aW9uIEZpcmV3YWxsIGNhbiBoZWxwIHByb3RlY3QgYWdhaW5zdCBhcHBsaWNhdGlvbi1sYXllciBhdHRhY2tzIHRoYXQgY2FuIGNvbXByb21pc2UgdGhlIHNlY3VyaXR5IG9mIHRoZSBzeXN0ZW0gb3IgcGxhY2UgdW5uZWNlc3NhcnkgbG9hZCBvbiB0aGVtLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5jbG91ZGZyb250LkNsb3VkRnJvbnREaXN0cmlidXRpb25XQUZJbnRlZ3JhdGlvbixcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIENvZGVCdWlsZCBlbnZpcm9ubWVudCBzdG9yZXMgc2Vuc2l0aXZlIGNyZWRlbnRpYWxzIChzdWNoIGFzIEFXU19BQ0NFU1NfS0VZX0lEIGFuZC9vciBBV1NfU0VDUkVUX0FDQ0VTU19LRVkpIGFzIHBsYWludGV4dCBlbnZpcm9ubWVudCB2YXJpYWJsZXMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkRvIG5vdCBzdG9yZSB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dC4gU3RvcmluZyB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dCBsZWFkcyB0byB1bmludGVuZGVkIGRhdGEgZXhwb3N1cmUgYW5kIHVuYXV0aG9yaXplZCBhY2Nlc3MuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEFuIGFjdG9yIHdobyBjYW4gdmlldyB0aGUgQ29kZUJ1aWxkIGVudmlyb25tZW50IHZhcmlhYmxlcyBjYW4gb2J0YWluIHRoZSBBV1MgQWNjZXNzIEtleSBhbmQgU2VjcmV0IEFjY2VzcyBLZXksIHdoaWNoIG1heSBsZWFkIHRvIHRoZSBhY3RvciBiZWluZyBhYmxlIHRvIGRvIGFueXRoaW5nIHRoZSBBV1Mga2V5cyBhcmUgYXV0aG9yaXNlZCB0byBkbyBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuY29kZWJ1aWxkLkNvZGVCdWlsZFByb2plY3RFbnZWYXJBd3NDcmVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQ29kZUJ1aWxkIHByb2plY3QgZG9lcyBub3QgdXNlIGFuIEFXUyBLTVMga2V5IGZvciBlbmNyeXB0aW9uLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJVc2luZyBhbiBBV1MgS01TIGtleSBoZWxwcyBmb2xsb3cgdGhlIHN0YW5kYXJkIHNlY3VyaXR5IGFkdmljZSBvZiBncmFudGluZyBsZWFzdCBwcml2aWxlZ2UgdG8gb2JqZWN0cyBnZW5lcmF0ZWQgYnkgdGhlIHByb2plY3QuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmNvZGVidWlsZC5Db2RlQnVpbGRQcm9qZWN0S01TRW5jcnlwdGVkQXJ0aWZhY3RzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQ29kZUJ1aWxkIHByb2plY3QgZG9lcyBub3QgdXNlIGltYWdlcyBwcm92aWRlZCBieSB0aGUgQ29kZUJ1aWxkIHNlcnZpY2Ugb3IgaGF2ZSBhIGNkay1uYWcgc3VwcHJlc3Npb24gcnVsZSBleHBsYWluaW5nIHRoZSBuZWVkIGZvciBhIGN1c3RvbSBpbWFnZS5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiRXhwbGFpbmluZyBkaWZmZXJlbmNlcy9lZGl0cyB0byBEb2NrZXIgaW1hZ2VzIGhlbHBzIG9wZXJhdG9ycyBiZXR0ZXIgdW5kZXJzdGFuZCBzeXN0ZW0gZGVwZW5kZW5jaWVzLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5jb2RlYnVpbGQuQ29kZUJ1aWxkUHJvamVjdE1hbmFnZWRJbWFnZXMsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBDb2RlQnVpbGQgcHJvamVjdCBoYXMgcHJpdmlsZWdlZCBtb2RlIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlByaXZpbGVnZWQgZ3JhbnRzIGVsZXZhdGVkIHJpZ2h0cyB0byB0aGUgc3lzdGVtLCB3aGljaCBpbnRyb2R1Y2VzIGFkZGl0aW9uYWwgcmlzay4gUHJpdmlsZWdlZCBtb2RlIHNob3VsZCBvbmx5IGJlIHNldCB0byB0cnVlIG9ubHkgaWYgdGhlIGJ1aWxkIHByb2plY3QgaXMgdXNlZCB0byBidWlsZCBEb2NrZXIgaW1hZ2VzLiBPdGhlcndpc2UsIGEgYnVpbGQgdGhhdCBhdHRlbXB0cyB0byBpbnRlcmFjdCB3aXRoIHRoZSBEb2NrZXIgZGFlbW9uIGZhaWxzLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5jb2RlYnVpbGQuQ29kZUJ1aWxkUHJvamVjdFByaXZpbGVnZWRNb2RlRGlzYWJsZWQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBDb2duaXRvIHVzZXIgcG9vbCBkb2VzIG5vdCBoYXZlIEFkdmFuY2VkU2VjdXJpdHlNb2RlIHNldCB0byBFTkZPUkNFRC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiQWR2YW5jZWQgc2VjdXJpdHkgZmVhdHVyZXMgZW5hYmxlIHRoZSBzeXN0ZW0gdG8gZGV0ZWN0IGFuZCBhY3QgdXBvbiBtYWxpY2lvdXMgc2lnbi1pbiBhdHRlbXB0cy5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuY29nbml0by5Db2duaXRvVXNlclBvb2xBZHZhbmNlZFNlY3VyaXR5TW9kZUVuZm9yY2VkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQVBJIEdhdGV3YXkgbWV0aG9kIGRvZXMgbm90IHVzZSBhIENvZ25pdG8gdXNlciBwb29sIGF1dGhvcml6ZXIuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkFQSSBHYXRld2F5IHZhbGlkYXRlcyB0aGUgdG9rZW5zIGZyb20gYSBzdWNjZXNzZnVsIHVzZXIgcG9vbCBhdXRoZW50aWNhdGlvbiwgYW5kIHVzZXMgdGhlbSB0byBncmFudCB5b3VyIHVzZXJzIGFjY2VzcyB0byByZXNvdXJjZXMgaW5jbHVkaW5nIExhbWJkYSBmdW5jdGlvbnMsIG9yIHlvdXIgb3duIEFQSS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuY29nbml0by5Db2duaXRvVXNlclBvb2xBUElHV0F1dGhvcml6ZXIsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBDb2duaXRvIHVzZXIgcG9vbCBpcyBub3QgY29uZmlndXJlZCB0byByZXF1aXJlIE1GQS5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiTXVsdGktZmFjdG9yIGF1dGhlbnRpY2F0aW9uIChNRkEpIGluY3JlYXNlcyBzZWN1cml0eSBmb3IgdGhlIGFwcGxpY2F0aW9uIGJ5IGFkZGluZyBhbm90aGVyIGF1dGhlbnRpY2F0aW9uIG1ldGhvZCwgYW5kIG5vdCByZWx5aW5nIHNvbGVseSBvbiB1c2VyIG5hbWUgYW5kIHBhc3N3b3JkLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5jb2duaXRvLkNvZ25pdG9Vc2VyUG9vbE1GQSxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIENvZ25pdG8gaWRlbnRpdHkgcG9vbCBhbGxvd3MgZm9yIHVuYXV0aGVudGljYXRlZCBsb2dpbnMgYW5kIGRvZXMgbm90IGhhdmUgYSBjZGstbmFnIHJ1bGUgc3VwcHJlc3Npb24gd2l0aCBhIHJlYXNvbi5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiQXBwbGljYXRpb25zIGRvIG5vdCB3YXJyYW50IHVuYXV0aGVudGljYXRlZCBndWVzdCBhY2Nlc3MgaW4gbWFueSBjYXNlcy4gTWV0YWRhdGEgZXhwbGFpbmluZyB0aGUgdXNlIGNhc2UgYWxsb3dzIGZvciB0cmFuc3BhcmVuY3kgdG8gb3BlcmF0b3JzLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciB3aG8gaGFzIGRpc2NvdmVyZWQgYSBwcm90b3R5cGUgZW5kcG9pbnQgd2l0aCBDb2duaXRvIHVuYXV0aGVudGljYXRlZCBsb2dpbnMgY2FuIGRvZXMgbm90IG5lZWQgdG8gcHJvdmlkZSBjcmVkZW50aWFscyB0byBpbnRlcmFjdCB3aXRoIHRoZSBlbmRwb2ludCwgd2hpY2ggbWF5IGxlYWQgdG8gcmVjb25uYWlzc2FuY2UgYW5kIGludHJ1c2lvbiBhY3Rpdml0aWVzIGJlaW5nIHBlcmZvcm1lZCBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhbmQgcmVzb3VyY2UgYXNzZXRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5jb2duaXRvLkNvZ25pdG9Vc2VyUG9vbE5vVW5hdXRoZW50aWNhdGVkTG9naW5zLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQ29nbml0byB1c2VyIHBvb2wgZG9lcyBub3QgaGF2ZSBhIHBhc3N3b3JkIHBvbGljeSB0aGF0IG1pbmltYWxseSBzcGVjaWZ5IGEgcGFzc3dvcmQgbGVuZ3RoIG9mIGF0IGxlYXN0IDggY2hhcmFjdGVycywgYXMgd2VsbCBhcyByZXF1aXJpbmcgdXBwZXJjYXNlLCBudW1lcmljLCBhbmQgc3BlY2lhbCBjaGFyYWN0ZXJzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJTdHJvbmcgcGFzc3dvcmQgcG9saWNpZXMgaW5jcmVhc2Ugc3lzdGVtIHNlY3VyaXR5IGJ5IGVuY291cmFnaW5nIHVzZXJzIHRvIGNyZWF0ZSByZWxpYWJsZSBhbmQgc2VjdXJlIHBhc3N3b3Jkcy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQW4gYWN0b3Igd2hvIGhhcyBkaXNjb3ZlcmVkIGEgcHJvdG90eXBlIGVuZHBvaW50IHdpdGggQ29nbml0byBhdXRoZW50aWNhdGVkIGxvZ2lucyBjYW4gcGVyZm9ybSBhIGRpY3Rpb25hcnkgb3IgYnJ1dGUgZm9yY2UgYXR0YWNrIHRvIGF1dGhlbnRpY2F0ZSBhcyBhbiBhdXRob3JpemVkIENvZ25pdG8gdXNlciwgd2hpY2ggbWF5IGxlYWQgdG8gdGhlIGFjdG9yIGJlaW5nIGFibGUgdG8gZG8gYW55dGhpbmcgdGhlIGFzc29jaWF0ZWQgQ29nbml0byB1c2VyIGlzIGF1dGhvcmlzZWQgdG8gZG8gcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLmNvZ25pdG8uQ29nbml0b1VzZXJQb29sU3Ryb25nUGFzc3dvcmRQb2xpY3ksXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBETVMgcmVwbGljYXRpb24gaW5zdGFuY2UgaXMgcHVibGljLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJETVMgcmVwbGljYXRpb24gaW5zdGFuY2VzIGNhbiBjb250YWluIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbiBhbmQgYWNjZXNzIGNvbnRyb2wgaXMgcmVxdWlyZWQgZm9yIHN1Y2ggYWNjb3VudHMuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIGNhbiBkaXNjb3ZlciBETVMgaW5zdGFuY2VzIHRoYXQgaGF2ZSBwdWJsaWMgSVAgYWRkcmVzc2VzLCB3aGljaCBtYXkgbGVhZCB0byByZWNvbm5haXNzYW5jZSBhbmQgaW50cnVzaW9uIGFjdGl2aXRpZXMgYmVpbmcgcGVyZm9ybWVkIGFnYWluc3QgdGhlIGV4cG9zZWQgYXR0YWNrIHN1cmZhY2UgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLmRtcy5ETVNSZXBsaWNhdGlvbk5vdFB1YmxpYyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIERvY3VtZW50IERCIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiRW5jcnlwdGluZyBkYXRhLWF0LXJlc3QgcHJvdGVjdHMgZGF0YSBjb25maWRlbnRpYWxpdHkgYW5kIHByZXZlbnRzIHVuYXV0aG9yaXplZCB1c2VycyBmcm9tIGFjY2Vzc2luZyBzZW5zaXRpdmUgaW5mb3JtYXRpb24uXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmRvY3VtZW50ZGIuRG9jdW1lbnREQkNsdXN0ZXJFbmNyeXB0aW9uQXRSZXN0LFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRG9jdW1lbnQgREIgY2x1c3RlciB1c2VzIHRoZSBkZWZhdWx0IGVuZHBvaW50IHBvcnQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlBvcnQgb2JmdXNjYXRpb24gKHVzaW5nIGEgbm9uIGRlZmF1bHQgZW5kcG9pbnQgcG9ydCkgYWRkcyBhbiBhZGRpdGlvbmFsIGxheWVyIG9mIGRlZmVuc2UgYWdhaW5zdCBub24tdGFyZ2V0ZWQgYXR0YWNrcyAoaS5lLiBNb25nb0RCIHBvcnQgMjcwMTcpLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5kb2N1bWVudGRiLkRvY3VtZW50REJDbHVzdGVyTm9uRGVmYXVsdFBvcnQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBEb2N1bWVudCBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgdGhlIHVzZXJuYW1lIGFuZCBwYXNzd29yZCBzdG9yZWQgaW4gU2VjcmV0cyBNYW5hZ2VyLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJTZWNyZXRzIE1hbmFnZXIgZW5hYmxlcyBvcGVyYXRvcnMgdG8gcmVwbGFjZSBoYXJkY29kZWQgY3JlZGVudGlhbHMgaW4geW91ciBjb2RlLCBpbmNsdWRpbmcgcGFzc3dvcmRzLCB3aXRoIGFuIEFQSSBjYWxsIHRvIFNlY3JldHMgTWFuYWdlciB0byByZXRyaWV2ZSB0aGUgc2VjcmV0IHByb2dyYW1tYXRpY2FsbHkuIFRoaXMgaGVscHMgZW5zdXJlIHRoZSBzZWNyZXQgY2FuJ3QgYmUgY29tcHJvbWlzZWQgYnkgc29tZW9uZSBleGFtaW5pbmcgc3lzdGVtIGNvZGUsIGJlY2F1c2UgdGhlIHNlY3JldCBubyBsb25nZXIgZXhpc3RzIGluIHRoZSBjb2RlLiBBbHNvLCBvcGVyYXRvcnMgY2FuIGNvbmZpZ3VyZSBTZWNyZXRzIE1hbmFnZXIgdG8gYXV0b21hdGljYWxseSByb3RhdGUgdGhlIHNlY3JldCBmb3IgeW91IGFjY29yZGluZyB0byBhIHNwZWNpZmllZCBzY2hlZHVsZS4gVGhpcyBlbmFibGVzIHlvdSB0byByZXBsYWNlIGxvbmctdGVybSBzZWNyZXRzIHdpdGggc2hvcnQtdGVybSBvbmVzLCBzaWduaWZpY2FudGx5IHJlZHVjaW5nIHRoZSByaXNrIG9mIGNvbXByb21pc2UuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEFuIGFjdG9yIHdobyBjYW4gdmlldyB0aGUgRG9jdW1lbnREQiBjb25maWd1cmF0aW9uIGNhbiBvYnRhaW4gdGhlIHVzZXJuYW1lIGFuZCBwYXNzd29yZCBmb3IgdGhlIERvY3VtZW50REIgY2x1c3Rlciwgd2hpY2ggbWF5IGxlYWQgdG8gdGhlIGFjdG9yIGJlaW5nIGFibGUgdG8gYWNjZXNzIGFueXRoaW5nIHRoZSBhc3NvY2lhdGVkIERvY3VtZW50REIgdXNlciBpcyBhdXRob3Jpc2VkIHRvIGRvIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhc3NldHMgaG9zdGVkIG9uIHRoZSBEb2N1bWVudERCIGNsdXN0ZXIgZm9yIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLmRvY3VtZW50ZGIuRG9jdW1lbnREQkNyZWRlbnRpYWxzSW5TZWNyZXRzTWFuYWdlcixcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIERBWCBjbHVzdGVyIGRvZXMgbm90IGhhdmUgc2VydmVyLXNpZGUgZW5jcnlwdGlvbiBlbmFibGVkLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJEYXRhIGluIGNhY2hlLCBjb25maWd1cmF0aW9uIGRhdGEgYW5kIGxvZyBmaWxlcyBzaG91bGQgYmUgZW5jcnlwdGVkIHVzaW5nIFNlcnZlci1TaWRlIEVuY3J5cHRpb24gaW4gb3JkZXIgdG8gcHJvdGVjdCBmcm9tIHVuYXV0aG9yaXplZCBhY2Nlc3MgdG8gdGhlIHVuZGVybHlpbmcgc3RvcmFnZS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZHluYW1vZGIuREFYRW5jcnlwdGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRUJTIHZvbHVtZSBoYXMgZW5jcnlwdGlvbiBkaXNhYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiV2l0aCBFQlMgZW5jcnlwdGlvbiwgeW91IGFyZW4ndCByZXF1aXJlZCB0byBidWlsZCwgbWFpbnRhaW4sIGFuZCBzZWN1cmUgeW91ciBvd24ga2V5IG1hbmFnZW1lbnQgaW5mcmFzdHJ1Y3R1cmUuIEVCUyBlbmNyeXB0aW9uIHVzZXMgS01TIGtleXMgd2hlbiBjcmVhdGluZyBlbmNyeXB0ZWQgdm9sdW1lcyBhbmQgc25hcHNob3RzLiBUaGlzIGhlbHBzIHByb3RlY3QgZGF0YSBhdCByZXN0LlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5lYzIuRUMyRUJTVm9sdW1lRW5jcnlwdGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRUMyIGluc3RhbmNlIGlzIGFzc29jaWF0ZWQgd2l0aCBhIHB1YmxpYyBJUCBhZGRyZXNzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJBbWF6b24gRUMyIGluc3RhbmNlcyBjYW4gY29udGFpbiBzZW5zaXRpdmUgaW5mb3JtYXRpb24gYW5kIGFjY2VzcyBjb250cm9sIGlzIHJlcXVpcmVkIGZvciBzdWNoIHJlc291cmNlcy5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZWMyLkVDMkluc3RhbmNlTm9QdWJsaWNJcCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEVDMiBpbnN0YW5jZSBkb2VzIG5vdCBoYXZlIGFuIGluc3RhbmNlIHByb2ZpbGUgYXR0YWNoZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVDMiBpbnN0YW5jZSBwcm9maWxlcyBwYXNzIGFuIElBTSByb2xlIHRvIGFuIEVDMiBpbnN0YW5jZS4gQXR0YWNoaW5nIGFuIGluc3RhbmNlIHByb2ZpbGUgdG8geW91ciBpbnN0YW5jZXMgY2FuIGFzc2lzdCB3aXRoIGxlYXN0IHByaXZpbGVnZSBhbmQgcGVybWlzc2lvbnMgbWFuYWdlbWVudC5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZWMyLkVDMkluc3RhbmNlUHJvZmlsZUF0dGFjaGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRUMyIGluc3RhbmNlIGlzIG5vdCB3aXRoaW4gYSBWUEMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkJlY2F1c2Ugb2YgdGhlaXIgbG9naWNhbCBpc29sYXRpb24sIGRvbWFpbnMgdGhhdCByZXNpZGUgd2l0aGluIGFuIEFtYXpvbiBWUEMgaGF2ZSBhbiBleHRyYSBsYXllciBvZiBzZWN1cml0eSB3aGVuIGNvbXBhcmVkIHRvIGRvbWFpbnMgdGhhdCB1c2UgcHVibGljIGVuZHBvaW50cy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3IgY2FuIGRpc2NvdmVyIEVDMiBpbnN0YW5jZXMgdGhhdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuZWMyLkVDMkluc3RhbmNlc0luVlBDLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRUMyIGluc3RhbmNlIGFsbG93cyB1bnJlc3RyaWN0ZWQgaW5ib3VuZCBJUHY0IFRDUCB0cmFmZmljIG9uIG9uZSBvciBtb3JlIGNvbW1vbiBwb3J0cyAoYnkgZGVmYXVsdCB0aGVzZSBwb3J0cyBpbmNsdWRlIDIwLCAyMSwgMzM4OSwgMzMwOSwgMzMwNiwgNDMzMykuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIk5vdCByZXN0cmljdGluZyBhY2Nlc3MgdG8gcG9ydHMgdG8gdHJ1c3RlZCBzb3VyY2VzIGNhbiBsZWFkIHRvIGF0dGFja3MgYWdhaW5zdCB0aGUgYXZhaWxhYmlsaXR5LCBpbnRlZ3JpdHkgYW5kIGNvbmZpZGVudGlhbGl0eSBvZiBzeXN0ZW1zLiBCeSBkZWZhdWx0LCBjb21tb24gcG9ydHMgd2hpY2ggc2hvdWxkIGJlIHJlc3RyaWN0ZWQgaW5jbHVkZSBwb3J0IG51bWJlcnMgMjAsIDIxLCAzMzg5LCAzMzA2LCBhbmQgNDMzMy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3IgY2FuIGRpc2NvdmVyIGV4cG9zZWQgc2VydmljZXMgKGUuZy4gVGVsbmV0LCBTU0gsIFJEUywgTXlTUUwpIHVzaW5nIHRoZWlyIGNvbW1vbiBwb3J0IG51bWJlcnMsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuZWMyLkVDMlJlc3RyaWN0ZWRDb21tb25Qb3J0cyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFNlY3VyaXR5IEdyb3VwIGFsbG93cyBmb3IgMC4wLjAuMC8wIG9yIDo6LzAgaW5ib3VuZCBhY2Nlc3MuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkxhcmdlIHBvcnQgcmFuZ2VzLCB3aGVuIG9wZW4sIGV4cG9zZSBpbnN0YW5jZXMgdG8gdW53YW50ZWQgYXR0YWNrcy4gTW9yZSB0aGFuIHRoYXQsIHRoZXkgbWFrZSB0cmFjZWFiaWxpdHkgb2YgdnVsbmVyYWJpbGl0aWVzIHZlcnkgZGlmZmljdWx0LiBGb3IgaW5zdGFuY2UsIHlvdXIgd2ViIHNlcnZlcnMgbWF5IG9ubHkgcmVxdWlyZSA4MCBhbmQgNDQzIHBvcnRzIHRvIGJlIG9wZW4sIGJ1dCBub3QgYWxsLiBPbmUgb2YgdGhlIG1vc3QgY29tbW9uIG1pc3Rha2VzIG9ic2VydmVkIGlzIHdoZW4gYWxsIHBvcnRzIGZvciAwLjAuMC4wLzAgcmFuZ2UgYXJlIG9wZW4gaW4gYSBydXNoIHRvIGFjY2VzcyB0aGUgaW5zdGFuY2UuIEVDMiBpbnN0YW5jZXMgbXVzdCBleHBvc2Ugb25seSB0byB0aG9zZSBwb3J0cyBlbmFibGVkIG9uIHRoZSBjb3JyZXNwb25kaW5nIHNlY3VyaXR5IGdyb3VwIGxldmVsLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciBjYW4gZGlzY292ZXIgRUMyIGluc3RhbmNlcyB0aGF0IGhhdmUgcHVibGljIElQIGFkZHJlc3NlcyBhbmQgYWxsb3cgaW5ncmVzcyB0byBhbGwgaW50ZXJuZXQgYWRkcmVzcyBvciBtb3ZlIGxhdGVyYWxseSB0byBub24tcHVibGljIEVDMiBpbnN0YW5jZXMsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuZWMyLkVDMlJlc3RyaWN0ZWRJbmJvdW5kLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgU2VjdXJpdHkgR3JvdXAgYWxsb3dzIHVucmVzdHJpY3RlZCBTU0ggYWNjZXNzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJOb3QgYWxsb3dpbmcgaW5ncmVzcyAob3IgcmVtb3RlKSB0cmFmZmljIGZyb20gMC4wLjAuMC8wIG9yIDo6LzAgdG8gcG9ydCAyMiBvbiB5b3VyIHJlc291cmNlcyBoZWxwcyB0byByZXN0cmljdCByZW1vdGUgYWNjZXNzLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciBjYW4gZGlzY292ZXIgRUMyIGluc3RhbmNlcyB0aGF0IGhhdmUgcHVibGljIElQIGFkZHJlc3NlcyBhbmQgYWxsb3cgaW5ncmVzcyB0byBhbGwgaW50ZXJuZXQgYWRkcmVzcyB0byBTU0ggb3IgbW92ZSBsYXRlcmFsbHkgdG8gbm9uLXB1YmxpYyBFQzIgaW5zdGFuY2VzIHZpYSBTU0gsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuZWMyLkVDMlJlc3RyaWN0ZWRTU0gsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBTZWN1cml0eSBHcm91cCBkb2VzIG5vdCBoYXZlIGEgZGVzY3JpcHRpb24uXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkRlc2NyaXB0aW9ucyBoZWxwIHNpbXBsaWZ5IG9wZXJhdGlvbnMgYW5kIHJlbW92ZSBhbnkgb3Bwb3J0dW5pdGllcyBmb3Igb3BlcmF0b3IgZXJyb3JzLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5lYzIuRUMyU2VjdXJpdHlHcm91cERlc2NyaXB0aW9uLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRUNSIFJlcG9zaXRvcnkgYWxsb3dzIG9wZW4gYWNjZXNzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJSZW1vdmluZyAqIHByaW5jaXBhbHMgaW4gYW4gRUNSIFJlcG9zaXRvcnkgaGVscHMgcHJvdGVjdCBhZ2FpbnN0IHVuYXV0aG9yaXplZCBhY2Nlc3MuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIHdobyBoYXMgZGlzY292ZXJlZCBhIEVDUiByZXBvc2l0b3J5IGNhbiBhY2Nlc3MgdGhlIGNvbnRhaW5lciBpbWFnZXMgaG9zdGVkIHdpdGhpbiB0aGUgcmVwb3NpdG9yeSwgd2hpY2ggbWF5IGxlYWQgdG8gaW5mb3JtYXRpb24gZGlzY2xvc3VyZSB0aGF0IGFpZHMgaW4gdGhlIGludHJ1c2lvbiBhY3Rpdml0aWVzIGJlaW5nIHN1Y2Nlc3NmdWwgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuZWNyLkVDUk9wZW5BY2Nlc3MsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIk9uZSBvciBtb3JlIGNvbnRhaW5lcnMgaW4gdGhlIEVDUyBUYXNrIERlZmluaXRpb24gZG8gbm90IGhhdmUgY29udGFpbmVyIGxvZ2dpbmcgZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiQ29udGFpbmVyIGxvZ2dpbmcgYWxsb3dzIG9wZXJhdG9ycyB0byB2aWV3IGFuZCBhZ2dyZWdhdGUgdGhlIGxvZ3MgZnJvbSB0aGUgY29udGFpbmVyLiBDb250YWluZXJzIHNob3VsZCB1c2UgdGhlICdhd3Nsb2dzJyBkcml2ZXIgYXQgYSBtaW5pbXVtLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5lY3MuRUNTVGFza0RlZmluaXRpb25Db250YWluZXJMb2dnaW5nLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRUNTIFRhc2sgRGVmaW5pdGlvbiBpbmNsdWRlcyBhIGNvbnRhaW5lciBkZWZpbml0aW9uIHRoYXQgZGlyZWN0bHkgc3BlY2lmaWVzIGVudmlyb25tZW50IHZhcmlhYmxlcy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiVXNlIHNlY3JldHMgdG8gaW5qZWN0IGVudmlyb25tZW50IHZhcmlhYmxlcyBkdXJpbmcgY29udGFpbmVyIHN0YXJ0dXAgZnJvbSBBV1MgU3lzdGVtcyBNYW5hZ2VyIFBhcmFtZXRlciBTdG9yZSBvciBTZWNyZXRzIE1hbmFnZXIgaW5zdGVhZCBvZiBkaXJlY3RseSBzcGVjaWZ5aW5nIHBsYWludGV4dCBlbnZpcm9ubWVudCB2YXJpYWJsZXMuIFVwZGF0ZXMgdG8gZGlyZWN0IGVudmlyb25tZW50IHZhcmlhYmxlcyByZXF1aXJlIG9wZXJhdG9ycyB0byBjaGFuZ2UgdGFzayBkZWZpbml0aW9ucyBhbmQgcGVyZm9ybSBuZXcgZGVwbG95bWVudHMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmVjcy5FQ1NUYXNrRGVmaW5pdGlvbk5vRW52aXJvbm1lbnRWYXJpYWJsZXMsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBFRlMgZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYW5kIHRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgZm9yIHlvdXIgQW1hem9uIEVsYXN0aWMgRmlsZSBTeXN0ZW0gKEVGUykuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmVmcy5FRlNFbmNyeXB0ZWQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBFS1MgQ2x1c3RlciBkb2VzIG5vdCBwdWJsaXNoICdhcGknLCAnYXVkaXQnLCAnYXV0aGVudGljYXRvciwgJ2NvbnRyb2xsZXJNYW5hZ2VyJywgYW5kICdzY2hlZHVsZXInIGNvbnRyb2wgcGxhbmUgbG9ncy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiRUtTIGNvbnRyb2wgcGxhbmUgbG9nZ2luZyBwcm92aWRlcyBhdWRpdCBhbmQgZGlhZ25vc3RpYyBsb2dzIGRpcmVjdGx5IGZyb20gdGhlIEFtYXpvbiBFS1MgY29udHJvbCBwbGFuZSB0byBDbG91ZFdhdGNoIExvZ3MgaW4geW91ciBhY2NvdW50LiBUaGVzZSBsb2dzIG1ha2UgaXQgZWFzeSBmb3IgeW91IHRvIHNlY3VyZSBhbmQgcnVuIHlvdXIgY2x1c3RlcnMuIFRoaXMgaXMgYSBncmFudWxhciBydWxlIHRoYXQgcmV0dXJucyBpbmRpdmlkdWFsIGZpbmRpbmdzIHRoYXQgY2FuIGJlIHN1cHByZXNzZWQgd2l0aCBhcHBsaWVzVG8uIFRoZSBmaW5kaW5ncyBhcmUgaW4gdGhlIGZvcm1hdCBMb2dFeHBvcnQ6Ojxsb2c+IGZvciBleHBvcnRlZCBsb2dzLiBFeGFtcGxlOiBhcHBsaWVzVG86IFsnTG9nRXhwb3J0OjphdXRoZW50aWNhdGUnXS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZWtzLkVLU0NsdXN0ZXJDb250cm9sUGxhbmVMb2dzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRUtTIGNsdXN0ZXIncyBLdWJlcm5ldGVzIEFQSSBzZXJ2ZXIgZW5kcG9pbnQgaGFzIHB1YmxpYyBhY2Nlc3MgZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiQSBjbHVzdGVyJ3MgS3ViZXJuZXRlcyBBUEkgc2VydmVyIGVuZHBvaW50IHNob3VsZCBub3QgYmUgcHVibGljbHkgYWNjZXNzaWJsZSBmcm9tIHRoZSBJbnRlcm5ldCBpbiBvcmRlciB0byBhdm9pZCBleHBvc2luZyBwcml2YXRlIGRhdGEgYW5kIG1pbmltaXppbmcgc2VjdXJpdHkgcmlza3MuIFRoZSBBUEkgc2VydmVyIGVuZHBvaW50cyBzaG91bGQgb25seSBiZSBhY2Nlc3NpYmxlIGZyb20gd2l0aGluIGEgQVdTIFZpcnR1YWwgUHJpdmF0ZSBDbG91ZCAoVlBDKS5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3Igd2hvIGhhcyBkaXNjb3ZlcmVkIGEgRUtTIGNsdXN0ZXIgS3ViZXJuZXRlcyBBUEkgc2VydmVyIGVuZHBvaW50IGNhbiBwZXJmb3JtIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlLCB3aGljaCBtYXkgbGVhZCB0byBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuZWtzLkVLU0NsdXN0ZXJOb0VuZHBvaW50UHVibGljQWNjZXNzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRWxhc3RpQ2FjaGUgY2x1c3RlciBpcyBub3QgcHJvdmlzaW9uZWQgaW4gYSBWUEMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlByb3Zpc2lvbmluZyB0aGUgY2x1c3RlciB3aXRoaW4gYSBWUEMgYWxsb3dzIGZvciBiZXR0ZXIgZmxleGliaWxpdHkgYW5kIGNvbnRyb2wgb3ZlciB0aGUgY2FjaGUgY2x1c3RlcnMgc2VjdXJpdHksIGF2YWlsYWJpbGl0eSwgdHJhZmZpYyByb3V0aW5nIGFuZCBtb3JlLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciBjYW4gZGlzY292ZXIgdGhlIEVsYXN0aUNhY2hlIGNsdXN0ZXIgdGhhdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgd2l0aCB0aGUgY2x1c3RlciB1c2VkIHdpdGhpbiB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5lbGFzdGljYWNoZS5FbGFzdGlDYWNoZUNsdXN0ZXJJblZQQyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEVsYXN0aUNhY2hlIGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBlbmRwb2ludCBwb3J0LlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJQb3J0IG9iZnVzY2F0aW9uICh1c2luZyBhIG5vbiBkZWZhdWx0IGVuZHBvaW50IHBvcnQpIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MgKGkuZS4gUmVkaXMgcG9ydCA2Mzc5IGFuZCBNZW1jYWNoZWQgcG9ydCAxMTIxMSkuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmVsYXN0aWNhY2hlLkVsYXN0aUNhY2hlQ2x1c3Rlck5vbkRlZmF1bHRQb3J0LFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgRWxhc3RpQ2FjaGUgUmVkaXMgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGJvdGggZW5jcnlwdGlvbiBpbiB0cmFuc2l0IGFuZCBhdCByZXN0IGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVuY3J5cHRpb24gaW4gdHJhbnNpdCBoZWxwcyBzZWN1cmUgY29tbXVuaWNhdGlvbnMgdG8gdGhlIGNsdXN0ZXIuIEVuY3J5cHRpb24gYXQgcmVzdCBoZWxwcyBwcm90ZWN0IGRhdGEgYXQgcmVzdCBmcm9tIHVuYXV0aG9yaXplZCBhY2Nlc3MuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmVsYXN0aWNhY2hlLkVsYXN0aUNhY2hlUmVkaXNDbHVzdGVyRW5jcnlwdGlvbixcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEVsYXN0aUNhY2hlIFJlZGlzIGNsdXN0ZXIgZG9lcyBub3QgdXNlIFJlZGlzIEFVVEggZm9yIHVzZXIgYXV0aGVudGljYXRpb24uXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlJlZGlzIGF1dGhlbnRpY2F0aW9uIHRva2VucyBlbmFibGUgUmVkaXMgdG8gcmVxdWlyZSBhIHRva2VuIChwYXNzd29yZCkgYmVmb3JlIGFsbG93aW5nIGNsaWVudHMgdG8gZXhlY3V0ZSBjb21tYW5kcywgdGhlcmVieSBpbXByb3ZpbmcgZGF0YSBzZWN1cml0eS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZWxhc3RpY2FjaGUuRWxhc3RpQ2FjaGVSZWRpc0NsdXN0ZXJSZWRpc0F1dGgsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBFbGFzdGljIEJlYW5zdGFsayBlbnZpcm9ubWVudCBkb2VzIG5vdCB1cGxvYWQgRUMyIEluc3RhbmNlIGxvZ3MgdG8gUzMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkJlYW5zdGFsayBlbnZpcm9ubWVudCBsb2dzIHNob3VsZCBiZSByZXRhaW5lZCBhbmQgdXBsb2FkZWQgdG8gQW1hem9uIFMzIGluIG9yZGVyIHRvIGtlZXAgdGhlIGxvZ2dpbmcgZGF0YSBmb3IgZnV0dXJlIGF1ZGl0cywgaGlzdG9yaWNhbCBwdXJwb3NlcyBvciB0byB0cmFjayBhbmQgYW5hbHl6ZSB0aGUgRUIgYXBwbGljYXRpb24gZW52aXJvbm1lbnQgYmVoYXZpb3IgZm9yIGEgbG9uZyBwZXJpb2Qgb2YgdGltZS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZWxhc3RpY2JlYW5zdGFsay5FbGFzdGljQmVhbnN0YWxrRUMySW5zdGFuY2VMb2dzVG9TMyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEVsYXN0aWMgQmVhbnN0YWxrIGVudmlyb25tZW50IGRvZXMgbm90IGhhdmUgbWFuYWdlZCB1cGRhdGVzIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVuYWJsZSBtYW5hZ2VkIHBsYXRmb3JtIHVwZGF0ZXMgZm9yIGJlYW5zdGFsayBlbnZpcm9ubWVudHMgaW4gb3JkZXIgdG8gcmVjZWl2ZSBidWcgZml4ZXMsIHNvZnR3YXJlIHVwZGF0ZXMgYW5kIG5ldyBmZWF0dXJlcy4gTWFuYWdlZCBwbGF0Zm9ybSB1cGRhdGVzIHBlcmZvcm0gaW1tdXRhYmxlIGVudmlyb25tZW50IHVwZGF0ZXMuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEFuIGFjdG9yIHdpdGggYSBuZXR3b3JrIHBhdGggdG8gdGhlIEVsYXN0aWMgQmVhbnN0YWxrIGVudmlyb25tZW50IGNhbiBhdHRlbXB0IHRvIHRha2UgYWR2YW50YWdlIG9mIGEga25vd24gdnVsbmVyYWJpbGl0eSBpbiBhIHBsYXRmb3JtIGNvbXBvbmVudCB1c2VkIGJ5IEVsYXN0aWMgQmVhbnN0YWxrLCB3aGljaCBtYXkgbGVhZCB0byB1bmtub3duIGltcGFjdHMgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLmVsYXN0aWNiZWFuc3RhbGsuRWxhc3RpY0JlYW5zdGFsa01hbmFnZWRVcGRhdGVzRW5hYmxlZCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEVsYXN0aWMgQmVhbnN0YWxrIGVudmlyb25tZW50IGlzIG5vdCBjb25maWd1cmVkIHRvIHVzZSBhIHNwZWNpZmljIFZQQy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiVXNlIGEgbm9uLWRlZmF1bHQgVlBDIGluIG9yZGVyIHRvIHNlcGFyYXRlIHlvdXIgZW52aXJvbm1lbnQgZnJvbSBkZWZhdWx0IHJlc291cmNlcy5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZWxhc3RpY2JlYW5zdGFsay5FbGFzdGljQmVhbnN0YWxrVlBDU3BlY2lmaWVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgQUxCJ3MgSFRUUCBsaXN0ZW5lcnMgYXJlIG5vdCBjb25maWd1cmVkIHRvIHJlZGlyZWN0IHRvIEhUVFBTLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJUbyBoZWxwIHByb3RlY3QgZGF0YSBpbiB0cmFuc2l0LCBlbnN1cmUgdGhhdCB5b3VyIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIgYXV0b21hdGljYWxseSByZWRpcmVjdHMgdW5lbmNyeXB0ZWQgSFRUUCByZXF1ZXN0cyB0byBIVFRQUy4gQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QsIGVuYWJsZSBlbmNyeXB0aW9uIGluIHRyYW5zaXQgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuZWxiLkFMQkh0dHBUb0h0dHBzUmVkaXJlY3Rpb24sXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBDTEIgZG9lcyBub3QgcmVzdHJpY3QgaXRzIGxpc3RlbmVycyB0byBvbmx5IHRoZSBTU0wgYW5kIEhUVFBTIHByb3RvY29scy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiRW5zdXJlIHRoYXQgeW91ciBDbGFzc2ljIExvYWQgQmFsYW5jZXJzIChDTEJzKSBhcmUgY29uZmlndXJlZCB3aXRoIFNTTCBvciBIVFRQUyBsaXN0ZW5lcnMuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0LCBlbmFibGUgZW5jcnlwdGlvbiBpbiB0cmFuc2l0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmVsYi5FTEJUbHNIdHRwc0xpc3RlbmVyc09ubHksXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBFTVIgY2x1c3RlciBkb2VzIG5vdCBpbXBsZW1lbnQgYXV0aGVudGljYXRpb24gdmlhIGFuIEVDMiBLZXkgUGFpciBvciBLZXJiZXJvcy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiU1NIIGNsaWVudHMgY2FuIHVzZSBhbiBFQzIga2V5IHBhaXIgdG8gYXV0aGVudGljYXRlIHRvIGNsdXN0ZXIgaW5zdGFuY2VzLiBBbHRlcm5hdGl2ZWx5LCB3aXRoIEVNUiByZWxlYXNlIHZlcnNpb24gNS4xMC4wIG9yIGxhdGVyLCBzb2x1dGlvbnMgY2FuIGNvbmZpZ3VyZSBLZXJiZXJvcyB0byBhdXRoZW50aWNhdGUgdXNlcnMgYW5kIFNTSCBjb25uZWN0aW9ucyB0byB0aGUgbWFzdGVyIG5vZGUuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmVtci5FTVJBdXRoRUMyS2V5UGFpck9yS2VyYmVyb3MsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBFTVIgY2x1c3RlciBkb2VzIG5vdCB1c2UgYSBzZWN1cml0eSBjb25maWd1cmF0aW9uIHdpdGggZW5jcnlwdGlvbiBpbiB0cmFuc2l0IGVuYWJsZWQgYW5kIGNvbmZpZ3VyZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVNUiBDbHVzdGVycyBzaG91bGQgaGF2ZSBhIG1ldGhvZCBmb3IgZW5jcnlwdGluZyBkYXRhIGluIHRyYW5zaXQgdXNpbmcgVHJhbnNwb3J0IExheWVyIFNlY3VyaXR5IChUTFMpLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5lbXIuRU1SRW5jcnlwdGlvbkluVHJhbnNpdCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEVNUiBjbHVzdGVyIGRvZXMgbm90IHVzZSBhIHNlY3VyaXR5IGNvbmZpZ3VyYXRpb24gd2l0aCBsb2NhbCBkaXNrIGVuY3J5cHRpb24gZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiTG9jYWwgZGlzayBlbmNyeXB0aW9uIHVzZXMgYSBjb21iaW5hdGlvbiBvZiBvcGVuLXNvdXJjZSBIREZTIGVuY3J5cHRpb24gYW5kIExVS1MgZW5jcnlwdGlvbiB0byBzZWN1cmUgZGF0YSBhdCByZXN0LlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5lbXIuRU1STG9jYWxEaXNrRW5jcnlwdGlvbixcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIGV2ZW50IGJ1cyBwb2xpY3kgYWxsb3dzIGZvciBvcGVuIGFjY2Vzcy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgICdBbiBvcGVuIHBvbGljeSAoXCIqXCIgcHJpbmNpcGFsIHdpdGhvdXQgYSBjb25kaXRpb24pIGdyYW50cyBhbm9ueW1vdXMgYWNjZXNzIHRvIGFuIGV2ZW50IGJ1cy4gVXNlIGEgY29uZGl0aW9uIHRvIGxpbWl0IHRoZSBwZXJtaXNzaW9uIHRvIGFjY291bnRzIHRoYXQgZnVsZmlsbCBhIGNlcnRhaW4gcmVxdWlyZW1lbnQsIHN1Y2ggYXMgYmVpbmcgYSBtZW1iZXIgb2YgYSBjZXJ0YWluIEFXUyBvcmdhbml6YXRpb24uXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIHdobyBoYXMgZGlzY292ZXJlZCB0aGUgRXZlbnQgQnJpZGdlIGV2ZW50IGJ1cyAoZS5nLiBFbmRwb2ludCBJRCkgY2FuIHB1dCBhcmJpdHJhcnkgZXZlbnRzIG9udG8gdGhlIGJ1cywgd2hpY2ggbWF5IGxlYWQgdG8gd2hpY2ggY291bGQgYmUgcHJvY2Vzc2VkIGJ5IHRoZSBwcm90b3R5cGUgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGUnLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMuZXZlbnRicmlkZ2UuRXZlbnRCdXNPcGVuQWNjZXNzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgR2x1ZSBjcmF3bGVyIG9yIGpvYiBkb2VzIG5vdCB1c2UgYSBzZWN1cml0eSBjb25maWd1cmF0aW9uIHdpdGggQ2xvdWRXYXRjaCBMb2cgZW5jcnlwdGlvbiBlbmFibGVkLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJFbmFibGluZyBlbmNyeXB0aW9uIGF0IHJlc3QgaGVscHMgcHJldmVudCB1bmF1dGhvcml6ZWQgdXNlcnMgZnJvbSBnZXR0aW5nIGFjY2VzcyB0byB0aGUgbG9nZ2luZyBkYXRhIHB1Ymxpc2hlZCB0byBDbG91ZFdhdGNoIExvZ3MuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmdsdWUuR2x1ZUVuY3J5cHRlZENsb3VkV2F0Y2hMb2dzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgR2x1ZSBqb2IgZG9lcyBub3QgaGF2ZSB1c2UgYSBzZWN1cml0eSBjb25maWd1cmF0aW9uIHdpdGggam9iIGJvb2ttYXJrIGVuY3J5cHRpb24gZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiSm9iIGJvb2ttYXJrIGVuY3J5cHRpb24gZW5jcnlwdHMgYm9va21hcmsgZGF0YSBiZWZvcmUgaXQgaXMgc2VudCB0byBBbWF6b24gUzMgZm9yIHN0b3JhZ2UuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmdsdWUuR2x1ZUpvYkJvb2ttYXJrRW5jcnlwdGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgSUFNIHVzZXIsIHJvbGUsIG9yIGdyb3VwIHVzZXMgQVdTIG1hbmFnZWQgcG9saWNpZXMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkFuIEFXUyBtYW5hZ2VkIHBvbGljeSBpcyBhIHN0YW5kYWxvbmUgcG9saWN5IHRoYXQgaXMgY3JlYXRlZCBhbmQgYWRtaW5pc3RlcmVkIGJ5IEFXUy4gQ3VycmVudGx5LCBtYW55IEFXUyBtYW5hZ2VkIHBvbGljaWVzIGRvIG5vdCByZXN0cmljdCByZXNvdXJjZSBzY29wZS4gUmVwbGFjZSBBV1MgbWFuYWdlZCBwb2xpY2llcyB3aXRoIHN5c3RlbSBzcGVjaWZpYyAoY3VzdG9tZXIpIG1hbmFnZWQgcG9saWNpZXMuIFRoaXMgaXMgYSBncmFudWxhciBydWxlIHRoYXQgcmV0dXJucyBpbmRpdmlkdWFsIGZpbmRpbmdzIHRoYXQgY2FuIGJlIHN1cHByZXNzZWQgd2l0aCBhcHBsaWVzVG8uIFRoZSBmaW5kaW5ncyBhcmUgaW4gdGhlIGZvcm1hdCBQb2xpY3k6Ojxwb2xpY3k+IGZvciBBV1MgbWFuYWdlZCBwb2xpY2llcy4gRXhhbXBsZTogYXBwbGllc1RvOiBbJ1BvbGljeTo6YXJuOjxBV1M6OlBhcnRpdGlvbj46aWFtOjphd3M6cG9saWN5L2ZvbyddLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5pYW0uSUFNTm9NYW5hZ2VkUG9saWNpZXMsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBJQU0gZW50aXR5IGNvbnRhaW5zIHdpbGRjYXJkIHBlcm1pc3Npb25zIGFuZCBkb2VzIG5vdCBoYXZlIGEgY2RrLW5hZyBydWxlIHN1cHByZXNzaW9uIHdpdGggZXZpZGVuY2UgZm9yIHRob3NlIHBlcm1pc3Npb24uXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIk1ldGFkYXRhIGV4cGxhaW5pbmcgdGhlIGV2aWRlbmNlIChlLmcuIHZpYSBzdXBwb3J0aW5nIGxpbmtzKSBmb3Igd2lsZGNhcmQgcGVybWlzc2lvbnMgYWxsb3dzIGZvciB0cmFuc3BhcmVuY3kgdG8gb3BlcmF0b3JzLiBUaGlzIGlzIGEgZ3JhbnVsYXIgcnVsZSB0aGF0IHJldHVybnMgaW5kaXZpZHVhbCBmaW5kaW5ncyB0aGF0IGNhbiBiZSBzdXBwcmVzc2VkIHdpdGggYXBwbGllc1RvLiBUaGUgZmluZGluZ3MgYXJlIGluIHRoZSBmb3JtYXQgQWN0aW9uOjo8YWN0aW9uPiBmb3IgcG9saWN5IGFjdGlvbnMgYW5kIFJlc291cmNlOjo8cmVzb3VyY2U+IGZvciByZXNvdXJjZXMuIEV4YW1wbGU6IGFwcGxpZXNUbzogWydBY3Rpb246OnMzOionXS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuaWFtLklBTU5vV2lsZGNhcmRQZXJtaXNzaW9ucyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIElBTSBwb2xpY3kgZ3JhbnRzIGFkbWluIGFjY2VzcyAtIG1lYW5pbmcgdGhlIHBvbGljeSBhbGxvd3MgYSBwcmluY2lwYWwgdG8gcGVyZm9ybSB1bmxpbWl0ZWQgYWN0aW9ucyBvbiBhbnkgc2VydmljZVwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJBV1MgSWRlbnRpdHkgYW5kIEFjY2VzcyBNYW5hZ2VtZW50IChJQU0pIGNhbiBoZWxwIHlvdSBpbmNvcnBvcmF0ZSB0aGUgcHJpbmNpcGxlcyBvZiBsZWFzdCBwcml2aWxlZ2UgYW5kIHNlcGFyYXRpb24gb2YgZHV0aWVzIHdpdGggYWNjZXNzIHBlcm1pc3Npb25zIGFuZCBhdXRob3JpemF0aW9ucywgYnkgZW5zdXJpbmcgdGhhdCBJQU0gZ3JvdXBzIGhhdmUgYXQgbGVhc3Qgb25lIElBTSB1c2VyLiBQbGFjaW5nIElBTSB1c2VycyBpbiBncm91cHMgYmFzZWQgb24gdGhlaXIgYXNzb2NpYXRlZCBwZXJtaXNzaW9ucyBvciBqb2IgZnVuY3Rpb24gaXMgb25lIHdheSB0byBpbmNvcnBvcmF0ZSBsZWFzdCBwcml2aWxlZ2UuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIHdobyBoYXMgc3VjY2Vzc2Z1bGx5IG9idGFpbmVkIHZhbGlkIGtleXMgb3IgYSBzZXNzaW9uIGFzc29jaWF0ZWQgb2YgdGhlIElBTSBQcmluY2lwYWwgYXNzb2NpYXRlZCB3aXRoIHRoZSBJQU0gcG9saWN5IGNhbiBwZXJmb3JtIHVubGltaXRlZCBBV1MgYWN0aW9ucyBvbiBhbnkgQVdTIHNlcnZpY2Ugd2hpY2ggYXJlIGV4cG9zZWQgdmlhIHRoZSBBV1MgQVBJL01hbmFnZW1lbnQgQ29uc29sZS9DTEksIHdoaWNoIG1heSBsZWFkIHRvIGJyb2FkIGFuZCB1bmtub3duIGltcGFjdHMgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLmlhbS5JQU1Qb2xpY3lOb1N0YXRlbWVudHNXaXRoQWRtaW5BY2Nlc3MsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBJQU0gcG9saWN5IGdyYW50cyBmdWxsIGFjY2VzcyAtIG1lYW5pbmcgdGhlIHBvbGljeSBhbGxvd3MgdW5saW1pdGVkIGFjdGlvbnMgZm9yIGEgZ2l2ZW4gQVdTIHNlcnZpY2VcIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiRW5zdXJlIElBTSBBY3Rpb25zIGFyZSByZXN0cmljdGVkIHRvIG9ubHkgdGhvc2UgYWN0aW9ucyB0aGF0IGFyZSBuZWVkZWQuIEFsbG93aW5nIHVzZXJzIHRvIGhhdmUgbW9yZSBwcml2aWxlZ2VzIHRoYW4gbmVlZGVkIHRvIGNvbXBsZXRlIGEgdGFzayBtYXkgdmlvbGF0ZSB0aGUgcHJpbmNpcGxlIG9mIGxlYXN0IHByaXZpbGVnZSBhbmQgc2VwYXJhdGlvbiBvZiBkdXRpZXMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmlhbS5JQU1Qb2xpY3lOb1N0YXRlbWVudHNXaXRoRnVsbEFjY2VzcyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEtpbmVzaXMgRGF0YSBGaXJlaG9zZSBkZWxpdmVyeSBzdHJlYW0gZG9lcyBub3QgaGF2ZSBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVuYWJsaW5nIGVuY3J5cHRpb24gYWxsb3dzIHRoZSBzeXN0ZW0gdG8gbWVldCBzdHJpY3QgcmVndWxhdG9yeSByZXF1aXJlbWVudHMgYW5kIGVuaGFuY2UgdGhlIHNlY3VyaXR5IG9mIHN5c3RlbSBkYXRhLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5raW5lc2lzLktpbmVzaXNEYXRhRmlyZWhvc2VTU0UsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiAnVGhlIEtpbmVzaXMgRGF0YSBTdHJlYW0gc3BlY2lmaWVzIHNlcnZlci1zaWRlIGVuY3J5cHRpb24gYW5kIGRvZXMgbm90IHVzZSB0aGUgXCJhd3Mva2luZXNpc1wiIGtleS4nLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJDdXN0b21lciBNYW5hZ2VkIEtleXMgY2FuIGluY3VyIGFkZGl0aW9uYWwgY29zdHMgdGhhdCBzY2FsZSB3aXRoIHRoZSBhbW91bnQgb2YgY29uc3VtZXJzIGFuZCBwcm9kdWNlcnMuIEVuc3VyZSB0aGF0IEN1c3RvbWVyIE1hbmFnZWQgS2V5cyBhcmUgcmVxdWlyZWQgZm9yIGNvbXBsaWFuY2UgYmVmb3JlIHVzaW5nIHRoZW0gKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdHJlYW1zL2xhdGVzdC9kZXYvY29zdHMtcGVyZm9ybWFuY2UuaHRtbCkuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmtpbmVzaXMuS2luZXNpc0RhdGFTdHJlYW1EZWZhdWx0S2V5V2hlblNTRSxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIEtpbmVzaXMgRGF0YSBTdHJlYW0gZG9lcyBub3QgaGF2ZSBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkRhdGEgaXMgZW5jcnlwdGVkIGJlZm9yZSBpdCdzIHdyaXR0ZW4gdG8gdGhlIEtpbmVzaXMgc3RyZWFtIHN0b3JhZ2UgbGF5ZXIsIGFuZCBkZWNyeXB0ZWQgYWZ0ZXIgaXTigJlzIHJldHJpZXZlZCBmcm9tIHN0b3JhZ2UuIFRoaXMgYWxsb3dzIHRoZSBzeXN0ZW0gdG8gbWVldCBzdHJpY3QgcmVndWxhdG9yeSByZXF1aXJlbWVudHMgYW5kIGVuaGFuY2UgdGhlIHNlY3VyaXR5IG9mIHN5c3RlbSBkYXRhLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5raW5lc2lzLktpbmVzaXNEYXRhU3RyZWFtU1NFLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgS01TIFN5bW1ldHJpYyBrZXkgZG9lcyBub3QgaGF2ZSBhdXRvbWF0aWMga2V5IHJvdGF0aW9uIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIktNUyBrZXkgcm90YXRpb24gYWxsb3cgYSBzeXN0ZW0gdG8gc2V0IGEgcm90YXRpb24gc2NoZWR1bGUgZm9yIGEgS01TIGtleSBzbyB3aGVuIGEgQVdTIEtNUyBrZXkgaXMgcmVxdWlyZWQgdG8gZW5jcnlwdCBuZXcgZGF0YSwgdGhlIEtNUyBzZXJ2aWNlIGNhbiBhdXRvbWF0aWNhbGx5IHVzZSB0aGUgbGF0ZXN0IHZlcnNpb24gb2YgdGhlIEhTQSBiYWNraW5nIGtleSB0byBwZXJmb3JtIHRoZSBlbmNyeXB0aW9uLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5rbXMuS01TQmFja2luZ0tleVJvdGF0aW9uRW5hYmxlZCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIExhbWJkYSBmdW5jdGlvbiBwZXJtaXNzaW9uIGdyYW50cyBwdWJsaWMgYWNjZXNzXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlB1YmxpYyBhY2Nlc3MgYWxsb3dzIGFueW9uZSBvbiB0aGUgaW50ZXJuZXQgdG8gcGVyZm9ybSB1bmF1dGhlbnRpY2F0ZWQgYWN0aW9ucyBvbiB5b3VyIGZ1bmN0aW9uIGFuZCBjYW4gcG90ZW50aWFsbHkgbGVhZCB0byBkZWdyYWRlZCBhdmFpbGFiaWxpdHkuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIHdobyBoYXMgZGlzY292ZXJlZCB0aGUgTGFtYmRhIGZ1bmN0aW9uIG5hbWUgb3IgQVJOIGNhbiBpbnZva2UvZGVsZXRlL21vZGlmeSB0aGUgTGFtYmRhIGZ1bmN0aW9uIHdpdGhvdXQgYW55IGF1dGhlbnRpY2F0aW9uLCB3aGljaCBtYXkgbGVhZCB0byByZWNvbm5haXNzYW5jZSBhbmQgaW50cnVzaW9uIGFjdGl2aXRpZXMgYmVpbmcgcGVyZm9ybWVkIGFnYWluc3QgdGhlIGV4cG9zZWQgYXR0YWNrIHN1cmZhY2UgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLmxhbWJkYS5MYW1iZGFGdW5jdGlvblB1YmxpY0FjY2Vzc1Byb2hpYml0ZWQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBMYW1iZGEgRnVuY3Rpb24gVVJMIGFsbG93cyBmb3IgcHVibGljLCB1bmF1dGhlbnRpY2F0ZWQgYWNjZXNzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJBV1MgTGFtYmRhIEZ1bmN0aW9uIFVSTHMgYWxsb3cgeW91IHRvIGludm9rZSB5b3VyIGZ1bmN0aW9uIHZpYSBhIEhUVFBTIGVuZC1wb2ludCwgc2V0dGluZyB0aGUgYXV0aGVudGljYXRpb24gdG8gTk9ORSBhbGxvd3MgYW55b25lIG9uIHRoZSBpbnRlcm5ldCB0byBpbnZva2UgeW91ciBmdW5jdGlvbi5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3Igd2hvIGhhcyBkaXNjb3ZlcmVkIHRoZSBMYW1iZGEgRnVuY3Rpb24gVVJMIGNhbiBpbnZva2UgdGhlIExhbWJkYSBmdW5jdGlvbiB3aXRob3V0IGFueSBhdXRoZW50aWNhdGlvbiwgd2hpY2ggbWF5IGxlYWQgdG8gcmVjb25uYWlzc2FuY2UgYW5kIGludHJ1c2lvbiBhY3Rpdml0aWVzIGJlaW5nIHBlcmZvcm1lZCBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhbmQgcmVzb3VyY2UgYXNzZXRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5sYW1iZGEuTGFtYmRhRnVuY3Rpb25VcmxBdXRoLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgbm9uLWNvbnRhaW5lciBMYW1iZGEgZnVuY3Rpb24gaXMgbm90IGNvbmZpZ3VyZWQgdG8gdXNlIHRoZSBsYXRlc3QgcnVudGltZSB2ZXJzaW9uLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJVc2UgdGhlIGxhdGVzdCBhdmFpbGFibGUgcnVudGltZSBmb3IgdGhlIHRhcmdldGVkIGxhbmd1YWdlIHRvIGF2b2lkIHRlY2huaWNhbCBkZWJ0LiBSdW50aW1lcyBzcGVjaWZpYyB0byBhIGxhbmd1YWdlIG9yIGZyYW1ld29yayB2ZXJzaW9uIGFyZSBkZXByZWNhdGVkIHdoZW4gdGhlIHZlcnNpb24gcmVhY2hlcyBlbmQgb2YgbGlmZS4gVGhpcyBydWxlIG9ubHkgYXBwbGllcyB0byBub24tY29udGFpbmVyIExhbWJkYSBmdW5jdGlvbnMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLmxhbWJkYS5MYW1iZGFMYXRlc3RWZXJzaW9uLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgTWVkaWFTdG9yZSBjb250YWluZXIgZG9lcyBub3QgZGVmaW5lIGEgQ09SUyBwb2xpY3kuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlVzaW5nIGEgQ09SUyBwb2xpY3kgaGVscHMgZm9sbG93IHRoZSBzdGFuZGFyZCBzZWN1cml0eSBhZHZpY2Ugb2YgZ3JhbnRpbmcgbGVhc3QgcHJpdmlsZWdlLCBvciBncmFudGluZyBvbmx5IHRoZSBwZXJtaXNzaW9ucyByZXF1aXJlZCB0byBhbGxvdyBuZWVkZWQgYWNjZXNzIHRvIHRoZSBjb250YWluZXIuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLm1lZGlhc3RvcmUuTWVkaWFTdG9yZUNvbnRhaW5lckNPUlNQb2xpY3ksXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBNZWRpYVN0b3JlIGNvbnRhaW5lciBkb2VzIG5vdCBkZWZpbmUgYSBjb250YWluZXIgcG9saWN5LlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJVc2luZyBhIGNvbnRhaW5lciBwb2xpY3kgaGVscHMgZm9sbG93IHRoZSBzdGFuZGFyZCBzZWN1cml0eSBhZHZpY2Ugb2YgZ3JhbnRpbmcgbGVhc3QgcHJpdmlsZWdlLCBvciBncmFudGluZyBvbmx5IHRoZSBwZXJtaXNzaW9ucyByZXF1aXJlZCB0byBhbGxvdyBuZWVkZWQgYWNjZXNzIHRvIHRoZSBjb250YWluZXIuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLm1lZGlhc3RvcmUuTWVkaWFTdG9yZUNvbnRhaW5lckhhc0NvbnRhaW5lclBvbGljeSxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIE1lZGlhU3RvcmUgY29udGFpbmVyIGRvZXMgbm90IHJlcXVpcmUgcmVxdWVzdHMgdG8gdXNlIFNTTC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiWW91IGNhbiB1c2UgSFRUUFMgKFRMUykgdG8gaGVscCBwcmV2ZW50IHBvdGVudGlhbCBhdHRhY2tlcnMgZnJvbSBlYXZlc2Ryb3BwaW5nIG9uIG9yIG1hbmlwdWxhdGluZyBuZXR3b3JrIHRyYWZmaWMgdXNpbmcgcGVyc29uLWluLXRoZS1taWRkbGUgb3Igc2ltaWxhciBhdHRhY2tzLiBZb3Ugc2hvdWxkIGFsbG93IG9ubHkgZW5jcnlwdGVkIGNvbm5lY3Rpb25zIG92ZXIgSFRUUFMgKFRMUykgdXNpbmcgdGhlIGF3czpTZWN1cmVUcmFuc3BvcnQgY29uZGl0aW9uIG9uIE1lZGlhU3RvcmUgY29udGFpbmVyIHBvbGljaWVzLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5tZWRpYXN0b3JlLk1lZGlhU3RvcmVDb250YWluZXJTU0xSZXF1ZXN0c09ubHksXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBNU0sgY2x1c3RlciB1c2VzIHBsYWludGV4dCBjb21tdW5pY2F0aW9uIGJldHdlZW4gYnJva2Vycy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiVExTIGNvbW11bmljYXRpb24gc2VjdXJlcyBkYXRhLWluLXRyYW5zaXQgYnkgZW5jcnlwdGluZyB0aGUgY29ubmVjdGlvbiBiZXR3ZWVuIGJyb2tlcnMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLm1zay5NU0tCcm9rZXJUb0Jyb2tlclRMUyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIE1TSyBjbHVzdGVyIHVzZXMgcGxhaW50ZXh0IGNvbW11bmljYXRpb24gYmV0d2VlbiBjbGllbnRzIGFuZCBicm9rZXJzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJUTFMgb25seSBjb21tdW5pY2F0aW9uIHNlY3VyZXMgZGF0YS1pbi10cmFuc2l0IGJ5IGVuY3J5cHRpbmcgdGhlIGNvbm5lY3Rpb24gYmV0d2VlbiB0aGUgY2xpZW50cyBhbmQgYnJva2Vycy5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMubXNrLk1TS0NsaWVudFRvQnJva2VyVExTLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgTmVwdHVuZSBEQiBpbnN0YW5jZSBkb2VzIGhhdmUgQXV0byBNaW5vciBWZXJzaW9uIFVwZ3JhZGUgZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiVGhlIE5lcHR1bmUgc2VydmljZSByZWd1bGFybHkgcmVsZWFzZXMgZW5naW5lIHVwZGF0ZXMuIEVuYWJsaW5nIEF1dG8gTWlub3IgVmVyc2lvbiBVcGdyYWRlIHdpbGwgYWxsb3cgdGhlIHNlcnZpY2UgdG8gYXV0b21hdGljYWxseSBhcHBseSB0aGVzZSB1cGdyYWRlcyB0byBEQiBJbnN0YW5jZXMuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEFuIGFjdG9yIHdpdGggYSBuZXR3b3JrIHBhdGggdG8gdGhlIE5lcHR1bmUgY2x1c3RlciBvciBpbnN0YW5jZSBjYW4gYXR0ZW1wdCB0byB0YWtlIGFkdmFudGFnZSBvZiBhIGtub3duIHZ1bG5lcmFiaWxpdHkgaW4gYSBjb21wb25lbnQgZXhwb3NlZCBieSBOZXB0dW5lLCB3aGljaCBtYXkgbGVhZCB0byB1bmtub3duIGltcGFjdHMgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLm5lcHR1bmUuTmVwdHVuZUNsdXN0ZXJBdXRvbWF0aWNNaW5vclZlcnNpb25VcGdyYWRlLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgTmVwdHVuZSBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgZW5jcnlwdGlvbiBhdCByZXN0IGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVuY3J5cHRpbmcgZGF0YS1hdC1yZXN0IHByb3RlY3RzIGRhdGEgY29uZmlkZW50aWFsaXR5IGFuZCBwcmV2ZW50cyB1bmF1dGhvcml6ZWQgdXNlcnMgZnJvbSBhY2Nlc3Npbmcgc2Vuc2l0aXZlIGluZm9ybWF0aW9uLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5uZXB0dW5lLk5lcHR1bmVDbHVzdGVyRW5jcnlwdGlvbkF0UmVzdCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIE5lcHR1bmUgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJXaXRoIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLCB0aGUgc3lzdGVtIGRvZXNuJ3QgaGF2ZSB0byB1c2UgYSBwYXNzd29yZCB3aGVuIGNvbm5lY3RpbmcgdG8gdGhlIGNsdXN0ZXIuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLm5lcHR1bmUuTmVwdHVuZUNsdXN0ZXJJQU1BdXRoLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBvbmx5IGdyYW50IGFjY2VzcyB2aWEgYWxsb3dsaXN0ZWQgSVAgYWRkcmVzc2VzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJVc2luZyBhbGxvd2xpc3RlZCBJUCBhZGRyZXNzZXMgaGVscHMgcHJvdGVjdCB0aGUgZG9tYWluIGFnYWluc3QgdW5hdXRob3JpemVkIGFjY2Vzcy5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMub3BlbnNlYXJjaC5PcGVuU2VhcmNoQWxsb3dsaXN0ZWRJUHMsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIGRvZXMgbm90IGhhdmUgZW5jcnlwdGlvbiBhdCByZXN0IGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGFuZCB0byBoZWxwIHByb3RlY3QgZGF0YSBhdCByZXN0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgKE9wZW5TZWFyY2ggU2VydmljZSkgZG9tYWlucy5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMub3BlbnNlYXJjaC5PcGVuU2VhcmNoRW5jcnlwdGVkQXRSZXN0LFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBpcyBub3QgcHJvdmlzaW9uZWQgaW5zaWRlIGEgVlBDLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJQcm92aXNpb25pbmcgdGhlIGRvbWFpbiB3aXRoaW4gYSBWUEMgZW5hYmxlcyBiZXR0ZXIgZmxleGliaWxpdHkgYW5kIGNvbnRyb2wgb3ZlciB0aGUgY2x1c3RlcnMgYWNjZXNzIGFuZCBzZWN1cml0eSBhcyB0aGlzIGZlYXR1cmUga2VlcHMgYWxsIHRyYWZmaWMgYmV0d2VlbiB0aGUgVlBDIGFuZCBPcGVuU2VhcmNoIGRvbWFpbnMgd2l0aGluIHRoZSBBV1MgbmV0d29yayBpbnN0ZWFkIG9mIGdvaW5nIG92ZXIgdGhlIHB1YmxpYyBJbnRlcm5ldC5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3IgY2FuIGRpc2NvdmVyIHRoZSBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIHRoYXQgaGF2ZSBwdWJsaWMgSVAgYWRkcmVzc2VzLCB3aGljaCBtYXkgbGVhZCB0byByZWNvbm5haXNzYW5jZSBhbmQgaW50cnVzaW9uIGFjdGl2aXRpZXMgYmVpbmcgcGVyZm9ybWVkIGFnYWluc3QgdGhlIGV4cG9zZWQgYXR0YWNrIHN1cmZhY2UgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIHdpdGggdGhlIGNsdXN0ZXIgdXNlZCB3aXRoaW4gdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMub3BlbnNlYXJjaC5PcGVuU2VhcmNoSW5WUENPbmx5LFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBoYXZlIG5vZGUtdG8tbm9kZSBlbmNyeXB0aW9uIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0LCBlbmFibGUgZW5jcnlwdGlvbiBpbiB0cmFuc2l0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEgd2l0aGluIHlvdXIgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSAoT3BlblNlYXJjaCBTZXJ2aWNlKSBkb21haW5zLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5vcGVuc2VhcmNoLk9wZW5TZWFyY2hOb2RlVG9Ob2RlRW5jcnlwdGlvbixcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3QgYWxsb3cgZm9yIHVuc2lnbmVkIHJlcXVlc3RzIG9yIGFub255bW91cyBhY2Nlc3MuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlJlc3RyaWN0aW5nIHB1YmxpYyBhY2Nlc3MgaGVscHMgcHJldmVudCB1bmF1dGhvcml6ZWQgYWNjZXNzIGFuZCBwcmV2ZW50cyBhbnkgdW5zaWduZWQgcmVxdWVzdHMgdG8gYmUgbWFkZSB0byB0aGUgcmVzb3VyY2VzLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBbiBhY3RvciB3aXRoIGEgbmV0d29yayBwYXRoIHRvIHRoZSBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIGNhbiBkaXJlY3RseSBhY2Nlc3MgdGhlIGRvbWFpbiB3aXRob3V0IGF1dGhlbnRpY2F0aW9uLCB3aGljaCBtYXkgbGVhZCB0byBhbGxvd2luZyBhY2Nlc3MgdG8gZGF0YSBob3N0ZWQgd2l0aGluIHRoZSBkb21haW4gcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFzc2V0cyBob3N0ZWQgb24gdGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZm9yIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLm9wZW5zZWFyY2guT3BlblNlYXJjaE5vVW5zaWduZWRPckFub255bW91c0FjY2VzcyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFF1aWNrc2lnaHQgZGF0YSBzb3VyY2VzIGNvbm5lY3Rpb24gaXMgbm90IGNvbmZpZ3VyZWQgdG8gdXNlIFNTTC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiU1NMIHNlY3VyZXMgY29tbXVuaWNhdGlvbnMgdG8gZGF0YSBzb3VyY2VzLCBlc3BlY2lhbGx5IHdoZW4gdXNpbmcgcHVibGljIG5ldHdvcmtzLiBVc2luZyBTU0wgd2l0aCBRdWlja1NpZ2h0IHJlcXVpcmVzIHRoZSB1c2Ugb2YgY2VydGlmaWNhdGVzIHNpZ25lZCBieSBhIHB1YmxpY2x5LXJlY29nbml6ZWQgY2VydGlmaWNhdGUgYXV0aG9yaXR5LlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5xdWlja3NpZ2h0LlF1aWNrc2lnaHRTU0xDb25uZWN0aW9ucyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFJEUyBBdXJvcmEgTXlTUUwvUG9zdGdyZXNTUUwgY2x1c3RlciBkb2VzIG5vdCBoYXZlIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJXaXRoIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLCB0aGUgc3lzdGVtIGRvZXNuJ3QgaGF2ZSB0byB1c2UgYSBwYXNzd29yZCB3aGVuIGNvbm5lY3RpbmcgdG8gdGhlIE15U1FML1Bvc3RncmVTUUwgZGF0YWJhc2UgaW5zdGFuY2VzLCBpbnN0ZWFkIGl0IHVzZXMgYW4gYXV0aGVudGljYXRpb24gdG9rZW4uXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnJkcy5BdXJvcmFNeVNRTFBvc3RncmVzSUFNQXV0aCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFJEUyBEQiBpbnN0YW5jZSBkb2VzIG5vdCBoYXZlIGF1dG9tYXRpYyBtaW5vciB2ZXJzaW9uIHVwZ3JhZGVzIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVuYWJsZSBhdXRvbWF0aWMgbWlub3IgdmVyc2lvbiB1cGdyYWRlcyBvbiB5b3VyIEFtYXpvbiBSZWxhdGlvbmFsIERhdGFiYXNlIFNlcnZpY2UgKFJEUykgaW5zdGFuY2VzIHRvIGVuc3VyZSB0aGUgbGF0ZXN0IG1pbm9yIHZlcnNpb24gdXBkYXRlcyB0byB0aGUgUmVsYXRpb25hbCBEYXRhYmFzZSBNYW5hZ2VtZW50IFN5c3RlbSAoUkRCTVMpIGFyZSBpbnN0YWxsZWQsIHdoaWNoIG1heSBpbmNsdWRlIHNlY3VyaXR5IHBhdGNoZXMgYW5kIGJ1ZyBmaXhlcy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQW4gYWN0b3Igd2l0aCBhIG5ldHdvcmsgcGF0aCB0byB0aGUgUkRTIGNsdXN0ZXIgb3IgaW5zdGFuY2UgY2FuIGF0dGVtcHQgdG8gdGFrZSBhZHZhbnRhZ2Ugb2YgYSBrbm93biB2dWxuZXJhYmlsaXR5IGluIGEgY29tcG9uZW50IGV4cG9zZWQgYnkgUkRTLCB3aGljaCBtYXkgbGVhZCB0byB1bmtub3duIGltcGFjdHMgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFzc2V0cyBob3N0ZWQgb24gdGhlIFJEUyBDbHVzdGVyIG9yIGluc3RhbmNlIGZvciB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5yZHMuUkRTQXV0b21hdGljTWlub3JWZXJzaW9uVXBncmFkZUVuYWJsZWQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBSRFMgREIgaW5zdGFuY2UgYWxsb3dzIHB1YmxpYyBhY2Nlc3MuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkFtYXpvbiBSRFMgZGF0YWJhc2UgaW5zdGFuY2VzIGNhbiBjb250YWluIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbiwgaGVuY2UgYXBwcm9wcmlhdGUgYWNjZXNzIGNvbnRyb2wgYW5kIHByaW5jaXBsZXMgb2YgbGVhc3QgcHJpdmlsZWdlIHNob3VsZCBiZSBhcHBsaWVkLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciB3aG8gaGFzIGRpc2NvdmVyZWQgdGhlIFJEUyBEQiBpbnN0YW5jZSBlbmRwb2ludCBjYW4gcGVyZm9ybSByZWNvbm5haXNzYW5jZSBhbmQgaW50cnVzaW9uIGFjdGl2aXRpZXMgKGUuZy4gYnJ1dGUgZm9yY2UvZGljdGlvbmFyeSBhdHRhY2sgdG8gYXV0aGVudGljYXRlIGFzIGEgdmFsaWQgdXNlcikgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSwgd2hpY2ggbWF5IGxlYWQgdG8gcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFzc2V0cyBob3N0ZWQgb24gdGhlIFJEUyBDbHVzdGVyIG9yIGluc3RhbmNlIGZvciB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5yZHMuUkRTSW5zdGFuY2VQdWJsaWNBY2Nlc3MsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBSRFMgaW5zdGFuY2Ugb3IgQXVyb3JhIERCIGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBlbmRwb2ludCBwb3J0LlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJQb3J0IG9iZnVzY2F0aW9uICh1c2luZyBhIG5vbiBkZWZhdWx0IGVuZHBvaW50IHBvcnQpIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MgKGkuZS4gTXlTUUwvQXVyb3JhIHBvcnQgMzMwNiwgU1FMIFNlcnZlciBwb3J0IDE0MzMsIFBvc3RncmVTUUwgcG9ydCA1NDMyLCBldGMpLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5yZHMuUkRTTm9uRGVmYXVsdFBvcnQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBSRFMgREIgU2VjdXJpdHkgR3JvdXAgYWxsb3dzIGZvciAwLjAuMC4wLzAgaW5ib3VuZCBhY2Nlc3MuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlJEUyBEQiBzZWN1cml0eSBncm91cHMgc2hvdWxkIG5vdCBhbGxvdyBhY2Nlc3MgZnJvbSAwLjAuMC4wLzAgKGkuZS4gYW55d2hlcmUsIGV2ZXJ5IG1hY2hpbmUgdGhhdCBoYXMgdGhlIGFiaWxpdHkgdG8gZXN0YWJsaXNoIGEgY29ubmVjdGlvbikgaW4gb3JkZXIgdG8gcmVkdWNlIHRoZSByaXNrIG9mIHVuYXV0aG9yaXplZCBhY2Nlc3MuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIGNhbiBkaXNjb3ZlciBSRFMgREIgaW5zdGFuY2VzIHRoYXQgaGF2ZSBwdWJsaWMgSVAgYWRkcmVzc2VzIGFuZCBhbGxvdyBpbmdyZXNzIHRvIGFsbCBpbnRlcm5ldCBhZGRyZXNzIG9yIG1vdmUgbGF0ZXJhbGx5IHRvIG5vbi1wdWJsaWMgUkRTIERCIGluc3RhbmNlcywgd2hpY2ggbWF5IGxlYWQgdG8gcmVjb25uYWlzc2FuY2UgYW5kIGludHJ1c2lvbiBhY3Rpdml0aWVzIGJlaW5nIHBlcmZvcm1lZCBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhc3NldHMgaG9zdGVkIG9uIHRoZSBSRFMgQ2x1c3RlciBvciBpbnN0YW5jZSBmb3IgdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMucmRzLlJEU1Jlc3RyaWN0ZWRJbmJvdW5kLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgUkRTIERCIGluc3RhbmNlIG9yIEF1cm9yYSBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgZW5jcnlwdGlvbiBhdCByZXN0IGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gQW1hem9uIFJEUyBEQiBpbnN0YW5jZXMsIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMucmRzLlJEU1N0b3JhZ2VFbmNyeXB0ZWQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBSZWRzaGlmdCBjbHVzdGVyIGRvZXMgbm90IGhhdmUgZW5jcnlwdGlvbiBhdCByZXN0IGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246IFwiRW5jcnlwdGluZyBkYXRhLWF0LXJlc3QgcHJvdGVjdHMgZGF0YSBjb25maWRlbnRpYWxpdHkuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnJlZHNoaWZ0LlJlZHNoaWZ0Q2x1c3RlckVuY3J5cHRpb25BdFJlc3QsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBSZWRzaGlmdCBjbHVzdGVyIGlzIG5vdCBwcm92aXNpb25lZCBpbiBhIFZQQy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiUHJvdmlzaW9uaW5nIHRoZSBjbHVzdGVyIHdpdGhpbiBhIFZQQyBhbGxvd3MgZm9yIGJldHRlciBmbGV4aWJpbGl0eSBhbmQgY29udHJvbCBvdmVyIHRoZSBSZWRzaGlmdCBjbHVzdGVycyBzZWN1cml0eSwgYXZhaWxhYmlsaXR5LCB0cmFmZmljIHJvdXRpbmcgYW5kIG1vcmUuXFxuXFxuRXhhbXBsZSB0aHJlYXQ6IEEgZ2xvYmFsIGludGVybmV0LWJhc2VkIGFjdG9yIGNhbiBkaXNjb3ZlciBhIFJlZFNoaWZ0IGNsdXN0ZXIgdGhhdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyAoZS5nLiBicnV0ZSBmb3JjZS9kaWN0aW9uYXJ5IGF0dGFjayB0byBhdXRoZW50aWNhdGUgYXMgYSB2YWxpZCB1c2VyKSBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgd2l0aCB0aGUgY2x1c3RlciB1c2VkIHdpdGhpbiB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5yZWRzaGlmdC5SZWRzaGlmdENsdXN0ZXJJblZQQyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBlbmRwb2ludCBwb3J0LlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJQb3J0IG9iZnVzY2F0aW9uICh1c2luZyBhIG5vbiBkZWZhdWx0IGVuZHBvaW50IHBvcnQpIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MgKGkuZS4gUmVkc2hpZnQgcG9ydCA1NDM5KS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMucmVkc2hpZnQuUmVkc2hpZnRDbHVzdGVyTm9uRGVmYXVsdFBvcnQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBcImF3c3VzZXJcIiB1c2VybmFtZS4nLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgJ1VzaW5nIGEgY3VzdG9tIHVzZXIgbmFtZSBpbnN0ZWFkIG9mIHRoZSBkZWZhdWx0IG1hc3RlciB1c2VyIG5hbWUgKGkuZS4gXCJhd3N1c2VyXCIpIHByb3ZpZGVzIGFuIGFkZGl0aW9uYWwgbGF5ZXIgb2YgZGVmZW5zZSBhZ2FpbnN0IG5vbi10YXJnZXRlZCBhdHRhY2tzLicsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnJlZHNoaWZ0LlJlZHNoaWZ0Q2x1c3Rlck5vbkRlZmF1bHRVc2VybmFtZSxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgYWxsb3dzIHB1YmxpYyBhY2Nlc3MuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkFtYXpvbiBSZWRzaGlmdCBjbHVzdGVycyBjYW4gY29udGFpbiBzZW5zaXRpdmUgaW5mb3JtYXRpb24sIGhlbmNlIGFwcHJvcHJpYXRlIGFjY2VzcyBjb250cm9sIGFuZCBwcmluY2lwbGVzIG9mIGxlYXN0IHByaXZpbGVnZSBzaG91bGQgYmUgYXBwbGllZC5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3Igd2hvIGhhcyBkaXNjb3ZlcmVkIHRoZSBSZWRzaGlmdCBjbHVzdGVyIGVuZHBvaW50IGNhbiBwZXJmb3JtIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyAoZS5nLiBicnV0ZSBmb3JjZS9kaWN0aW9uYXJ5IGF0dGFjayB0byBhdXRoZW50aWNhdGUgYXMgYSB2YWxpZCB1c2VyKSBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlLCB3aGljaCBtYXkgbGVhZCB0byBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYXNzZXRzIGhvc3RlZCBvbiB0aGUgUmVkc2hpZnQgY2x1c3RlciBmb3IgdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMucmVkc2hpZnQuUmVkc2hpZnRDbHVzdGVyUHVibGljQWNjZXNzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIHZlcnNpb24gdXBncmFkZSBlbmFibGVkLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJWZXJzaW9uIFVwZ3JhZGUgbXVzdCBlbmFibGVkIG9uIHRoZSBjbHVzdGVyIGluIG9yZGVyIHRvIGF1dG9tYXRpY2FsbHkgcmVjZWl2ZSB1cGdyYWRlcyBkdXJpbmcgdGhlIG1haW50ZW5hbmNlIHdpbmRvdy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQW4gYWN0b3Igd2l0aCBhIG5ldHdvcmsgcGF0aCB0byB0aGUgUmVkc2hpZnQgY2x1c3RlciBjYW4gYXR0ZW1wdCB0byB0YWtlIGFkdmFudGFnZSBvZiBhIGtub3duIHZ1bG5lcmFiaWxpdHkgaW4gYSBjb21wb25lbnQgZXhwb3NlZCBieSBSZWRzaGlmdCwgd2hpY2ggbWF5IGxlYWQgdG8gdW5rbm93biBpbXBhY3RzIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhc3NldHMgaG9zdGVkIG9uIHRoZSBSZWRzaGlmdCBjbHVzdGVyIGZvciB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5yZWRzaGlmdC5SZWRzaGlmdENsdXN0ZXJWZXJzaW9uVXBncmFkZSxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgZG9lcyBub3QgcmVxdWlyZSBUTFMvU1NMIGVuY3J5cHRpb24uXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkVuc3VyZSB0aGF0IHlvdXIgQW1hem9uIFJlZHNoaWZ0IGNsdXN0ZXJzIHJlcXVpcmUgVExTL1NTTCBlbmNyeXB0aW9uIHRvIGNvbm5lY3QgdG8gU1FMIGNsaWVudHMuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0LCBlbmFibGUgZW5jcnlwdGlvbiBpbiB0cmFuc2l0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnJlZHNoaWZ0LlJlZHNoaWZ0UmVxdWlyZVRsc1NTTCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFMzIGJ1Y2tldCBkb2VzIG5vdCBwcm9oaWJpdCBwdWJsaWMgYWNjZXNzIHRocm91Z2ggYnVja2V0IGxldmVsIHNldHRpbmdzLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJLZWVwIHNlbnNpdGl2ZSBkYXRhIHNhZmUgZnJvbSB1bmF1dGhvcml6ZWQgcmVtb3RlIHVzZXJzIGJ5IHByZXZlbnRpbmcgcHVibGljIGFjY2VzcyBhdCB0aGUgYnVja2V0IGxldmVsLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciB3aG8gaGFzIGRpc2NvdmVyZWQgYSBTMyBidWNrZXQgY29uZmlndXJlZCBmb3IgcHVibGljIHJlYWQgb3Igd3JpdGUgY2FuIHJlYWQgb3Igd3JpdGUgZGF0YSB0byBvciBmcm9tIHRoZSBTMyBidWNrZXQsIHdoaWNoIG1heSBsZWFkIHRvIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhc3NldHMgaG9zdGVkIG9uIHRoZSBTMyBidWNrZXQgZm9yIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLnMzLlMzQnVja2V0TGV2ZWxQdWJsaWNBY2Nlc3NQcm9oaWJpdGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgUzMgQnVja2V0IGRvZXMgbm90IGhhdmUgc2VydmVyIGFjY2VzcyBsb2dzIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkFtYXpvbiBTaW1wbGUgU3RvcmFnZSBTZXJ2aWNlIChBbWF6b24gUzMpIHNlcnZlciBhY2Nlc3MgbG9nZ2luZyBwcm92aWRlcyBhIG1ldGhvZCB0byBtb25pdG9yIHRoZSBuZXR3b3JrIGZvciBwb3RlbnRpYWwgY3liZXJzZWN1cml0eSBldmVudHMuIFRoZSBldmVudHMgYXJlIG1vbml0b3JlZCBieSBjYXB0dXJpbmcgZGV0YWlsZWQgcmVjb3JkcyBmb3IgdGhlIHJlcXVlc3RzIHRoYXQgYXJlIG1hZGUgdG8gYW4gQW1hem9uIFMzIGJ1Y2tldC4gRWFjaCBhY2Nlc3MgbG9nIHJlY29yZCBwcm92aWRlcyBkZXRhaWxzIGFib3V0IGEgc2luZ2xlIGFjY2VzcyByZXF1ZXN0LiBUaGUgZGV0YWlscyBpbmNsdWRlIHRoZSByZXF1ZXN0ZXIsIGJ1Y2tldCBuYW1lLCByZXF1ZXN0IHRpbWUsIHJlcXVlc3QgYWN0aW9uLCByZXNwb25zZSBzdGF0dXMsIGFuZCBhbiBlcnJvciBjb2RlLCBpZiByZWxldmFudC5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMuczMuUzNCdWNrZXRMb2dnaW5nRW5hYmxlZCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFMzIEJ1Y2tldCBkb2VzIG5vdCBwcm9oaWJpdCBwdWJsaWMgcmVhZCBhY2Nlc3MgdGhyb3VnaCBpdHMgQmxvY2sgUHVibGljIEFjY2VzcyBjb25maWd1cmF0aW9ucyBhbmQgYnVja2V0IEFDTHMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlRoZSBtYW5hZ2VtZW50IG9mIGFjY2VzcyBzaG91bGQgYmUgY29uc2lzdGVudCB3aXRoIHRoZSBjbGFzc2lmaWNhdGlvbiBvZiB0aGUgZGF0YS5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3Igd2hvIGhhcyBkaXNjb3ZlcmVkIGEgUzMgYnVja2V0IGNvbmZpZ3VyZWQgZm9yIHB1YmxpYyByZWFkIGNhbiByZWFkIGRhdGEgZnJvbSB0aGUgUzMgYnVja2V0LCB3aGljaCBtYXkgbGVhZCB0byBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSBvZiB0aGUgZGF0YSBhc3NldHMgaG9zdGVkIG9uIHRoZSBTMyBidWNrZXQgZm9yIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLnMzLlMzQnVja2V0UHVibGljUmVhZFByb2hpYml0ZWQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBTMyBCdWNrZXQgZG9lcyBub3QgcHJvaGliaXQgcHVibGljIHdyaXRlIGFjY2VzcyB0aHJvdWdoIGl0cyBCbG9jayBQdWJsaWMgQWNjZXNzIGNvbmZpZ3VyYXRpb25zIGFuZCBidWNrZXQgQUNMcy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiVGhlIG1hbmFnZW1lbnQgb2YgYWNjZXNzIHNob3VsZCBiZSBjb25zaXN0ZW50IHdpdGggdGhlIGNsYXNzaWZpY2F0aW9uIG9mIHRoZSBkYXRhLlxcblxcbkV4YW1wbGUgdGhyZWF0OiBBIGdsb2JhbCBpbnRlcm5ldC1iYXNlZCBhY3RvciB3aG8gaGFzIGRpc2NvdmVyZWQgYSBTMyBidWNrZXQgY29uZmlndXJlZCBmb3IgcHVibGljIHdyaXRlIGNhbiB3cml0ZSBkYXRhIHRvLCBvciBvdmVyd3JpdGUgZGF0YSB3aXRoaW4gdGhlIFMzIGJ1Y2tldCwgd2hpY2ggbWF5IGxlYWQgdG8gcG9zc2libHkgaW1wYWN0aW5nIHRoZSBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhc3NldHMgaG9zdGVkIG9uIHRoZSBTMyBidWNrZXQgZm9yIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLnMzLlMzQnVja2V0UHVibGljV3JpdGVQcm9oaWJpdGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgUzMgQnVja2V0IGRvZXMgbm90IGhhdmUgZGVmYXVsdCBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gQW1hem9uIFMzIGJ1Y2tldHMsIGVuYWJsZSBlbmNyeXB0aW9uIHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnMzLlMzQnVja2V0U2VydmVyU2lkZUVuY3J5cHRpb25FbmFibGVkLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgUzMgQnVja2V0IG9yIGJ1Y2tldCBwb2xpY3kgZG9lcyBub3QgcmVxdWlyZSByZXF1ZXN0cyB0byB1c2UgU1NML1RMUy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiWW91IGNhbiB1c2UgSFRUUFMgKFRMUykgdG8gaGVscCBwcmV2ZW50IHBvdGVudGlhbCBhdHRhY2tlcnMgZnJvbSBlYXZlc2Ryb3BwaW5nIG9uIG9yIG1hbmlwdWxhdGluZyBuZXR3b3JrIHRyYWZmaWMgdXNpbmcgcGVyc29uLWluLXRoZS1taWRkbGUgb3Igc2ltaWxhciBhdHRhY2tzLiBZb3Ugc2hvdWxkIGFsbG93IG9ubHkgZW5jcnlwdGVkIGNvbm5lY3Rpb25zIG92ZXIgSFRUUFMgKFRMUykgdXNpbmcgdGhlIGF3czpTZWN1cmVUcmFuc3BvcnQgY29uZGl0aW9uIG9uIEFtYXpvbiBTMyBidWNrZXQgcG9saWNpZXMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnMzLlMzQnVja2V0U1NMUmVxdWVzdHNPbmx5LFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgUzMgc3RhdGljIHdlYnNpdGUgYnVja2V0IGVpdGhlciBoYXMgYW4gb3BlbiB3b3JsZCBidWNrZXQgcG9saWN5IG9yIGRvZXMgbm90IHVzZSBhIENsb3VkRnJvbnQgT3JpZ2luIEFjY2VzcyBJZGVudGl0eSAoT0FJKSBpbiB0aGUgYnVja2V0IHBvbGljeSBmb3IgbGltaXRlZCBnZXRPYmplY3QgYW5kL29yIHB1dE9iamVjdCBwZXJtaXNzaW9ucy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiQW4gT0FJIGFsbG93cyB5b3UgdG8gcHJvdmlkZSBhY2Nlc3MgdG8gY29udGVudCBpbiB5b3VyIFMzIHN0YXRpYyB3ZWJzaXRlIGJ1Y2tldCB0aHJvdWdoIENsb3VkRnJvbnQgVVJMcyB3aXRob3V0IGVuYWJsaW5nIHB1YmxpYyBhY2Nlc3MgdGhyb3VnaCBhbiBvcGVuIGJ1Y2tldCBwb2xpY3ksIGRpc2FibGluZyBTMyBCbG9jayBQdWJsaWMgQWNjZXNzIHNldHRpbmdzLCBhbmQvb3IgdGhyb3VnaCBvYmplY3QgQUNMcy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3Igd2hvIGhhcyBkaXNjb3ZlcmVkIGEgUzMgaG9zdGVkIHdlYnNpdGUgY2FuIGRpc2NvdmVyIHByb3RvdHlwZSB3ZWIgYXNzZXRzIHRoYXQgYXJlIGhvc3RlZCBvbiB0aGUgd2Vic2l0ZSwgd2hpY2ggbWF5IGxlYWQgdG8gcmVjb24gYW5kIGludHJ1c2lvbiBhY3Rpdml0aWVzIGJlaW5nIHBlcmZvcm1lZCBhZ2FpbnN0IHRoZSBleHBvc2VkIGF0dGFjayBzdXJmYWNlIHBvc3NpYmx5IGltcGFjdGluZyB0aGUgY29uZmlkZW50aWFsaXR5LCBpbnRlZ3JpdHkgYW5kIGF2YWlsYWJpbGl0eSBvZiB0aGUgZGF0YSBhbmQgcmVzb3VyY2UgYXNzZXRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcHJvdG90eXBlXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICBydWxlOiBydWxlcy5zMy5TM1dlYkJ1Y2tldE9BSUFjY2VzcyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFNhZ2VNYWtlciBub3RlYm9vayBpbnN0YW5jZSBkb2VzIG5vdCBoYXZlIGFuIGVuY3J5cHRlZCBzdG9yYWdlIHZvbHVtZS5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiRW5jcnlwdGluZyBzdG9yYWdlIHZvbHVtZXMgaGVscHMgcHJvdGVjdCBTYWdlTWFrZXIgZGF0YS1hdC1yZXN0LlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5zYWdlbWFrZXIuU2FnZU1ha2VyTm90ZWJvb2tJbnN0YW5jZUtNU0tleUNvbmZpZ3VyZWQsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBTYWdlTWFrZXIgbm90ZWJvb2sgaW5zdGFuY2UgaXMgbm90IHByb3Zpc2lvbmVkIGluc2lkZSBhIFZQQy5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiUHJvdmlzaW9uaW5nIHRoZSBub3RlYm9vayBpbnN0YW5jZXMgaW5zaWRlIGEgVlBDIGVuYWJsZXMgdGhlIG5vdGVib29rIHRvIGFjY2VzcyBWUEMtb25seSByZXNvdXJjZXMgc3VjaCBhcyBFRlMgZmlsZSBzeXN0ZW1zLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5zYWdlbWFrZXIuU2FnZU1ha2VyTm90ZWJvb2tJblZQQyxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFNhZ2VNYWtlciBub3RlYm9vayBkb2VzIG5vdCBkaXNhYmxlIGRpcmVjdCBpbnRlcm5ldCBhY2Nlc3MuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIkJ5IHByZXZlbnRpbmcgZGlyZWN0IGludGVybmV0IGFjY2VzcywgeW91IGNhbiBrZWVwIHNlbnNpdGl2ZSBkYXRhIGZyb20gYmVpbmcgYWNjZXNzZWQgYnkgdW5hdXRob3JpemVkIHVzZXJzLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5zYWdlbWFrZXIuU2FnZU1ha2VyTm90ZWJvb2tOb0RpcmVjdEludGVybmV0QWNjZXNzLFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgc2VjcmV0IGRvZXMgbm90IGhhdmUgYXV0b21hdGljIHJvdGF0aW9uIHNjaGVkdWxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiUm90YXRpbmcgc2VjcmV0cyBvbiBhIHJlZ3VsYXIgc2NoZWR1bGUgY2FuIHNob3J0ZW4gdGhlIHBlcmlvZCBhIHNlY3JldCBpcyBhY3RpdmUsIGFuZCBwb3RlbnRpYWxseSByZWR1Y2UgdGhlIGJ1c2luZXNzIGltcGFjdCBpZiB0aGUgc2VjcmV0IGlzIGNvbXByb21pc2VkLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5zZWNyZXRzbWFuYWdlci5TZWNyZXRzTWFuYWdlclJvdGF0aW9uRW5hYmxlZCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFNOUyB0b3BpYyBkb2VzIG5vdCBoYXZlIEtNUyBlbmNyeXB0aW9uIGVuYWJsZWQuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIlRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSB0aGF0IHlvdXIgQW1hem9uIFNpbXBsZSBOb3RpZmljYXRpb24gU2VydmljZSAoQW1hem9uIFNOUykgdG9waWNzIHJlcXVpcmUgZW5jcnlwdGlvbiB1c2luZyBBV1MgS2V5IE1hbmFnZW1lbnQgU2VydmljZSAoQVdTIEtNUykuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gcHVibGlzaGVkIG1lc3NhZ2VzLCBlbmFibGUgZW5jcnlwdGlvbiBhdCByZXN0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnNucy5TTlNFbmNyeXB0ZWRLTVMsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBTTlMgVG9waWMgZG9lcyBub3QgcmVxdWlyZSBwdWJsaXNoZXJzIHRvIHVzZSBTU0wuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIldpdGhvdXQgSFRUUFMgKFRMUyksIGEgbmV0d29yay1iYXNlZCBhdHRhY2tlciBjYW4gZWF2ZXNkcm9wIG9uIG5ldHdvcmsgdHJhZmZpYyBvciBtYW5pcHVsYXRlIGl0LCB1c2luZyBhbiBhdHRhY2sgc3VjaCBhcyBtYW4taW4tdGhlLW1pZGRsZS4gQWxsb3cgb25seSBlbmNyeXB0ZWQgY29ubmVjdGlvbnMgb3ZlciBIVFRQUyAoVExTKSB1c2luZyB0aGUgYXdzOlNlY3VyZVRyYW5zcG9ydCBjb25kaXRpb24gYW5kIHRoZSAnc25zOlB1Ymxpc2gnIGFjdGlvbiBpbiB0aGUgdG9waWMgcG9saWN5IHRvIGZvcmNlIHB1Ymxpc2hlcnMgdG8gdXNlIFNTTC4gSWYgU1NFIGlzIGFscmVhZHkgZW5hYmxlZCB0aGVuIHRoaXMgY29udHJvbCBpcyBhdXRvIGVuZm9yY2VkLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5zbnMuU05TVG9waWNTU0xQdWJsaXNoT25seSxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFNRUyBRdWV1ZSBkb2VzIG5vdCBoYXZlIHNlcnZlci1zaWRlIGVuY3J5cHRpb24gZW5hYmxlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiU2VydmVyIHNpZGUgZW5jcnlwdGlvbiBhZGRzIGFkZGl0aW9uYWwgcHJvdGVjdGlvbiBvZiBzZW5zaXRpdmUgZGF0YSBkZWxpdmVyZWQgYXMgbWVzc2FnZXMgdG8gc3Vic2NyaWJlcnMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnNxcy5TUVNRdWV1ZVNTRSxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiVGhlIFNRUyBxdWV1ZSBkb2VzIG5vdCByZXF1aXJlIHJlcXVlc3RzIHRvIHVzZSBTU0wuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIldpdGhvdXQgSFRUUFMgKFRMUyksIGEgbmV0d29yay1iYXNlZCBhdHRhY2tlciBjYW4gZWF2ZXNkcm9wIG9uIG5ldHdvcmsgdHJhZmZpYyBvciBtYW5pcHVsYXRlIGl0LCB1c2luZyBhbiBhdHRhY2sgc3VjaCBhcyBtYW4taW4tdGhlLW1pZGRsZS4gQWxsb3cgb25seSBlbmNyeXB0ZWQgY29ubmVjdGlvbnMgb3ZlciBIVFRQUyAoVExTKSB1c2luZyB0aGUgYXdzOlNlY3VyZVRyYW5zcG9ydCBjb25kaXRpb24gaW4gdGhlIHF1ZXVlIHBvbGljeSB0byBmb3JjZSByZXF1ZXN0cyB0byB1c2UgU1NMLlwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICBydWxlOiBydWxlcy5zcXMuU1FTUXVldWVTU0xSZXF1ZXN0c09ubHksXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBUaW1lc3RyZWFtIGRhdGFiYXNlIGRvZXMgbm90IHVzZSBhIEN1c3RvbWVyIE1hbmFnZWQgS01TIEtleSBmb3IgYXQgcmVzdCBlbmNyeXB0aW9uLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJBbGwgVGltZXN0cmVhbSB0YWJsZXMgaW4gYSBkYXRhYmFzZSBhcmUgZW5jcnlwdGVkIGF0IHJlc3QgYnkgZGVmYXVsdCB1c2luZyBhbiBBV1MgTWFuYWdlZCBLZXkuIFRoZXNlIGtleXMgYXJlIHJvdGF0ZWQgZXZlcnkgdGhyZWUgeWVhcnMuIERhdGEgYXQgcmVzdCBtdXN0IGJlIGVuY3J5cHRlZCB1c2luZyBDTUtzIGlmIHlvdSByZXF1aXJlIG1vcmUgY29udHJvbCBvdmVyIHRoZSBwZXJtaXNzaW9ucyBhbmQgbGlmZWN5Y2xlIG9mIHlvdXIga2V5cywgaW5jbHVkaW5nIHRoZSBhYmlsaXR5IHRvIGhhdmUgdGhlbSBhdXRvbWF0aWNhbGx5IHJvdGF0ZWQgb24gYW4gcGVyaW9kaWMgYmFzaXMuXCIsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgIHJ1bGU6IHJ1bGVzLnRpbWVzdHJlYW0uVGltZXN0cmVhbURhdGFiYXNlQ3VzdG9tZXJNYW5hZ2VkS2V5LFxuICB9LFxuICB7XG4gICAgaW5mbzogXCJUaGUgVlBDJ3MgZGVmYXVsdCBzZWN1cml0eSBncm91cCBhbGxvd3MgaW5ib3VuZCBvciBvdXRib3VuZCB0cmFmZmljLlwiLFxuICAgIGV4cGxhbmF0aW9uOlxuICAgICAgXCJXaGVuIGNyZWF0aW5nIGEgVlBDIHRocm91Z2ggQ2xvdWRGb3JtYXRpb24sIHRoZSBkZWZhdWx0IHNlY3VyaXR5IGdyb3VwIHdpbGwgYWx3YXlzIGJlIG9wZW4uIFRoZXJlZm9yZSBpdCBpcyBpbXBvcnRhbnQgdG8gYWx3YXlzIGNsb3NlIHRoZSBkZWZhdWx0IHNlY3VyaXR5IGdyb3VwIGFmdGVyIHN0YWNrIGNyZWF0aW9uIHdoZW5ldmVyIGEgVlBDIGlzIGNyZWF0ZWQuIFJlc3RyaWN0aW5nIGFsbCB0aGUgdHJhZmZpYyBvbiB0aGUgZGVmYXVsdCBzZWN1cml0eSBncm91cCBoZWxwcyBpbiByZXN0cmljdGluZyByZW1vdGUgYWNjZXNzIHRvIHlvdXIgQVdTIHJlc291cmNlcy5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3IgY2FuIGRpc2NvdmVyIFZQQy1hdHRhY2hlZCByZXNvdXJjZXMgKGUuZy4gRUMyIGluc3RhbmNlcykgdGhhdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMgYW5kIGFsbG93IGluZ3Jlc3MgdG8gYWxsIGludGVybmV0IGFkZHJlc3Mgb3IgbW92ZSBsYXRlcmFsbHkgdG8gbm9uLXB1YmxpYyBWUEMtYXR0YWNoZWQgaW5zdGFuY2VzLCB3aGljaCBtYXkgbGVhZCB0byByZWNvbm5haXNzYW5jZSBhbmQgaW50cnVzaW9uIGFjdGl2aXRpZXMgYmVpbmcgcGVyZm9ybWVkIGFnYWluc3QgdGhlIGV4cG9zZWQgYXR0YWNrIHN1cmZhY2UgcG9zc2libHkgaW1wYWN0aW5nIHRoZSBjb25maWRlbnRpYWxpdHksIGludGVncml0eSBhbmQgYXZhaWxhYmlsaXR5IG9mIHRoZSBkYXRhIGFuZCByZXNvdXJjZSBhc3NldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm90b3R5cGVcIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgIHJ1bGU6IHJ1bGVzLnZwYy5WUENEZWZhdWx0U2VjdXJpdHlHcm91cENsb3NlZCxcbiAgfSxcbiAge1xuICAgIGluZm86IFwiQSBOZXR3b3JrIEFDTCBvciBOZXR3b3JrIEFDTCBlbnRyeSBoYXMgYmVlbiBpbXBsZW1lbnRlZC5cIixcbiAgICBleHBsYW5hdGlvbjpcbiAgICAgIFwiTmV0d29yayBBQ0xzIHNob3VsZCBiZSB1c2VkIHNwYXJpbmdseSBmb3IgdGhlIGZvbGxvd2luZyByZWFzb25zOiB0aGV5IGNhbiBiZSBjb21wbGV4IHRvIG1hbmFnZSwgdGhleSBhcmUgc3RhdGVsZXNzLCBldmVyeSBJUCBhZGRyZXNzIG11c3QgYmUgZXhwbGljaXRseSBvcGVuZWQgaW4gZWFjaCAoaW5ib3VuZC9vdXRib3VuZCkgZGlyZWN0aW9uLCBhbmQgdGhleSBhZmZlY3QgYSBjb21wbGV0ZSBzdWJuZXQuIFVzZSBzZWN1cml0eSBncm91cHMgd2hlbiBwb3NzaWJsZSBhcyB0aGV5IGFyZSBzdGF0ZWZ1bCBhbmQgZWFzaWVyIHRvIG1hbmFnZS5cIixcbiAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgcnVsZTogcnVsZXMudnBjLlZQQ05vTkFDTHMsXG4gIH0sXG4gIHtcbiAgICBpbmZvOiBcIlRoZSBzdWJuZXQgYXV0by1hc3NpZ25zIHB1YmxpYyBJUCBhZGRyZXNzZXMuXCIsXG4gICAgZXhwbGFuYXRpb246XG4gICAgICBcIk1hbmFnZSBhY2Nlc3MgdG8gdGhlIEFXUyBDbG91ZCBieSBlbnN1cmluZyBBbWF6b24gVmlydHVhbCBQcml2YXRlIENsb3VkIChWUEMpIHN1Ym5ldHMgYXJlIG5vdCBhdXRvbWF0aWNhbGx5IGFzc2lnbmVkIGEgcHVibGljIElQIGFkZHJlc3MuIEFtYXpvbiBFbGFzdGljIENvbXB1dGUgQ2xvdWQgKEVDMikgaW5zdGFuY2VzIHRoYXQgYXJlIGxhdW5jaGVkIGludG8gc3VibmV0cyB0aGF0IGhhdmUgdGhpcyBhdHRyaWJ1dGUgZW5hYmxlZCBoYXZlIGEgcHVibGljIElQIGFkZHJlc3MgYXNzaWduZWQgdG8gdGhlaXIgcHJpbWFyeSBuZXR3b3JrIGludGVyZmFjZS5cXG5cXG5FeGFtcGxlIHRocmVhdDogQSBnbG9iYWwgaW50ZXJuZXQtYmFzZWQgYWN0b3IgY2FuIGRpc2NvdmVyIFZQQy1hdHRhY2hlZCByZXNvdXJjZXMgKGUuZy4gRUMyIGluc3RhbmNlcykgd2l0aGluIHRoZSBzdWJuZXQgaW4gcXVlc3Rpb24gdGhhdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMsIHdoaWNoIG1heSBsZWFkIHRvIHJlY29ubmFpc3NhbmNlIGFuZCBpbnRydXNpb24gYWN0aXZpdGllcyBiZWluZyBwZXJmb3JtZWQgYWdhaW5zdCB0aGUgZXhwb3NlZCBhdHRhY2sgc3VyZmFjZSBwb3NzaWJseSBpbXBhY3RpbmcgdGhlIGNvbmZpZGVudGlhbGl0eSwgaW50ZWdyaXR5IGFuZCBhdmFpbGFiaWxpdHkgb2YgdGhlIGRhdGEgYW5kIHJlc291cmNlIGFzc2V0cyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb3RvdHlwZVwiLFxuICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgcnVsZTogcnVsZXMudnBjLlZQQ1N1Ym5ldEF1dG9Bc3NpZ25QdWJsaWNJcERpc2FibGVkLFxuICB9LFxuXTtcbiJdfQ==