"""
A machine learning model on aidkit can cover multiple model versions.

Machine learning models must be configured using the web GUI. After a
model has been configured, model versions can be uploaded using the
python client.
"""
import io
import os
import string
from pathlib import Path
from typing import Any, BinaryIO, Dict, List, Literal, Optional, Tuple, Union

from tqdm import tqdm

from aidkit_client._endpoints.ml_models import MLModelsAPI
from aidkit_client._endpoints.models import (
    AutogeneratedClasses,
    BatchDim,
    BoundingBoxDimensionsTypes,
    BPETokenizerSpecs,
    ByteLevelPreTokenizerSpecs,
    ClassificationModelOutputSpecs,
    ClassificationScoreDimensionsModel,
    ColorChannelDimensionModel,
    ColorImageInputDimensionsModel,
    ColorImageInputSpecs,
    DetectionDictOutputFormat,
    DetectionModelOutputSpecs,
    DetectionScoreDimensionsTypes,
    DictOutputFormat,
    FrameworkType,
    GrayScaleChannelDimensionModel,
    GrayScaleImageInputDimensionsModel,
    GrayScaleImageInputSpecs,
    ImageModelContextSpecs,
    MLModelResponse,
    MLModelVersionResponse,
)
from aidkit_client._endpoints.models import NMSPostProcessor as CoreNMSPostProcessor
from aidkit_client._endpoints.models import (
    NoPreTokenizerSpecs,
    NoTokenizerSpecs,
    SegmentationModelOutputSpecs,
    SegmentationScoreDimensionsModel,
    SequenceOutputFormat,
    SingleOutputFormat,
    SpecificLengthDimensionModel,
    SplitPreTokenizerSpecs,
    TextInputDimensionsModel,
    TextInputSpecs,
    TextModelContextSpecs,
    UnspecifiedLengthDimensionModel,
    WordLevelTokenizerSpecs,
)
from aidkit_client.aidkit_api import HTTPService
from aidkit_client.configuration import get_api_client
from aidkit_client.exceptions import ResourceWithNameNotFoundError
from aidkit_client.resources.ml_model_input_configuration import (
    ByteLevel,
    BytePairEncoding,
    GrayscaleCustomScaler,
    ImageInputConfig,
    ImageNetPresetScaler,
    MinusOneToOneScaler,
    RGBCustomScaler,
    Split,
    WordLevel,
    ZeroToOneScaler,
)
from aidkit_client.resources.ml_model_output_formats import (
    _BOUNDING_BOXES_FORMATS,
    ClassificationOutputFormat,
    DetectionOutputFormat,
    DictionaryOfTensors,
    SegmentationOutputFormat,
)

TF_EXT = ["h5", "gz", "tar", "zip"]
TORCH_EXT = ["pt", "ptc", "torchscript"]


def dim_order_str_to_axis_dict(
    dim_order: Literal["hwc", "whc", "chw", "cwh", "hcw", "wch"],  # allow all permutations of 'hwc'
) -> Dict[str, int]:
    """
    Transform dimension order string to dict of axis indices.

    :param dim_order: Any possible dimension order given as permutation
        of the string 'hwc'.
    :return: Dictionary with keys being ['height', 'width', 'channel']
        and values being the corresponding axis indices.
    """
    height_axis = dim_order.index("h")
    width_axis = dim_order.index("w")
    channel_axis = dim_order.index("c")

    return {
        "height": height_axis,
        "width": width_axis,
        "channel": channel_axis,
    }


class MLModelVersion:
    """
    Version of a machine learning model.

    An instance of this class references a model version which is
    uploaded to an aidkit server.
    """

    def __init__(
        self, api_service: HTTPService, model_version_response: MLModelVersionResponse
    ) -> None:
        """
        Create a new instance from the server response.

        :param api_service: Service instance to use for the communication with the
            server.
        :param model_version_response: Server response describing the model
            version to be uploaded.
        """
        self._api_service = api_service
        self._model_version_response = model_version_response

    @staticmethod
    def model_to_file_object(model: Any) -> Tuple[BinaryIO, str]:
        """
        Convert the passed model to binary.

        :param model: Can be a string path to the model file, a torchscript model or a
            TF model. This parameter value is typed as Any so we can avoid additional
            imports of either torch or tensorflow.
        :return: Model binary and file name.
        :raises ImportError: When unsupported model is passed.
        :raises ValueError: When torch.nn.Module is passed.
        """
        if isinstance(model, str):
            if model.split(".")[-1] not in TF_EXT + TORCH_EXT:
                raise ImportError(
                    "Unsupported file format. "
                    f"The supported formats are Torch: {TORCH_EXT}, TF: {TF_EXT}"
                )
            return open(model, "rb"), model  # pylint: disable=consider-using-with
        try:
            import torch  # pylint: disable=import-outside-toplevel

            if isinstance(model, torch.jit.ScriptModule):
                bio = io.BytesIO()
                torch.jit.save(model, bio)
                bio.seek(0, os.SEEK_SET)
                return bio, "temp.ptc"

            if isinstance(model, torch.nn.Module):
                ValueError(
                    f"""ML Models of type {type(model)} are not supported.
                Please convert the torch model to a torch.jit.ScriptModule by
                calling torch.jit.trace(model, example_input), see
                https://pytorch.org/docs/stable/jit.html"""
                )
        except ImportError:
            pass
        try:
            import h5py  # pylint: disable=import-outside-toplevel
            import tensorflow as tf  # pylint: disable=import-outside-toplevel

            if isinstance(model, tf.keras.Model):
                bio = io.BytesIO()
                with h5py.File(bio, "w") as file:
                    model.save(file)
                bio.seek(0, os.SEEK_SET)
                return bio, "temp.h5"
            from tensorflow.python.trackable.base import (  # pylint:disable=C0415
                Trackable,
            )

            # TF objects that contain a trackable state. Such objects inherit from the
            # Trackable class and can be saved using tf.saved_model.save(obj, dir)
            if isinstance(model, Trackable):
                tmp = bytes()
                tf.saved_model.save(model, tmp)
                return io.BytesIO(tmp), "temp.tf2"

        except ImportError:
            pass

        raise ValueError(f"ML Model type {type(model)} not understood.")

    @classmethod
    async def upload(
        cls,
        model_version: Any,
        model: Union[int, "MLModel"],
        model_version_name: str,
        progress_bar: bool = False,
    ) -> "MLModelVersion":
        """
        Upload a new model version.

        :param model_version: Model version to upload. How this parameter is interpreted
            depends on its type:

            * If the parameter is a string, it is interpreted as the path of the model \
                file to be uploaded.
            * If the parameter is an instance of a keras model, the model is saved and \
                the resulting file is uploaded.
            * If the parameter is a torch ``ScriptModule``, the ``ScriptModule`` is \
                saved and the resulting file is uploaded.

        :param model: Instance of the model or ID of the model to upload a new version of.
        :param model_version_name: Name of the new model version.
        :param progress_bar: Whether to display a progress bar while uploading.
        :return: Instance representing the newly uploaded model version.
        """
        api_service = get_api_client()
        model_file_object, file_name = cls.model_to_file_object(model_version)
        model_file_object.seek(0, os.SEEK_END)
        model_file_size = model_file_object.tell()
        model_file_object.seek(0, 0)

        with tqdm.wrapattr(
            model_file_object,
            "read",
            total=model_file_size,
            desc="Uploading ML Model Version",
            disable=not progress_bar,
        ) as wrapped_file_object:
            model_version_response = await MLModelsAPI(api_service).upload_model_version(
                model_id=model.id if isinstance(model, MLModel) else model,
                model_version_name=model_version_name,
                model_file_content=wrapped_file_object,
                file_name=file_name,
            )

        return MLModelVersion(
            api_service=api_service, model_version_response=model_version_response
        )

    @classmethod
    async def get_by_id(cls, model_id: int, model_version_id: int) -> "MLModelVersion":
        """
        Get an ML model version by its ID.

        :param model_id: ID of the model to get the version from.
        :param model_version_id: ID of the model version to get.

        :return: Instance of MLModelVersion.
        :raises ResourceWithNameNotFoundError: When model version is not found.
        """
        api_service = get_api_client()
        model_version_response_list = await MLModelsAPI(api_service).get_all_model_versions(
            model_id
        )
        try:
            (model_version_response,) = [
                model_version_response
                for model_version_response in model_version_response_list
                if model_version_response.id == model_version_id
            ]
        except ValueError as wrong_entry_number:
            available_versions = ", ".join(
                str(model_version_response.id)
                for model_version_response in model_version_response_list
            )
            raise ResourceWithNameNotFoundError(
                f"Ml Model Version with ID '{model_version_id}' not found. "
                f"Existing Ml Model Versions: '[{available_versions}]'."
            ) from wrong_entry_number

        return MLModelVersion(
            api_service=api_service, model_version_response=model_version_response
        )

    @property
    def id(self) -> int:
        """
        Get the ID by which the model version instance is referenced on the
        server.

        :return: ID of the model version instance.
        """
        return self._model_version_response.id

    @property
    def name(self) -> str:
        """
        Get the name of the model version instance.

        :return: Name of the model version instance.
        """
        return self._model_version_response.name

    @classmethod
    async def delete(cls, model_id: int, model_version_id: int) -> None:
        """
        Delete a model version.

        :param model_id: ID of the model to delete a version from.
        :param model_version_id: ID of the model version.

        :return: None.
        """
        api_service = get_api_client()
        await MLModelsAPI(api_service).delete_model_version(model_id, model_version_id)


class MLModel:
    """
    A machine learning model.

    An instance of this class references a model which has been created
    on an aidkit server.
    """

    def __init__(self, api_service: HTTPService, model_response: MLModelResponse) -> None:
        """
        Create a new instance from the server response.

        :param api_service: Service instance to use for the communication with the
            server.
        :param model_response: Server response describing the model
            to be created.
        """
        self._api_service = api_service
        self._model_response = model_response

    @property
    def id(self) -> int:
        """
        Get the ID by which the model instance is referenced on the server.

        :return: ID of the model instance.
        """
        return self._model_response.id

    @property
    def name(self) -> str:
        """
        Get the name of the model instance.

        :return: Name of the model instance.
        """
        return self._model_response.name

    @staticmethod
    def _validate_name(name: str) -> str:
        """
        Validate whether `name` contains only allowed characters.

        :param name: Name to validate.
        :returns: Validated name.
        :raises ValueError: If the name is not valid.
        """
        allowed_chars = set(string.ascii_letters + string.digits + "-")
        used_chars = set(name)

        if used_chars.issubset(allowed_chars):
            return name

        raise ValueError(
            f"The provided name {name} is not valid. "
            f"It should only contain latin letters, digits and dashes '-'. "
            f"Whitespaces and other symbols are not allowed."
        )

    @staticmethod
    def _get_class_names(
        class_names: Union[int, Path, List[str]]
    ) -> Union[List[str], AutogeneratedClasses]:
        """
        Get the class names by parsing a file or autogenerating them if needed.

        :param class_names: Either the number of classes, the path to the file
            containing the class names or a list of class names as strings.
        :returns: Class names as a list of strings or an instance of
            `AutogeneratedClasses`.
        """
        if isinstance(class_names, Path):
            with open(class_names, mode="r", encoding="utf-8") as file_pointer:
                return [name.strip() for name in file_pointer.read().split(",")]
        if isinstance(class_names, int):
            return AutogeneratedClasses(num_classes=class_names)

        return class_names

    @staticmethod
    async def get_all() -> List["MLModel"]:
        """
        Get all ML Models.

        :return: List of MLModels
        """
        api_service = get_api_client()
        return [
            MLModel(api_service, response)
            for response in await MLModelsAPI(api=api_service).get_all()
        ]

    async def get_all_model_versions(self) -> List[MLModelVersion]:
        """
        Get all versions of this ML Model.

        :return: List of MLModelVersions
        """
        api_service = get_api_client()
        return [
            MLModelVersion(api_service, response)
            for response in await MLModelsAPI(api=api_service).get_all_model_versions(self.id)
        ]

    @classmethod
    async def delete(cls, model_id: int) -> None:
        """
        Delete an ML model.

        :param model_id: Model ID.
        :return: None.
        """
        await MLModelsAPI(get_api_client()).delete(model_id=model_id)


class ClassificationOutputMixin:
    """
    Mixin to add classification output functionality to a class derived from
    MLModel.
    """

    @staticmethod
    def _build_classification_output_specs(
        class_names_parsed: Union[List[str], AutogeneratedClasses],
        output_format: ClassificationOutputFormat,
    ) -> ClassificationModelOutputSpecs:
        """
        Build classification output specifications containing information about
        the class names and the model output format.

        :param class_names_parsed: Either the class names as a list of strings or an
            instance of `AutogeneratedClasses`.
        :param output_format: Output format of the model.
        :returns: Output specification object for a classification model context.
        """
        dimensions = ClassificationScoreDimensionsModel(
            class_=UnspecifiedLengthDimensionModel(
                axis=0,
            ),
        )

        if isinstance(output_format.format, DictionaryOfTensors):
            format_type: Union[SingleOutputFormat, DictOutputFormat] = DictOutputFormat(
                output_target=output_format.format.output_target
            )
        else:
            format_type = SingleOutputFormat()

        return ClassificationModelOutputSpecs(
            class_names=class_names_parsed,
            output_format=format_type,
            dimensions=dimensions,
        )


class ImageModel(ClassificationOutputMixin, MLModel):
    """
    A machine learning model for image tasks.

    An instance of this class references a model which has been created
    on an aidkit server.
    """

    @staticmethod
    def _build_input_specs(
        image_input_config: ImageInputConfig,
        batch: Optional[Literal["first", "last"]] = None,
        image_scaler: Optional[
            Union[
                ZeroToOneScaler,
                MinusOneToOneScaler,
                ImageNetPresetScaler,
                RGBCustomScaler,
                GrayscaleCustomScaler,
            ]
        ] = None,
    ) -> Union[GrayScaleImageInputSpecs, ColorImageInputSpecs]:
        """
        Build input specifications for the create method of child classes.

        :param image_input_config: Image input configuration containing information
            about the image type and axis order.
        :param batch: Indicates the position and existence of a batch axis.
        :param image_scaler: Optional image scaler.
        :returns: Output specification object for a model context specification.
        """
        batch_dim = BatchDim("No Batch")
        if batch == "first":
            batch_dim = BatchDim("First")
        elif batch == "last":
            batch_dim = BatchDim("Last")

        dim_dict = dim_order_str_to_axis_dict(dim_order=image_input_config.input_dim_order)

        height_channel_dimension_model = SpecificLengthDimensionModel(
            axis=dim_dict["height"], length=image_input_config.height
        )
        width_channel_dimension_model = SpecificLengthDimensionModel(
            axis=dim_dict["width"], length=image_input_config.width
        )

        if not image_scaler:
            image_scaler = ZeroToOneScaler()

        if image_input_config.image_type == "grayscale":
            channel_gray = GrayScaleChannelDimensionModel(axis=dim_dict["channel"])
            dimensions_gray = GrayScaleImageInputDimensionsModel(
                height=height_channel_dimension_model,
                width=width_channel_dimension_model,
                channel=channel_gray,
            )

            return GrayScaleImageInputSpecs(
                batch_dim=batch_dim,
                dimensions=dimensions_gray,
                image_scaler=image_scaler,
            )

        channel_color = ColorChannelDimensionModel(axis=dim_dict["channel"])
        dimensions_color = ColorImageInputDimensionsModel(
            height=height_channel_dimension_model,
            width=width_channel_dimension_model,
            channel=channel_color,
        )
        return ColorImageInputSpecs(
            batch_dim=batch_dim,
            dimensions=dimensions_color,
            image_scaler=image_scaler,
        )

    @staticmethod
    def _build_segmentation_output_specs(
        class_names_parsed: Union[List[str], AutogeneratedClasses],
        output_format: SegmentationOutputFormat,
    ) -> SegmentationModelOutputSpecs:
        """
        Build image segmentation output specifications containing information
        about the class names and the model output format.

        :param class_names_parsed: Either the class names as a list of strings or an
            instance of `AutogeneratedClasses`.
        :param output_format: Output format of the model.
        :returns: Output specification object for an image segmentation model context.
        """
        dim_dict = dim_order_str_to_axis_dict(output_format.output_dim_order)

        dimensions = SegmentationScoreDimensionsModel(
            class_=UnspecifiedLengthDimensionModel(
                axis=dim_dict["channel"],
            ),
            height=UnspecifiedLengthDimensionModel(
                axis=dim_dict["height"],
            ),
            width=UnspecifiedLengthDimensionModel(
                axis=dim_dict["width"],
            ),
        )

        if isinstance(output_format.format, DictionaryOfTensors):
            format_type: Union[SingleOutputFormat, DictOutputFormat] = DictOutputFormat(
                output_target=output_format.format.output_target
            )
        else:
            format_type = SingleOutputFormat()

        return SegmentationModelOutputSpecs(
            class_names=class_names_parsed,
            output_format=format_type,
            dimensions=dimensions,
        )

    @staticmethod
    def _build_detection_output_specs(
        class_names_parsed: Union[List[str], AutogeneratedClasses],
        output_format: DetectionOutputFormat,
    ) -> DetectionModelOutputSpecs:
        """
        Build output specifications based on class names and output format.

        :param class_names_parsed: Optional list of class name strings.
            an object of type `AutogeneratedClasses`.
        :param output_format: `DetectionOutputFormat` descibing the format that the model
            output will respect.
        :returns: Specification object for a model context specification.
        """
        score_dimension_type = (
            DetectionScoreDimensionsTypes.boxes_x_classes
            if output_format.scores_shape == "boxes_classes"
            else DetectionScoreDimensionsTypes.classes_x_boxes
        )
        box_dimension_type = (
            BoundingBoxDimensionsTypes.boxes_x_coordinates
            if output_format.boxes.shape == "boxes_coordinates"
            else BoundingBoxDimensionsTypes.coordinates_x_boxes
        )

        parsed_output_format: Union[DetectionDictOutputFormat, SequenceOutputFormat]

        if output_format.format == "dictionary":
            parsed_output_format = DetectionDictOutputFormat(
                scores_key=output_format.scores_identifier,
                boxes_key=output_format.boxes.identifier,
                score_dimension_type=score_dimension_type,
                box_dimension_type=box_dimension_type,
            )
        else:
            parsed_output_format = SequenceOutputFormat(
                scores_key=output_format.scores_identifier,
                boxes_key=output_format.boxes.identifier,
                score_dimension_type=score_dimension_type,
                box_dimension_type=box_dimension_type,
            )

        coordinate_format = _BOUNDING_BOXES_FORMATS[output_format.boxes.format]

        post_processor: Optional[CoreNMSPostProcessor] = None
        if output_format.post_processing:
            post_processor = CoreNMSPostProcessor(**output_format.post_processing.dict())

        return DetectionModelOutputSpecs(
            class_names=class_names_parsed,
            output_format=parsed_output_format,
            coordinate_format=coordinate_format,
            post_processor=post_processor,
        )

    @classmethod
    async def create(
        cls,
        model: Any,
        model_name: str,
        model_version_name: str,
        framework: Literal["TensorFlow", "PyTorch"],
        class_names: Union[int, List[str], Path],
        image_input_config: ImageInputConfig,
        output_format: Union[
            ClassificationOutputFormat, SegmentationOutputFormat, DetectionOutputFormat
        ],
        batch_axis: Optional[Literal["first", "last"]] = None,
        image_scaler: Optional[
            Union[
                ZeroToOneScaler,
                MinusOneToOneScaler,
                ImageNetPresetScaler,
                RGBCustomScaler,
                GrayscaleCustomScaler,
            ]
        ] = None,
    ) -> Tuple["ImageModel", MLModelVersion]:
        """
        Create and upload a new image model. Currently supports image
        classification and segmentation models.

        :param model: Model to upload. How this parameter is interpreted
            depends on its type:

            * If the parameter is a string, it is interpreted as the path of the model \
                file to be uploaded.
            * If the parameter is an instance of a keras model, the model is saved and \
                the resulting file is uploaded.
            * If the parameter is a torch ``ScriptModule``, the ``ScriptModule`` is \
                saved and the resulting file is uploaded.

        :param model_name: Name of the model to create.
        :param model_version_name: Name of the first model version.
        :param framework: Framework of the model, either "TensorFlow" or "PyTorch".
        :param class_names: Names of the classes. How this parameter is interpreted
            depends on its type:

            * If the parameter is a string, it is interpreted as the path of a file \
            containing the comma separated list of class names.
            * If the parameter is a list of strings, each string is interpreted as a \
            class name.
            * If the parameter is an integer, it is interpreted as the number of \
            classes and the names will be autogenerated.

        :param image_input_config: Image input configuration.
        :param output_format: Model output format.
        :param batch_axis: Optional string indicating the batch dimension, either
            "first" or "last".
        :param image_scaler: Optional image scaler.
        :return: Tuple containing the newly created model and the corresponding first
            model version.
        :raises ValueError: If the output format doesn't correspond to an image
            classification or image segmentation model.
        """
        api_service = get_api_client()

        model_name = cls._validate_name(model_name)
        model_version_name = cls._validate_name(model_version_name)

        class_names_parsed = cls._get_class_names(class_names=class_names)

        input_specs = cls._build_input_specs(
            batch=batch_axis, image_input_config=image_input_config, image_scaler=image_scaler
        )

        output_specs: Union[
            ClassificationModelOutputSpecs, SegmentationModelOutputSpecs, DetectionModelOutputSpecs
        ]
        if isinstance(output_format, ClassificationOutputFormat):
            output_specs = cls._build_classification_output_specs(
                class_names_parsed=class_names_parsed, output_format=output_format
            )
        elif isinstance(output_format, SegmentationOutputFormat):
            output_specs = cls._build_segmentation_output_specs(
                class_names_parsed=class_names_parsed, output_format=output_format
            )
        elif isinstance(output_format, DetectionOutputFormat):
            output_specs = cls._build_detection_output_specs(
                class_names_parsed=class_names_parsed, output_format=output_format
            )
        else:
            raise ValueError(
                "The provided model output format is not valid. "
                "The aidkit Python client currently only supports the "
                "creation of image classification, image segmentation "
                "and object detection tasks."
            )

        image_model_context_specs = ImageModelContextSpecs(
            input_specification=input_specs,
            output_specification=output_specs,
        )

        model_file_object, _ = MLModelVersion.model_to_file_object(model)
        model_response: MLModelResponse = await MLModelsAPI(api_service).create_model(
            model_name=model_name,
            model_version_name=model_version_name,
            framework=FrameworkType(framework.upper()),
            context=image_model_context_specs,
            model_file_content=model_file_object,
        )

        assert model_response.latest_version is not None  # noqa: S101 allow assert
        model_version_response: MLModelVersionResponse = model_response.latest_version

        return (
            cls(api_service=api_service, model_response=model_response),
            MLModelVersion(api_service=api_service, model_version_response=model_version_response),
        )

    @classmethod
    async def get_latest_version(cls, model_id: int) -> MLModelVersion:
        """
        Get the latest model version for a given model.

        :param model_id: ID of the model for which to retrieve the latest version.
        :raises ResourceWithNameNotFoundError: If no model with the given name and version name
            exists.
        :return: Instance of the model with the given version name.
        """
        api_service = get_api_client()
        model_response_list = await MLModelsAPI(api_service).get_all()

        try:
            model_version_response: MLModelVersionResponse = [  # type: ignore
                model_response.latest_version
                for model_response in model_response_list
                if model_response.id == model_id
            ][0]
            return MLModelVersion(
                api_service=api_service, model_version_response=model_version_response
            )
        except ValueError as wrong_entry_number:
            available_names = ", ".join(
                model_response.name for model_response in model_response_list
            )
            raise ResourceWithNameNotFoundError(
                f"Model with ID '{model_id}' not found."
                f"Existing model names: '[{available_names}]'."
            ) from wrong_entry_number


class TextModel(ClassificationOutputMixin, MLModel):
    """
    A machine learning model for text tasks.

    An instance of this class references a model which has been created
    on an aidkit server.
    """

    @staticmethod
    def _build_input_specs(
        batch: Optional[Literal["first", "last"]] = None,
        pre_tokenizer: Union[Split, ByteLevel] = None,
        tokenizer: Union[WordLevel, BytePairEncoding] = None,
    ) -> TextInputSpecs:
        """
        Build input specifications for the create method.

        :param batch: Indicates the position and existence of a batch axis.
        :param pre_tokenizer: Optional Pre-Tokenizer for splitting the input.
        :param tokenizer: Optional Tokenizer to transform tokens into a
            numerical representation.
        :returns: Output specification object for a model context specification.
        """
        batch_dim = BatchDim("No Batch")
        if batch == "first":
            batch_dim = BatchDim("First")
        elif batch == "last":
            batch_dim = BatchDim("Last")

        pre_tokenizer_specs: Union[
            SplitPreTokenizerSpecs, ByteLevelPreTokenizerSpecs, NoPreTokenizerSpecs
        ] = NoPreTokenizerSpecs()
        if isinstance(pre_tokenizer, Split):
            pre_tokenizer_specs = SplitPreTokenizerSpecs(
                convert_to_lower_case=pre_tokenizer.convert_to_lower_case,
                whitespace=pre_tokenizer.split_by_whitespace,
                keep_punctuation=pre_tokenizer.keep_punctuation,
            )
        elif isinstance(pre_tokenizer, ByteLevel):
            pre_tokenizer_specs = ByteLevelPreTokenizerSpecs()

        tokenizer_specs: Union[
            WordLevelTokenizerSpecs, BPETokenizerSpecs, NoTokenizerSpecs
        ] = NoTokenizerSpecs()
        if isinstance(tokenizer, WordLevel):
            if isinstance(tokenizer.vocab, Path):
                with open(tokenizer.vocab, encoding="utf8") as file:
                    tokenizer.vocab = file.read()
                tokenizer_specs = WordLevelTokenizerSpecs(
                    unk_token=tokenizer.unk_token, vocab=tokenizer.vocab
                )
        elif isinstance(tokenizer, BytePairEncoding):
            if isinstance(tokenizer.vocab, Path):
                with open(tokenizer.vocab, encoding="utf8") as file:
                    tokenizer.vocab = file.read()
            if isinstance(tokenizer.merges, Path):
                with open(tokenizer.merges, encoding="utf8") as file:
                    tokenizer.merges = file.read()
            tokenizer_specs = BPETokenizerSpecs(
                unk_token=tokenizer.unk_token, vocab=tokenizer.vocab, merges=tokenizer.merges
            )
        return TextInputSpecs(
            batch_dim=batch_dim,
            pre_tokenizer=pre_tokenizer_specs,
            tokenizer=tokenizer_specs,
            dimensions=TextInputDimensionsModel(),
        )

    @classmethod
    async def create(
        cls,
        model: Any,
        model_name: str,
        model_version_name: str,
        framework: Literal["TensorFlow", "PyTorch"],
        class_names: Union[int, List[str], Path],
        output_format: ClassificationOutputFormat,
        batch_axis: Optional[Literal["first", "last"]] = None,
        pre_tokenizer: Union[Split, ByteLevel] = None,
        tokenizer: Union[WordLevel, BytePairEncoding] = None,
    ) -> Tuple["TextModel", "MLModelVersion"]:
        """
        Create and upload a new text classification model.

        :param model: Model to upload. How this parameter is interpreted
            depends on its type:

            * If the parameter is a string, it is interpreted as the path of the model \
                file to be uploaded.
            * If the parameter is an instance of a keras model, the model is saved and \
                the resulting file is uploaded.
            * If the parameter is a torch ``ScriptModule``, the ``ScriptModule`` is \
                saved and the resulting file is uploaded.

        :param model_name: Name of the model to create.
        :param model_version_name: Name of the first model version.
        :param framework: Framework of the model, either "TensorFlow" or "PyTorch".
        :param class_names: Names of the classes. How this parameter is interpreted
            depends on its type:

            * If the parameter is a string, it is interpreted as the path of a file \
            containing the comma separated list of class names.
            * If the parameter is a list of strings, each string is interpreted as a \
            class name.
            * If the parameter is an integer, it is interpreted as the number of \
            classes and the names will be autogenerated.

        :param output_format: Model output format.
        :param batch_axis: Optional string indicating the batch dimension, either
            "first" or "last".
        :param pre_tokenizer: Optional Pre-Tokenizer for splitting the input
            into tokens.
        :param tokenizer: Optional Tokenizer to transform tokens into a
            numerical representation.
        :return: Tuple containing the newly created model and the corresponding first
            model version.
        """
        api_service = get_api_client()

        model_name = cls._validate_name(model_name)
        model_version_name = cls._validate_name(model_version_name)

        class_names_parsed = cls._get_class_names(class_names=class_names)

        input_specs = cls._build_input_specs(
            batch=batch_axis, pre_tokenizer=pre_tokenizer, tokenizer=tokenizer
        )

        output_specs = cls._build_classification_output_specs(
            class_names_parsed=class_names_parsed, output_format=output_format
        )

        text_model_context_specs = TextModelContextSpecs(
            input_specification=input_specs, output_specification=output_specs
        )

        model_file_object, _ = MLModelVersion.model_to_file_object(model)
        model_response: MLModelResponse = await MLModelsAPI(api_service).create_model(
            model_name=model_name,
            model_version_name=model_version_name,
            framework=FrameworkType(framework.upper()),
            context=text_model_context_specs,
            model_file_content=model_file_object,
        )

        assert model_response.latest_version is not None  # noqa: S101 allow assert
        model_version_response: MLModelVersionResponse = model_response.latest_version

        return (
            cls(api_service=api_service, model_response=model_response),
            MLModelVersion(api_service=api_service, model_version_response=model_version_response),
        )
