"""
A dataset is a collection of observations of the same type.

A subset is a set of observations. Observations of a Dataset can be part
of multiple overlapping subsets.
"""
import io
from hashlib import md5
from pathlib import Path
from typing import Any, BinaryIO, Dict, Iterable, List, Optional, Sequence, Tuple, Union

from PIL import Image

from aidkit_client._endpoints.datasets import DatasetAPI
from aidkit_client._endpoints.models import (
    DatasetResponse,
    ObservationResponse,
    ObservationType,
    SubsetResponse,
)
from aidkit_client._endpoints.observations import ObservationAPI
from aidkit_client._endpoints.subsets import SubsetAPI
from aidkit_client.aidkit_api import HTTPService
from aidkit_client.configuration import get_api_client
from aidkit_client.exceptions import AidkitClientError
from aidkit_client.resources.data_point import DataPointType, RemoteFile


def _bytes_to_hash(bytes_to_hash: bytes) -> str:
    # we are fine with using an unsafe hash function
    md5_hash = md5()  # noqa
    md5_hash.update(bytes_to_hash)
    return md5_hash.hexdigest()


def _observation_to_file_object_and_name(observation: Any) -> Tuple[BinaryIO, str]:
    if isinstance(observation, Path):
        #  using a context manager doesn't work here
        return (open(observation, "rb"), observation.name)  # pylint: disable=consider-using-with
    if isinstance(observation, str):
        str_in_bytes = observation.encode("UTF-8")
        return (io.BytesIO(str_in_bytes), _bytes_to_hash(str_in_bytes))
    if isinstance(observation, bytes):
        return (io.BytesIO(observation), _bytes_to_hash(observation))
    if isinstance(observation, Image.Image):
        bio = io.BytesIO()
        observation.save(bio, format="PNG")
        bio.seek(0)
        if hasattr(observation, "filename"):
            name = Path(observation.filename).name
        else:
            name = _bytes_to_hash(bio.getvalue())
        return (bio, name)

    raise ValueError(f"Loading observation of type {type(observation)} currently not supported.")


class Observation:
    """
    An observation.

    An instance of this class references an observation.
    """

    def __init__(self, api_service: HTTPService, observation_response: ObservationResponse) -> None:
        """
        Create a new instance from the server response.

        :param api_service: Service instance to use for communicating with the
            server.
        :param observation_response: Server response describing the observation
            to be created.
        """
        self._data = observation_response
        self._api_service = api_service

    @classmethod
    async def create(
        cls,
        dataset: Union["Dataset", int],
        file: Any,
        observation_type: ObservationType,
        subsets: Sequence[Union["Subset", int]],
        name: Optional[str] = None,
        odd_tags: Optional[List[str]] = None,
    ) -> "Observation":
        """
        Create and upload a single observation.

        :param dataset: Dataset to add the observation to.
        :param file: Observation to upload. How this parameter is interpreted
            depends on its type:

            * If the parameter is a string, it is interpreted as a text-observation to be uploaded.
            * If the parameter is a ``PIL.Image``, it is interpreted as an PNG image to be uploaded.
            * If the parameter is a ``pathlib.Path``, it is interpreted as the path of the \
                observation file to be uploaded.
            * If the parameter is a ``bytes``-object, it is uploaded unchanged.

        :param observation_type: Type of the observation to add.
        :param subsets: List of subsets this observation belongs to. Strings are
            interpreted as subset names.
        :param name: Name of the observation to create. If set to None, a name
            is autogenerated by the following rule:

            * If the ``file`` parameter is an instance of ``pathlib.Path``, the file name is used.
            * If the ``file`` parameter is an instance of ``PIL.Image``, and has an attribute
                ``filename``, ``file.filename`` is used. This is the case if the ``file`` argument
                has been created using ``PIL.Image.open``.
            * Otherwise, the MD5 hash of the ``file`` attribute converted to bytes, is computed.
                The hex representation of this hash is used as observation name.
        :param odd_tags: List of parameters from the Operational Design Domain (ODD) that
            apply to this observation.
        :return: Created observation.
        """
        if isinstance(dataset, Dataset):
            dataset = dataset.id

        subset_ids = [subset if isinstance(subset, int) else subset.id for subset in subsets]

        api_service = get_api_client()
        observation_file_object, name_from_file = _observation_to_file_object_and_name(file)
        if name is None:
            name = name_from_file

        observation_response = await ObservationAPI(api_service).create(
            dataset_id=dataset,
            observation_type=observation_type,
            subset_ids=subset_ids,
            odd_tags=odd_tags,
            obs_name=name,
            obs_data=observation_file_object,
        )
        return Observation(api_service=api_service, observation_response=observation_response)

    @classmethod
    async def get_by_id(cls, observation_id: int) -> "Observation":
        """
        Get an observation by its ID.

        :param observation_id: ID of the observation to fetch.
        :return: Instance of the observation with the given ID.
        """
        api_service = get_api_client()
        pipeline_response = await ObservationAPI(api_service).get_by_id(observation_id)
        return Observation(api_service, pipeline_response)

    @classmethod
    async def delete(cls, observation_id: int) -> None:
        """
        Delete an observation by its ID.

        :param observation_id: int
        :return: None.
        """
        await ObservationAPI(get_api_client()).delete(observation_id)

    def as_remote_file(self) -> RemoteFile:
        """
        Get the corresponding remote file, which can be used to download the
        observation.

        :return: RemoteFile corresponding to the observation.
        :raises AidkitClientError: If the observation type is unknown or the
            observation url is not set.
        """
        if self._data.type in ["COLOR_IMAGE", "GREY_SCALE_IMAGE"]:
            data_point_type = DataPointType.IMAGE
        elif self._data.type == "TEXT":
            data_point_type = DataPointType.TEXT
        else:
            raise AidkitClientError(f"Unknown observation type {self._data.type}.")
        if self._data.file_url is None:
            raise AidkitClientError(f"Observation with id {self.id} has no url.")
        return RemoteFile(url=self._data.file_url, type=data_point_type)

    @property
    def id(self) -> int:
        """
        Get the ID of the instance.

        :return: ID of the instance.
        """
        return self._data.id

    @property
    def name(self) -> str:
        """
        Get the name the instance.

        :return: Name of the instance.
        """
        return self._data.file_name


class Subset:
    """
    A dataset subset.

    An instance of this class references a subset.
    """

    def __init__(self, api_service: HTTPService, subset_response: SubsetResponse) -> None:
        """
        Create a new instance from the server response.

        :param api_service: Service instance to use for communicating with the
            server.
        :param subset_response: response describing the subset
            to be created.
        """
        self._data = subset_response
        self._api_service = api_service

    @classmethod
    async def create(
        cls,
        name: str,
        dataset: Union["Dataset", int],
        observations: Sequence[Union[Observation, int]],
    ) -> "Subset":
        """
        Create a subset of a dataset.

        :param name: Name of the subset.
        :param dataset: Dataset to upload observations to. Strings are
            interpreted as dataset names.
        :param observations: Observations to be in the subset. Integers are
            interpreted as observation IDs.
        :return: Created subset.
        """
        if isinstance(dataset, Dataset):
            dataset = dataset.id

        api_service = get_api_client()
        observation_ids = [obs if isinstance(obs, int) else obs.id for obs in observations]
        subset_response = await SubsetAPI(api_service).create(
            subset_name=name, dataset_id=dataset, observation_ids=observation_ids
        )
        return Subset(api_service=api_service, subset_response=subset_response)

    async def update(self, observations: Sequence[Union[Observation, int]]) -> "Subset":
        """
        Update the observations within a subset.

        :param observations: Observations to add to the dataset. Integers are
            interpreted as observation IDs.
        :return: Instance of the updated subset.
        """
        api_service = get_api_client()
        observation_ids = [obs if isinstance(obs, int) else obs.id for obs in observations]
        subset_response = await SubsetAPI(api_service).update(
            subset_id=self.id, observation_ids=observation_ids
        )
        return Subset(api_service=api_service, subset_response=subset_response)

    @classmethod
    async def get_by_id(cls, subset_id: int) -> "Subset":
        """
        Get a subset by its ID.

        :param subset_id: ID of the subset to create an instance of.
        :return: Instance of the subset with the given ID.
        """
        api_service = get_api_client()
        response = await SubsetAPI(api_service).get(subset_id)
        return Subset(api_service, response)

    @property
    def id(self) -> int:
        """
        Get the ID of the instance.

        :return: ID of the instance.
        """
        return self._data.id

    @property
    def name(self) -> str:
        """
        Get the name the instance.

        :return: Name of the instance.
        """
        return self._data.name

    @property
    def observation_ids(self) -> List[int]:
        """
        Get the observation ids contained in the subset.

        :return: List of observation ids in the subset.
        """
        return self._data.observation_ids

    @classmethod
    async def delete(cls, subset_id: int) -> None:
        """
        Delete a subset.

        :param subset_id: int
        :return: None.
        """
        await SubsetAPI(get_api_client()).delete(subset_id)


class Dataset:
    """
    A dataset.

    An instance of this class references a dataset.
    """

    def __init__(self, api_service: HTTPService, dataset_response: DatasetResponse) -> None:
        """
        Create a new instance from the server response.

        :param api_service: Service instance to use for communicating with the
            server.
        :param dataset_response: Server response describing the dataset
            to be created.
        """
        self._data = dataset_response
        self._api_service = api_service

    @classmethod
    async def create(
        cls,
        dataset_name: str,
        dataset_type: ObservationType,
        files: Optional[Union[Dict[str, Any], List[Any]]] = None,
        subsets: Optional[Sequence[Union[Subset, str]]] = None,
    ) -> Tuple["Dataset", List[Subset]]:
        """
        Create a dataset.

        :param dataset_name: Name of the dataset.
        :param dataset_type: Type of the dataset.
        :param files: Files to upload.
            Can be either a dictionary of the form ``{observation_name: file}`` or a list of files
            to upload.

            How a file is interpreted depends on its type:

            * If a file is a string, it is interpreted as a text-observation to be uploaded.
            * If a file is a ``PIL.Image``, it is interpreted as an PNG image to be uploaded.
            * If a file is a ``pathlib.Path``, it is interpreted as the path of the \
                observation file to be uploaded.
            * If a file is a ``bytes``-object, it is uploaded unchanged.

            If a list of files is passed, the names of the created observations are autogenerated
            by the following rule:

            * If a file is an instance of ``pathlib.Path``, the file name is used.
            * If a file is an instance of ``PIL.Image``, and has an attribute
                ``filename``, ``file.filename`` is used. This is the case if the file
                has been created using ``PIL.Image.open``.
            * Otherwise, the MD5 hash of the file converted to bytes is computed.
                The hex representation of this hash is used as observation name.

        :param subsets: List of subsets to add in the dataset. If the ``files`` argument is not
            ``None``, all the created observations are added to all created subsets.
        :return: Tuple containing the created dataset and List of created subsets (may be empty)
        """
        if subsets is None:
            subsets = []
        if files is None:
            files = []
        api_service = get_api_client()
        dataset_response = await DatasetAPI(api_service).create(
            dataset_name=dataset_name,
            dataset_type=dataset_type,
        )
        dataset = Dataset(api_service=api_service, dataset_response=dataset_response)
        created_subsets = []

        for subset in subsets:
            subset_name = subset if isinstance(subset, str) else subset.name
            created_subset = await dataset.create_subset(subset_name, [])
            created_subsets.append(created_subset)

        await dataset.upload_data(files, [subset.id for subset in created_subsets])
        if subsets or files:
            # reload the dataset to reflect the changes
            dataset = await Dataset.get_by_id(dataset.id)

        return dataset, created_subsets

    @classmethod
    async def delete(cls, dataset_id: int) -> None:
        """
        Delete a dataset.

        :param dataset_id: int
        :return: None.
        """
        await DatasetAPI(get_api_client()).delete(dataset_id)

    async def create_subset(
        self, name: str, observations: Sequence[Union[Observation, int]]
    ) -> Subset:
        """
        Create a subset of the dataset.

        :param name: Name of the subset to create.
        :param observations: Observations to be part of the subset. Integers are
            interpreted as observation IDs.
        :return: Created subsets.
        """
        return await Subset.create(name=name, dataset=self.id, observations=observations)

    async def upload_data(
        self,
        files: Union[Dict[str, Any], List[Any]],
        subsets: Sequence[Union[Subset, int]],
        odd_tags: Optional[Sequence[Optional[List[str]]]] = None,
    ) -> List[Observation]:
        """
        Upload data to the dataset.

        :param files: Files to upload.
            Can be either a dictionary of the form ``{observation_name: file}`` or a list of files
            to upload.

            How a file is interpreted depends on its type:

            * If a file is a string, it is interpreted as a text-observation to be uploaded.
            * If a file is a ``PIL.Image``, it is interpreted as an PNG image to be uploaded.
            * If a file is a ``pathlib.Path``, it is interpreted as the path of the \
                observation file to be uploaded.
            * If a file is a ``bytes``-object, it is uploaded unchanged.

            If a list of files is passed, the names of the created observations are autogenerated
            by the following rule:

            * If a file is an instance of ``pathlib.Path``, the file name is used.
            * If a file is an instance of ``PIL.Image``, and has an attribute
                ``filename``, ``file.filename`` is used. This is the case if the file
                has been created using ``PIL.Image.open``.
            * Otherwise, the MD5 hash of the file converted to bytes is computed.
                The hex representation of this hash is used as observation name.

        :param subsets: List of subsets the observations are in. Strings are
            interpreted as subset names.
        :param odd_tags: List of parameters from the Operational Design Domain (ODD) for every
                observation that will be uploaded.
        :raises ValueError: If the number of ODD tags specified does not match the number
            of observations.
        :return: Created observations.
        """
        observations = []
        if isinstance(files, dict):
            name_and_files_list: Iterable[Tuple[Optional[str], Any]] = files.items()
        else:
            name_and_files_list = [(None, file) for file in files]

        if not odd_tags:
            converted_odd_tags: Iterable[Optional[List[str]]] = [None for _ in name_and_files_list]
        else:
            if len(odd_tags) != len(files):
                raise ValueError(
                    (
                        "The number of provided odd_tags does not match the number of "
                        f"observations ({len(odd_tags)} != {len(files)})"
                    )
                )
            converted_odd_tags = odd_tags

        for (name, file), odd_tag in zip(name_and_files_list, converted_odd_tags):
            uploaded_observation = await Observation.create(
                dataset=self.id,
                file=file,
                observation_type=ObservationType(self._data.type),
                subsets=subsets,
                name=name,
                odd_tags=odd_tag,
            )
            observations.append(uploaded_observation)

        return observations

    @classmethod
    async def get_all(cls) -> List["Dataset"]:
        """
        Get all Datasets.

        :return: List of Datasets
        """
        api_service = get_api_client()
        return [
            Dataset(api_service, response) for response in await DatasetAPI(api_service).get_all()
        ]

    async def get_all_subsets(self) -> List["Subset"]:
        """
        Get all Subsets for the Dataset.

        :return: List of Subsets
        """
        api_service = get_api_client()
        return [
            Subset(api_service, response)
            for response in await SubsetAPI(api_service).get_all(self.id)
        ]

    @classmethod
    async def get_by_id(cls, dataset_id: int) -> "Dataset":
        """
        Get a Dataset by its ID.

        :param dataset_id: ID of the dataset to create an instance of.
        :return: Instance of the Dataset with the given ID.
        """
        api_service = get_api_client()
        pipeline_response = await DatasetAPI(api_service).get(dataset_id)
        return Dataset(api_service, pipeline_response)

    @property
    def id(self) -> int:
        """
        Get the ID of the instance.

        :return: ID of the instance.
        """
        return self._data.id

    @property
    def name(self) -> str:
        """
        Get the name the instance.

        :return: Name of the instance.
        """
        return self._data.name
