"""
Dataclasses describing single data points.
"""

from dataclasses import dataclass
from enum import Enum
from io import BytesIO
from typing import Union

from PIL import Image

from aidkit_client.configuration import get_api_client
from aidkit_client.exceptions import AidkitClientError


class DataPointType(Enum):
    """
    Possible types of data points.
    """

    TEXT = 1
    IMAGE = 2


@dataclass
class RemoteFile:
    """
    Data point stored on an aidkit server, either uploaded by a user or
    generated by aidkit.
    """

    url: str
    type: DataPointType

    @classmethod
    def from_url(cls, url: str) -> "RemoteFile":
        """
        Create a data point from a URL, by guessing the type based on the URL
        suffix.

        :param url: URL where the data point resides.
        :return: Instance of a DataPoint with the given URL.
        :raises AidkitClientError: If the data point type cannot be inferred from
            the URL suffix.
        """
        url_suffix = url.rsplit(".", maxsplit=1)[-1]
        if url_suffix == "txt":
            data_point_type = DataPointType.TEXT
        elif url_suffix == "png":
            data_point_type = DataPointType.IMAGE
        else:
            raise AidkitClientError(f"Cannot infer data point type from URL suffix {url_suffix}.")
        return cls(url=url, type=data_point_type)

    async def fetch_remote_file(self) -> Union[str, Image.Image]:
        """
        Fetch the remote file from the URL.

        :raises AidkitClientError: If the response of the aidkit server is invalid.
        :return: Fetched remote file from the given URL. Depending on whether the
            referenced remote file is a text or an image data point, the returned object is
            either a string or a pillow image.
        """
        api_service = get_api_client()
        response = await api_service.get_from_cdn(self.url)
        remote_file_type = self.url.rsplit(".", maxsplit=1)[-1]
        content = response.body_dict_or_error(f"Could not fetch data point at URL {self.url}")[
            "content"
        ]
        if not isinstance(content, bytes):
            raise AidkitClientError(
                f"Response body-content must be bytes but was type {type(content)}"
            )
        if self.type == DataPointType.TEXT:
            return content.decode("utf-8")

        if self.type == DataPointType.IMAGE:
            return Image.open(BytesIO(content))

        raise AidkitClientError(
            f"Unknown data point type received. "
            f"Valid are 'png' and 'txt' but received '{remote_file_type}'."
        )
