#!/usr/bin/env python3

import unittest

import torch
from linear_operator.utils.interpolation import left_interp

from gpytorch.test.utils import approx_equal
from gpytorch.utils.interpolation import Interpolation


class TestCubicInterpolation(unittest.TestCase):
    def test_interpolation(self):
        x = torch.linspace(0.01, 1, 100).unsqueeze(1)
        grid = torch.linspace(-0.05, 1.05, 50).unsqueeze(1)
        indices, values = Interpolation().interpolate(grid, x)
        indices = indices.squeeze_(0)
        values = values.squeeze_(0)
        test_func_grid = grid.squeeze(1).pow(2)
        test_func_x = x.pow(2).squeeze(-1)

        interp_func_x = left_interp(indices, values, test_func_grid.unsqueeze(1)).squeeze()

        self.assertTrue(approx_equal(interp_func_x, test_func_x))

    def test_multidim_interpolation(self):
        x = torch.tensor([[0.25, 0.45, 0.65, 0.85], [0.35, 0.375, 0.4, 0.425], [0.45, 0.5, 0.55, 0.6]]).t().contiguous()
        grid = torch.linspace(0.0, 1.0, 11).unsqueeze(1).repeat(1, 3)

        indices, values = Interpolation().interpolate(grid, x)

        actual_indices = torch.cat(
            [
                torch.tensor(
                    [
                        [146, 147, 148, 149, 157, 158, 159, 160, 168, 169, 170, 171, 179],
                        [389, 390, 391, 392, 400, 401, 402, 403, 411, 412, 413, 414, 422],
                        [642, 643, 644, 645, 653, 654, 655, 656, 664, 665, 666, 667, 675],
                        [885, 886, 887, 888, 896, 897, 898, 899, 907, 908, 909, 910, 918],
                    ],
                    dtype=torch.long,
                ),
                torch.tensor(
                    [
                        [180, 181, 182, 267, 268, 269, 270, 278, 279, 280, 281, 289, 290],
                        [423, 424, 425, 510, 511, 512, 513, 521, 522, 523, 524, 532, 533],
                        [676, 677, 678, 763, 764, 765, 766, 774, 775, 776, 777, 785, 786],
                        [919, 920, 921, 1006, 1007, 1008, 1009, 1017, 1018, 1019, 1020, 1028, 1029],
                    ],
                    dtype=torch.long,
                ),
                torch.tensor(
                    [
                        [291, 292, 300, 301, 302, 303, 388, 389, 390, 391, 399, 400, 401],
                        [534, 535, 543, 544, 545, 546, 631, 632, 633, 634, 642, 643, 644],
                        [787, 788, 796, 797, 798, 799, 884, 885, 886, 887, 895, 896, 897],
                        [1030, 1031, 1039, 1040, 1041, 1042, 1127, 1128, 1129, 1130, 1138, 1139, 1140],
                    ],
                    dtype=torch.long,
                ),
                torch.tensor(
                    [
                        [402, 410, 411, 412, 413, 421, 422, 423, 424, 509, 510, 511, 512],
                        [645, 653, 654, 655, 656, 664, 665, 666, 667, 752, 753, 754, 755],
                        [898, 906, 907, 908, 909, 917, 918, 919, 920, 1005, 1006, 1007, 1008],
                        [1141, 1149, 1150, 1151, 1152, 1160, 1161, 1162, 1163, 1248, 1249, 1250, 1251],
                    ],
                    dtype=torch.long,
                ),
                torch.tensor(
                    [
                        [520, 521, 522, 523, 531, 532, 533, 534, 542, 543, 544, 545],
                        [763, 764, 765, 766, 774, 775, 776, 777, 785, 786, 787, 788],
                        [1016, 1017, 1018, 1019, 1027, 1028, 1029, 1030, 1038, 1039, 1040, 1041],
                        [1259, 1260, 1261, 1262, 1270, 1271, 1272, 1273, 1281, 1282, 1283, 1284],
                    ],
                    dtype=torch.long,
                ),
            ],
            1,
        )
        self.assertTrue(approx_equal(indices, actual_indices))

        actual_values = torch.cat(
            [
                torch.tensor(
                    [
                        [-0.0002, 0.0022, 0.0022, -0.0002, 0.0022, -0.0198, -0.0198, 0.0022, 0.0022, -0.0198],
                        [0.0000, 0.0015, 0.0000, 0.0000, -0.0000, -0.0142, -0.0000, -0.0000, -0.0000, -0.0542],
                        [0.0000, -0.0000, -0.0000, 0.0000, 0.0039, -0.0352, -0.0352, 0.0039, 0.0000, -0.0000],
                        [0.0000, 0.0044, 0.0000, 0.0000, -0.0000, -0.0542, -0.0000, -0.0000, -0.0000, -0.0142],
                    ]
                ),
                torch.tensor(
                    [
                        [-0.0198, 0.0022, -0.0002, 0.0022, 0.0022, -0.0002, 0.0022, -0.0198, -0.0198, 0.0022],
                        [-0.0000, -0.0000, 0.0000, 0.0044, 0.0000, 0.0000, -0.0000, -0.0132, -0.0000, -0.0000],
                        [-0.0000, 0.0000, 0.0000, -0.0000, -0.0000, 0.0000, -0.0000, 0.0000, 0.0000, -0.0000],
                        [-0.0000, -0.0000, 0.0000, 0.0015, 0.0000, 0.0000, -0.0000, -0.0396, -0.0000, -0.0000],
                    ]
                ),
                torch.tensor(
                    [
                        [-0.0198, 0.1780, 0.1780, -0.0198, -0.0198, 0.1780, 0.1780, -0.0198, 0.0022, -0.0198],
                        [0.0000, 0.1274, 0.0000, 0.0000, 0.0000, 0.4878, 0.0000, 0.0000, -0.0000, -0.0396],
                        [-0.0352, 0.3164, 0.3164, -0.0352, -0.0000, 0.0000, 0.0000, -0.0000, -0.0000, 0.0000],
                        [0.0000, 0.4878, 0.0000, 0.0000, 0.0000, 0.1274, 0.0000, 0.0000, -0.0000, -0.0132],
                    ]
                ),
                torch.tensor(
                    [
                        [-0.0198, 0.0022, 0.0022, -0.0198, -0.0198, 0.0022, -0.0198, 0.1780, 0.1780, -0.0198],
                        [-0.0000, -0.0000, -0.0000, -0.0132, -0.0000, -0.0000, 0.0000, 0.1274, 0.0000, 0.0000],
                        [0.0000, -0.0000, -0.0000, 0.0000, 0.0000, -0.0000, -0.0352, 0.3164, 0.3164, -0.0352],
                        [-0.0000, -0.0000, -0.0000, -0.0396, -0.0000, -0.0000, 0.0000, 0.4878, 0.0000, 0.0000],
                    ]
                ),
                torch.tensor(
                    [
                        [-0.0198, 0.1780, 0.1780, -0.0198, 0.0022, -0.0198, -0.0198, 0.0022, -0.0002, 0.0022],
                        [0.0000, 0.4878, 0.0000, 0.0000, -0.0000, -0.0396, -0.0000, -0.0000, 0.0000, 0.0015],
                        [-0.0000, 0.0000, 0.0000, -0.0000, -0.0000, 0.0000, 0.0000, -0.0000, 0.0000, -0.0000],
                        [0.0000, 0.1274, 0.0000, 0.0000, -0.0000, -0.0132, -0.0000, -0.0000, 0.0000, 0.0044],
                    ]
                ),
                torch.tensor(
                    [
                        [0.0022, -0.0002, 0.0022, -0.0198, -0.0198, 0.0022, 0.0022, -0.0198, -0.0198, 0.0022],
                        [0.0000, 0.0000, -0.0000, -0.0142, -0.0000, -0.0000, -0.0000, -0.0542, -0.0000, -0.0000],
                        [-0.0000, 0.0000, 0.0039, -0.0352, -0.0352, 0.0039, 0.0000, -0.0000, -0.0000, 0.0000],
                        [0.0000, 0.0000, -0.0000, -0.0542, -0.0000, -0.0000, -0.0000, -0.0142, -0.0000, -0.0000],
                    ]
                ),
                torch.tensor(
                    [
                        [-0.0002, 0.0022, 0.0022, -0.0002],
                        [0.0000, 0.0044, 0.0000, 0.0000],
                        [0.0000, -0.0000, -0.0000, 0.0000],
                        [0.0000, 0.0015, 0.0000, 0.0000],
                    ]
                ),
            ],
            1,
        )
        self.assertTrue(approx_equal(values, actual_values))


if __name__ == "__main__":
    unittest.main()
