"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const synthetics_canned_metrics_generated_1 = require("./synthetics-canned-metrics.generated");
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * (experimental) Specify a test that the canary should run.
 *
 * @experimental
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * (experimental) Specify a custom test with your own code.
     *
     * @param options The configuration options.
     * @returns `Test` associated with the specified Code object
     * @experimental
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
_a = JSII_RTTI_SYMBOL_1;
Test[_a] = { fqn: "@aws-cdk/aws-synthetics.Test", version: "1.106.1" };
/**
 * (experimental) Runtime options for a canary.
 *
 * @experimental
 */
class Runtime {
    /**
     * @param name The name of the runtime version.
     * @experimental
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
_b = JSII_RTTI_SYMBOL_1;
Runtime[_b] = { fqn: "@aws-cdk/aws-synthetics.Runtime", version: "1.106.1" };
/**
 * (experimental) `syn-1.0` includes the following:.
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-1.0
 * @experimental
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * (experimental) `syn-nodejs-2.0` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 83.0.4103.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-2.0
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_0 = new Runtime('syn-nodejs-2.0');
/**
 * (experimental) `syn-nodejs-2.1` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 83.0.4103.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-2.1
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_1 = new Runtime('syn-nodejs-2.1');
/**
 * (experimental) `syn-nodejs-2.2` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 83.0.4103.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-2.2
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_2 = new Runtime('syn-nodejs-2.2');
/**
 * (experimental) `syn-nodejs-puppeteer-3.0` includes the following: - Lambda runtime Node.js 12.x - Puppeteer-core version 5.5.0 - Chromium version 88.0.4298.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-nodejs-puppeteer-3.0
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_0 = new Runtime('syn-nodejs-puppeteer-3.0');
/**
 * (experimental) `syn-nodejs-puppeteer-3.1` includes the following: - Lambda runtime Node.js 12.x - Puppeteer-core version 5.5.0 - Chromium version 88.0.4298.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-nodejs-puppeteer-3.1
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_1 = new Runtime('syn-nodejs-puppeteer-3.1');
/**
 * (experimental) Define a new Canary.
 *
 * @experimental
 */
class Canary extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k, _l;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.string({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_e = (_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.bucket) !== null && _e !== void 0 ? _e : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_f = props.role) !== null && _f !== void 0 ? _f : this.createDefaultRole((_g = props.artifactsBucketLocation) === null || _g === void 0 ? void 0 : _g.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_h = props.artifactsBucketLocation) === null || _h === void 0 ? void 0 : _h.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_j = props.startAfterCreation) !== null && _j !== void 0 ? _j : true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_k = props.failureRetentionPeriod) === null || _k === void 0 ? void 0 : _k.toDays(),
            successRetentionPeriod: (_l = props.successRetentionPeriod) === null || _l === void 0 ? void 0 : _l.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * (experimental) Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricDuration(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.durationAverage, options);
    }
    /**
     * (experimental) Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricSuccessPercent(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.successPercentAverage, options);
    }
    /**
     * (experimental) Measure the number of failed canary runs over a given time period.
     *
     * Default: sum over 5 minutes
     *
     * @param options - configuration options for the metric.
     * @experimental
     */
    metricFailed(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.failedSum, options);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        const { partition } = cdk.Stack.of(this);
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: [`arn:${partition}:logs:::*`],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _d, _e, _f;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_d = codeConfig.s3Location) === null || _d === void 0 ? void 0 : _d.bucketName,
            s3Key: (_e = codeConfig.s3Location) === null || _e === void 0 ? void 0 : _e.objectKey,
            s3ObjectVersion: (_f = codeConfig.s3Location) === null || _f === void 0 ? void 0 : _f.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _d, _e, _f, _g;
        return {
            durationInSeconds: String(`${(_e = (_d = props.timeToLive) === null || _d === void 0 ? void 0 : _d.toSeconds()) !== null && _e !== void 0 ? _e : 0}`),
            expression: (_g = (_f = props.schedule) === null || _f === void 0 ? void 0 : _f.expressionString) !== null && _g !== void 0 ? _g : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
    cannedMetric(fn, props) {
        return new aws_cloudwatch_1.Metric({
            ...fn({ CanaryName: this.canaryName }),
            ...props,
        }).attachTo(this);
    }
}
exports.Canary = Canary;
_c = JSII_RTTI_SYMBOL_1;
Canary[_c] = { fqn: "@aws-cdk/aws-synthetics.Canary", version: "1.106.1" };
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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