"""
This submodule is used to parse the input configuration into an object.
The configuration object is a python dataclass created by ConfigSchema
which is a marshmallow schema. The data class contains fields defined
by ConfigSchema. Subclass ConfigSchema to add fields.
"""

import logging
from dataclasses import make_dataclass
from marshmallow import Schema, fields, post_load, ValidationError

_LOGGER = logging.getLogger('BuildUp')

class _CategorySchema(Schema):
    """
    Marshmallow schema for validating custom BuildUp categories. On load it converts
    the input dictionary to a python dataclass.
    """
    reuse = fields.Bool(required=True, data_key='Reuse')
    display_name = fields.Str(required=True, data_key='DisplayName')

    @post_load
    def make_object(self, data, **_): #pylint: disable=no-self-use
        """
        Auto generates a dataclass for the category
        """
        cat = make_dataclass('Category', data.keys())
        return cat(**data)

def _create_base_categories():
    cat_schema = _CategorySchema()
    return {"part": cat_schema.load({"Reuse": False, "DisplayName": "Parts"}),
            "tool": cat_schema.load({"Reuse": True, "DisplayName": "Tools"})}

class _CategoriesField(fields.Dict):
    """CategoriesField is used by marshmallow to serialise and de-serialise
    custom_categories. All category names are turned lower case."""

    def _deserialize(self, value, attr, data, **kwargs):
        value = super(_CategoriesField, self)._deserialize(value, attr, data, **kwargs)
        lowered = _create_base_categories()
        for key in value:
            lowered[key.lower()] = value[key]
        return lowered

class _DefaultCategory(fields.Str):
    """DefaultCategory is used by marshmallow to serialise and de-serialise
    the DefaultCategories. This object will hopefully be removed once we
    can find a better way to compare the field to the de-serialised output
    of custom_categories"""

    def _deserialize(self, value, attr, data, **kwargs):
        value = super(_DefaultCategory, self)._deserialize(value, attr, data, **kwargs)
        value = value.lower()

        if value in ['part', 'tool']:
            return value
        if 'CustomCategories' in data and isinstance(data['CustomCategories'], dict):
            for key in data['CustomCategories']:
                if str(key).lower() == value:
                    return value
        raise ValidationError(f'{value} is not a valid category.')

class _NavSchema(Schema):
    """
    Marshmallow schema for parsing the navigation
    """
    title = fields.Str(required=True, data_key='Title')
    link = fields.Str(required=True, data_key='Link')
    subnavigation = fields.List(cls_or_instance=fields.Nested("_NavSchema"),
                                data_key='SubNavigation')

class ConfigSchema(Schema):
    """
    This is the schema for the main configuration object that is used in BuildUp.
    The configuration object is generated by passing a dictionary into this marshmallow
    schema. A validation error is returned for extra fields. If you want to use the
    configuration dictionary to hold to add information you can either subclass this schema,
    or you can use  ConfigSchema.validate() to find extra fields and remove them before the
    configuration object is created.
    Extra fields are not allowed to help catch typos the configuration.
    """
    title = fields.Str(missing=None, allow_none=True, data_key='Title')
    page_bom_title = fields.Str(missing="", data_key='PageBOMTitle')
    custom_categories = _CategoriesField(missing=_create_base_categories,
                                         keys=fields.Str(),
                                         values=fields.Nested(_CategorySchema),
                                         data_key='CustomCategories')
    default_category = _DefaultCategory(missing="part", data_key='DefaultCategory')
    navigation = fields.List(cls_or_instance=fields.Nested(_NavSchema),
                             missing=list,
                             data_key='Navigation')
    landing_page = fields.Str(missing=None, allow_none=True, data_key='LandingPage')
    remove_landing_title = fields.Bool(missing=False, data_key='RemoveLandingTitle')
    force_output = fields.List(cls_or_instance=fields.Str,
                               missing=list,
                               data_key='ForceOutput')

    @post_load
    def make_object(self, data, **_): #pylint: disable=no-self-use
        """
        Auto generates a dataclass for the configuration.
        """
        data['categories'] = data['custom_categories']
        del data['custom_categories']

        buildup_config = make_dataclass('BuildUpConfig',
                                        data.keys())
        return buildup_config(**data)
