"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SelfManagedKafkaEventSource = exports.ManagedKafkaEventSource = exports.AuthenticationMethod = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const stream_1 = require("./stream");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * The authentication method to use with SelfManagedKafkaEventSource
 */
var AuthenticationMethod;
(function (AuthenticationMethod) {
    /**
     * SASL_SCRAM_512_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_512_AUTH"] = "SASL_SCRAM_512_AUTH";
    /**
     * SASL_SCRAM_256_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_256_AUTH"] = "SASL_SCRAM_256_AUTH";
    /**
     * BASIC_AUTH (SASL/PLAIN) authentication method for your Kafka cluster
     */
    AuthenticationMethod["BASIC_AUTH"] = "BASIC_AUTH";
    /**
     * CLIENT_CERTIFICATE_TLS_AUTH (mTLS) authentication method for your Kafka cluster
     */
    AuthenticationMethod["CLIENT_CERTIFICATE_TLS_AUTH"] = "CLIENT_CERTIFICATE_TLS_AUTH";
})(AuthenticationMethod = exports.AuthenticationMethod || (exports.AuthenticationMethod = {}));
/**
 * Use a MSK cluster as a streaming source for AWS Lambda
 */
class ManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        this._eventSourceMappingId = undefined;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_event_sources_ManagedKafkaEventSourceProps(props);
        this.innerProps = props;
    }
    bind(target) {
        var _c;
        const eventSourceMapping = target.addEventSourceMapping(`KafkaEventSource:${core_1.Names.nodeUniqueId(target.node)}${this.innerProps.topic}`, this.enrichMappingOptions({
            eventSourceArn: this.innerProps.clusterArn,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
            kafkaTopic: this.innerProps.topic,
        }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
        target.addToRolePolicy(new iam.PolicyStatement({
            actions: ['kafka:DescribeCluster', 'kafka:GetBootstrapBrokers', 'kafka:ListScramSecrets'],
            resources: [this.innerProps.clusterArn],
        }));
        (_c = target.role) === null || _c === void 0 ? void 0 : _c.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaMSKExecutionRole'));
    }
    sourceAccessConfigurations() {
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            // "Amazon MSK only supports SCRAM-SHA-512 authentication." from https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html#msk-password-limitations
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH,
                uri: this.innerProps.secret.secretArn,
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
    /**
    * The identifier for this EventSourceMapping
    */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KafkaEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.ManagedKafkaEventSource = ManagedKafkaEventSource;
_a = JSII_RTTI_SYMBOL_1;
ManagedKafkaEventSource[_a] = { fqn: "@aws-cdk/aws-lambda-event-sources.ManagedKafkaEventSource", version: "1.152.0" };
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 */
class SelfManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_event_sources_SelfManagedKafkaEventSourceProps(props);
        if (props.vpc) {
            if (!props.securityGroup) {
                throw new Error('securityGroup must be set when providing vpc');
            }
            if (!props.vpcSubnets) {
                throw new Error('vpcSubnets must be set when providing vpc');
            }
        }
        else if (!props.secret) {
            throw new Error('secret must be set if Kafka brokers accessed over Internet');
        }
        this.innerProps = props;
    }
    bind(target) {
        if (!core_2.Construct.isConstruct(target)) {
            throw new Error('Function is not a construct. Unexpected error.');
        }
        target.addEventSourceMapping(this.mappingId(target), this.enrichMappingOptions({
            kafkaBootstrapServers: this.innerProps.bootstrapServers,
            kafkaTopic: this.innerProps.topic,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
    }
    mappingId(target) {
        let hash = crypto.createHash('md5');
        hash.update(JSON.stringify(core_1.Stack.of(target).resolve(this.innerProps.bootstrapServers)));
        const idHash = hash.digest('hex');
        return `KafkaEventSource:${idHash}:${this.innerProps.topic}`;
    }
    sourceAccessConfigurations() {
        var _c;
        let authType;
        switch (this.innerProps.authenticationMethod) {
            case AuthenticationMethod.BASIC_AUTH:
                authType = lambda.SourceAccessConfigurationType.BASIC_AUTH;
                break;
            case AuthenticationMethod.CLIENT_CERTIFICATE_TLS_AUTH:
                authType = lambda.SourceAccessConfigurationType.CLIENT_CERTIFICATE_TLS_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_256_AUTH:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_256_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_512_AUTH:
            default:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH;
                break;
        }
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            sourceAccessConfigurations.push({ type: authType, uri: this.innerProps.secret.secretArn });
        }
        if (this.innerProps.vpcSubnets !== undefined && this.innerProps.securityGroup !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.VPC_SECURITY_GROUP,
                uri: this.innerProps.securityGroup.securityGroupId,
            });
            (_c = this.innerProps.vpc) === null || _c === void 0 ? void 0 : _c.selectSubnets(this.innerProps.vpcSubnets).subnetIds.forEach((id) => {
                sourceAccessConfigurations.push({ type: lambda.SourceAccessConfigurationType.VPC_SUBNET, uri: id });
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.SelfManagedKafkaEventSource = SelfManagedKafkaEventSource;
_b = JSII_RTTI_SYMBOL_1;
SelfManagedKafkaEventSource[_b] = { fqn: "@aws-cdk/aws-lambda-event-sources.SelfManagedKafkaEventSource", version: "1.152.0" };
//# sourceMappingURL=data:application/json;base64,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