################################################################################
#
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corp. 2017
# US Government Users Restricted Rights - Use, duplication disclosure restricted
# by GSA ADP Schedule Contract with IBM Corp.
#
################################################################################


import logging
import re

from watson_machine_learning_client.libs.repo.mlrepository import  ModelArtifact
from watson_machine_learning_client.libs.repo.mlrepository import MetaNames, MetaProps
from watson_machine_learning_client.libs.repo.mlrepository.scikit_model_artifact import ScikitModelArtifact
from watson_machine_learning_client.libs.repo.mlrepository.tensorflow_model_artifact import TensorflowModelArtifact
from watson_machine_learning_client.libs.repo.mlrepository.generic_archive_model_artifact import GenericArchiveModelArtifact
from watson_machine_learning_client.libs.repo.mlrepositoryartifact import SparkPipelineModelLoader, SparkPipelineModelContentLoader,\
    IBMSparkPipelineModelContentLoader, MLPipelineModelContentLoader
from watson_machine_learning_client.libs.repo.mlrepositoryartifact.content_loaders import ScikitPipelineModelContentLoader,TensorflowPipelineModelContentLoader
from watson_machine_learning_client.libs.repo.mlrepositoryartifact.scikit_pipeline_model_loader import ScikitPipelineModelLoader
from watson_machine_learning_client.libs.repo.mlrepositoryartifact.tensorflow_pipeline_model_loader import TensorflowPipelineModelLoader
from watson_machine_learning_client.libs.repo.mlrepositoryartifact.generic_file_pipeline_model_loader import GenericFilePipelineModelLoader
from watson_machine_learning_client.libs.repo.mlrepositoryartifact.hybrid_pipeline_model_artifact import HybridPipelineModelArtifact
from watson_machine_learning_client.libs.repo.mlrepositoryartifact.hybrid_pipeline_model_loader import  HybridPipelineModelLoader
from watson_machine_learning_client.libs.repo.swagger_client.models import AuthorRepository
from watson_machine_learning_client.libs.repo.util.generic_archive_file_check import GenericArchiveFrameworkCheck
import ast,json

from watson_machine_learning_client.libs.repo.util.library_imports import LibraryChecker
from watson_machine_learning_client.libs.repo.base_constants import *

lib_checker = LibraryChecker()

logger = logging.getLogger('ModelAdapter')


class ModelAdapter(object):
    """
    Adapter creating pipeline model artifact using output from service.
    """

    @staticmethod
    def __strip_output(output):
        return ast.literal_eval(json.dumps(output))

    def __init__(self, model_output, model_version_output, client):
        self.model_output = model_output
        self.model_version_output = model_version_output
        self.client = client
        self.model_type = model_output.entity['type']
        self.model_entity = model_output.entity
        self.model_metadata = model_output.metadata

    def artifact(self):
        if re.match('mllib', self.model_type) is not None:
            lib_checker.check_lib(PYSPARK)
            model_artifact_builder = type(
                "ModelArtifact",
                (SparkPipelineModelContentLoader, SparkPipelineModelLoader, ModelArtifact, object),
                {}
            )
        elif re.match('wml_1.', self.model_type) is not None:
            lib_checker.check_lib(PYSPARK)
            lib_checker.check_lib(MLPIPELINE)
            model_artifact_builder = type(
                "WMLModelArtifact",
                (MLPipelineModelContentLoader, SparkPipelineModelLoader, ModelArtifact, object),
                {}
            )
        elif re.match('scikit-learn', self.model_type) is not None:
            lib_checker.check_lib(SCIKIT)
            model_artifact_builder = type(
                "ScikitModelArtifact",
                (ScikitPipelineModelContentLoader, ScikitPipelineModelLoader, ScikitModelArtifact, object),
                {}
            )
        elif re.match('tensorflow', self.model_type) is not None:
            lib_checker.check_lib(TENSORFLOW)
            model_artifact_builder = type(
                "TensorflowModelArtifact",
                (TensorflowPipelineModelContentLoader, TensorflowPipelineModelLoader, TensorflowModelArtifact, object),
                {}
            )

        elif re.match('xgboost', self.model_type) is not None:
            lib_checker.check_lib(SCIKIT)
            lib_checker.check_lib(XGBOOST)
            model_artifact_builder = type(
                "ScikitModelArtifact",
                (ScikitPipelineModelContentLoader, ScikitPipelineModelLoader, ScikitModelArtifact, object),
                {}
            )
        elif re.match('hybrid', self.model_type) is not None or \
             re.match('wml-hybrid', self.model_type) is not None:
            model_artifact_builder = type(
                "HybridModelArtifact",
                (HybridPipelineModelLoader, HybridPipelineModelArtifact,object),
                {}
            )
        elif GenericArchiveFrameworkCheck.is_archive_framework(self.model_type):
            model_artifact_builder = type(
                "GenericModelArchiveArtifact",
                (GenericFilePipelineModelLoader, GenericArchiveModelArtifact, object),
                {}
            )
        else:
            raise ValueError('Invalid model_type: {}'.format(self.model_type.get('name')))

        prop_map = {
            MetaNames.TYPE: self.model_type
        }

        if self.model_metadata.created_at is not None:
            prop_map[MetaNames.CREATION_TIME] = self.model_output.metadata.created_at

        if self.model_output.metadata.modified_at is not None:
            prop_map[MetaNames.LAST_UPDATED] = self.model_output.metadata.modified_at

        if self.model_entity.get('runtime') is not None:
            runtimeval = self.model_entity.get('runtime')
            if runtimeval['href'] is not None:
                prop_map[MetaNames.RUNTIMES] = self.model_entity.get('runtime')['href']

        if self.model_entity.get('pipeline') is not None:
            runtimeval = self.model_entity.get('pipeline')
            if runtimeval['href'] is not None:
                prop_map[MetaNames.PIPELINE_UID] = self.model_entity.get('pipeline')['href']

        if self.model_entity.get('training_lib') is not None:
            runtimeval = self.model_entity.get('training_lib')
            if runtimeval['href'] is not None:
                prop_map[MetaNames.TRAINING_LIB_UID] = self.model_entity.get('training_lib')['href']
        if self.model_entity.get('space') is not None:
            runtimeval = self.model_entity.get('space')
            if runtimeval['href'] is not None:
                prop_map[MetaNames.SPACE_UID] = self.model_entity.get('space')['href']

        if self.model_entity.get('project') is not None:
            projectval = self.model_entity.get('project')
            if projectval['href'] is not None:
                prop_map[MetaNames.PROJECT_UID] = self.model_entity.get('project')['href']

        # pipeline_version is not' present for scikit model
        # if self.model_entity.get('type').startswith('mllib'):
        #     ##TODOOOO (Add pipeline href)
        #     if self.model_entity.get('pipeline') is not None:
        #         training_def_ver_url = self.model_entity.get('training_definition_url')
        #         prop_map[MetaNames.TRAINING_DEFINITION_VERSION_URL] = training_def_ver_url

        if self.model_entity.get('domain', None) is not None:
            prop_map[MetaNames.DOMAIN] = self.model_entity.get('domain')

        if self.model_entity.get('custom', None) is not None:
            prop_map[MetaNames.CUSTOM] = self.model_entity.get('custom')

        if self.model_entity.get('label_column', None) is not None:
            prop_map[MetaNames.LABEL_FIELD] = self.model_entity.get('label_column')

        if self.model_entity.get('description', None) is not None:
            prop_map[MetaNames.DESCRIPTION] = self.model_entity.get('description')
        model_url = self.model_output.metadata.href
        prop_map[MetaNames.MODEL_URL] = model_url
        # if self.model_entity.get('training_data_schema', None) is not None:
        #     prop_map[MetaNames.TRAINING_DATA_SCHEMA] = self.model_entity.get('training_data_schema')
        #
        # TODO: add later
        # if self.model_entity['schemas']['input_data_schema'] is not None:
        #      prop_map[MetaNames.INPUT_DATA_SCHEMA] = self.model_entity['schemas']['input_data_schema']
        # #
        # if self.model_entity['schemas']['output_data_schema'] is not None:
        #     prop_map[MetaNames.OUTPUT_DATA_SCHEMA] = self.model_entity['schemas']['output_data_schema']

        if self.model_entity.get('transformed_label', None) is not None:
            prop_map[MetaNames.TRANSFORMED_LABEL_FIELD] = self.model_entity.get('transformed_label')

        if self.model_entity.get('tags', None) is not None:
            prop_map[MetaNames.TAGS] = self.model_entity.get('tags')

        #if self.model_entity.get('author', None) and isinstance(self.model_entity.get('author'), AuthorRepository):
        # if self.model_entity.get('author', None) is not None:
        #     authorval = self.model_entity.get('author')
        #     if authorval.get('name', None ) is not None:
        #         prop_map[MetaNames.AUTHOR_NAME] = self.model_entity.get('author').get('name')

        if "training_data_references" in self.model_entity and self.model_entity.get('training_data_references', None) is not None:
                prop_map[MetaNames.TRAINING_DATA_REFERENCES] = self.model_entity.get('training_data_references')
# TODO Fix Evaluation
#         try:
#             evaluation_data = self.model_entity.get('evaluation', None)
#             if evaluation_data is not None:
#                 prop_map[MetaNames.EVALUATION_METHOD] = evaluation_data.get('method', None)
#                 if evaluation_data.get('metrics', None) is not None:
#                     prop_map[MetaNames.EVALUATION_METRICS] = evaluation_data.get('metrics')
#         #    evaluation_data = self.model_entity['evaluation']
#         #    if evaluation_data is not None:t
#         #        prop_map[MetaNames.EVALUATION_METHOD] = evaluation_data['method']
#         #        if evaluation_data['metrics'] is not None:
#         #           metrics = evaluation_data['metrics']
#         #            prop_map[MetaNames.EVALUATION_METRICS] = metrics
#         except KeyError:
#             print("No Evlauation method given")

        #name = ''.join(map(lambda x: chr(ord(x)), self.model_output.entity['name']))
        name = self.model_entity.get('name')

        model_id = self.model_metadata.guid


        model_artifact = model_artifact_builder(
            uid=model_id,
            name=name,
            meta_props=MetaProps(prop_map)
        )

        model_artifact.client = self.client

        if model_artifact.meta.prop(MetaNames.TYPE).startswith('mllib'):
            model_artifact._pipeline_url = self.model_entity.get('pipeline')['href']

        if model_artifact.meta.prop(MetaNames.TYPE).startswith('wml_1.'):
            model_artifact._pipeline_url = self.model_entity.get('pipeline')['href']
        if self.model_entity is not None:

            version_props = MetaProps({})
            # content_status, content_location = None, None
            if 'content_status' in self.model_entity:
                if self.model_entity['content_status'] is not None:
                    version_props.add(MetaNames.CONTENT_STATUS, self.model_entity['content_status'])

            if 'import' in self.model_entity:
                if self.model_entity['import'] is not None:
                    version_props.add(MetaNames.IMPORT, self.model_entity['import'])

            if 'hyper_parameters' in self.model_entity:
                if self.model_entity['hyper_parameters'] is not None:
                    version_props.add(MetaNames.HYPER_PARAMETERS, self.model_entity['hyper_parameters'])
                    version_props.add(MetaNames.HYPER_PARAMETERS, self.model_entitya['hyper_parameters'])

            model_artifact.meta.merge(version_props)

            model_artifact._content_href = "/v4/models/"+self.model_metadata.guid+"/content"
        else:
            model_artifact._content_href = None

        return model_artifact
