"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const util_1 = require("./util");
class Counter extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, { type: 'My::Counter', properties: { Count: props.Count } });
        this.arn = this.getAtt('Arn').toString();
        this.url = this.getAtt('URL').toString();
        this.count = props.Count;
    }
    increment(by = 1) {
        this.count += by;
    }
    get cfnProperties() {
        return { Count: this.count };
    }
}
function withoutHash(logId) {
    return logId.substr(0, logId.length - 8);
}
class CustomizableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, { type: 'MyResourceType', properties: props });
        if (props !== undefined) {
            this.prop1 = props.prop1;
            this.prop2 = props.prop2;
            this.prop3 = props.prop3;
        }
    }
    renderProperties() {
        const props = this.updatedProperites;
        const render = {};
        for (const key of Object.keys(props)) {
            render[key.toUpperCase()] = props[key];
        }
        return render;
    }
    get updatedProperites() {
        const props = {
            prop1: this.prop1,
            prop2: this.prop2,
            prop3: this.prop3,
        };
        const cleanProps = {};
        for (const key of Object.keys(props)) {
            if (props[key] === undefined) {
                continue;
            }
            cleanProps[key] = props[key];
        }
        return cleanProps;
    }
}
module.exports = {
    'all resources derive from Resource, which derives from Entity'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnResource(stack, 'MyResource', {
            type: 'MyResourceType',
            properties: {
                Prop1: 'p1', Prop2: 123
            }
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: {
                    Type: 'MyResourceType',
                    Properties: {
                        Prop1: 'p1',
                        Prop2: 123
                    }
                }
            }
        });
        test.done();
    },
    'resources must reside within a Stack and fail upon creation if not'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.CfnResource(root, 'R1', { type: 'ResourceType' }));
        test.done();
    },
    'all entities have a logical ID calculated based on their full path in the tree'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack');
        const level1 = new lib_1.Construct(stack, 'level1');
        const level2 = new lib_1.Construct(level1, 'level2');
        const level3 = new lib_1.Construct(level2, 'level3');
        const res1 = new lib_1.CfnResource(level1, 'childoflevel1', { type: 'MyResourceType1' });
        const res2 = new lib_1.CfnResource(level3, 'childoflevel3', { type: 'MyResourceType2' });
        test.equal(withoutHash(stack.resolve(res1.logicalId)), 'level1childoflevel1');
        test.equal(withoutHash(stack.resolve(res2.logicalId)), 'level1level2level3childoflevel3');
        test.done();
    },
    'resource.props can only be accessed by derived classes'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 10 });
        res.increment();
        res.increment(2);
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 13 } }
            }
        });
        test.done();
    },
    'resource attributes can be retrieved using getAtt(s) or attribute properties'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 10 });
        new lib_1.CfnResource(stack, 'YourResource', {
            type: 'Type',
            properties: {
                CounterName: res.getAtt('Name'),
                CounterArn: res.arn,
                CounterURL: res.url,
            }
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 10 } },
                YourResource: {
                    Type: 'Type',
                    Properties: {
                        CounterName: { 'Fn::GetAtt': ['MyResource', 'Name'] },
                        CounterArn: { 'Fn::GetAtt': ['MyResource', 'Arn'] },
                        CounterURL: { 'Fn::GetAtt': ['MyResource', 'URL'] }
                    }
                }
            }
        });
        test.done();
    },
    'ARN-type resource attributes have some common functionality'(test) {
        const stack = new lib_1.Stack();
        const res = new Counter(stack, 'MyResource', { Count: 1 });
        new lib_1.CfnResource(stack, 'MyResource2', {
            type: 'Type',
            properties: {
                Perm: res.arn
            }
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyResource: { Type: 'My::Counter', Properties: { Count: 1 } },
                MyResource2: {
                    Type: 'Type',
                    Properties: {
                        Perm: {
                            'Fn::GetAtt': ['MyResource', 'Arn']
                        }
                    }
                }
            }
        });
        test.done();
    },
    'resource.addDependency(e) can be used to add a DependsOn on another resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new Counter(stack, 'Counter1', { Count: 1 });
        const r2 = new Counter(stack, 'Counter2', { Count: 1 });
        const r3 = new lib_1.CfnResource(stack, 'Resource3', { type: 'MyResourceType' });
        r2.node.addDependency(r1);
        r2.node.addDependency(r3);
        lib_1.ConstructNode.prepare(stack.node);
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Counter1: {
                    Type: 'My::Counter',
                    Properties: { Count: 1 }
                },
                Counter2: {
                    Type: 'My::Counter',
                    Properties: { Count: 1 },
                    DependsOn: [
                        'Counter1',
                        'Resource3'
                    ]
                },
                Resource3: { Type: 'MyResourceType' }
            }
        });
        test.done();
    },
    'if addDependency is called multiple times with the same resource, it will only appear once'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const r1 = new Counter(stack, 'Counter1', { Count: 1 });
        const dependent = new lib_1.CfnResource(stack, 'Dependent', { type: 'R' });
        // WHEN
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        dependent.addDependsOn(r1);
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Counter1: {
                    Type: 'My::Counter',
                    Properties: {
                        Count: 1
                    }
                },
                Dependent: {
                    Type: 'R',
                    DependsOn: [
                        'Counter1'
                    ]
                }
            }
        });
        test.done();
    },
    'conditions can be attached to a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        const cond = new lib_1.CfnCondition(stack, 'MyCondition', { expression: lib_1.Fn.conditionNot(lib_1.Fn.conditionEquals('a', 'b')) });
        r1.cfnOptions.condition = cond;
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: { Resource: { Type: 'Type', Condition: 'MyCondition' } },
            Conditions: { MyCondition: { 'Fn::Not': [{ 'Fn::Equals': ['a', 'b'] }] } }
        });
        test.done();
    },
    'creation/update/updateReplace/deletion policies can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.creationPolicy = { autoScalingCreationPolicy: { minSuccessfulInstancesPercent: 10 } };
        // tslint:disable-next-line:max-line-length
        r1.cfnOptions.updatePolicy = {
            autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: false },
            autoScalingReplacingUpdate: { willReplace: true },
            codeDeployLambdaAliasUpdate: {
                applicationName: 'CodeDeployApplication',
                deploymentGroupName: 'CodeDeployDeploymentGroup',
                beforeAllowTrafficHook: 'lambda1',
            },
        };
        r1.cfnOptions.deletionPolicy = lib_1.CfnDeletionPolicy.RETAIN;
        r1.cfnOptions.updateReplacePolicy = lib_1.CfnDeletionPolicy.SNAPSHOT;
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    CreationPolicy: { AutoScalingCreationPolicy: { MinSuccessfulInstancesPercent: 10 } },
                    UpdatePolicy: {
                        AutoScalingScheduledAction: { IgnoreUnmodifiedGroupSizeProperties: false },
                        AutoScalingReplacingUpdate: { WillReplace: true },
                        CodeDeployLambdaAliasUpdate: {
                            ApplicationName: 'CodeDeployApplication',
                            DeploymentGroupName: 'CodeDeployDeploymentGroup',
                            BeforeAllowTrafficHook: 'lambda1',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Snapshot'
                }
            }
        });
        test.done();
    },
    'update policies UseOnlineResharding flag'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.updatePolicy = { useOnlineResharding: true };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    UpdatePolicy: {
                        UseOnlineResharding: true,
                    },
                }
            }
        });
        test.done();
    },
    'metadata can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.metadata = {
            MyKey: 10,
            MyValue: 99
        };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    Metadata: {
                        MyKey: 10,
                        MyValue: 99
                    }
                }
            }
        });
        test.done();
    },
    'the "type" property is required when creating a resource'(test) {
        const stack = new lib_1.Stack();
        test.throws(() => new lib_1.CfnResource(stack, 'Resource', { notypehere: true }));
        test.done();
    },
    'removal policy is a high level abstraction of deletion policy used by l2'(test) {
        const stack = new lib_1.Stack();
        const retain = new lib_1.CfnResource(stack, 'Retain', { type: 'T1' });
        const destroy = new lib_1.CfnResource(stack, 'Destroy', { type: 'T3' });
        const def = new lib_1.CfnResource(stack, 'Default1', { type: 'T4' });
        const def2 = new lib_1.CfnResource(stack, 'Default2', { type: 'T4' });
        retain.applyRemovalPolicy(lib_1.RemovalPolicy.RETAIN);
        destroy.applyRemovalPolicy(lib_1.RemovalPolicy.DESTROY);
        def.applyRemovalPolicy(undefined, { default: lib_1.RemovalPolicy.DESTROY });
        def2.applyRemovalPolicy(undefined);
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Retain: { Type: 'T1', DeletionPolicy: 'Retain', UpdateReplacePolicy: 'Retain' },
                Destroy: { Type: 'T3', DeletionPolicy: 'Delete', UpdateReplacePolicy: 'Delete' },
                Default1: { Type: 'T4', DeletionPolicy: 'Delete', UpdateReplacePolicy: 'Delete' },
                Default2: { Type: 'T4', DeletionPolicy: 'Retain', UpdateReplacePolicy: 'Retain' } // implicit default
            }
        });
        test.done();
    },
    'addDependency adds all dependencyElements of dependent constructs'(test) {
        class C1 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                this.r1 = new lib_1.CfnResource(this, 'R1', { type: 'T1' });
                this.r2 = new lib_1.CfnResource(this, 'R2', { type: 'T2' });
            }
        }
        class C2 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                this.r3 = new lib_1.CfnResource(this, 'R3', { type: 'T3' });
            }
        }
        // C3 returns [ c2 ] for it's dependency elements
        // this should result in 'flattening' the list of elements.
        class C3 extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new C2(this, 'C2');
            }
        }
        const stack = new lib_1.Stack();
        const c1 = new C1(stack, 'MyC1');
        const c2 = new C2(stack, 'MyC2');
        const c3 = new C3(stack, 'MyC3');
        const dependingResource = new lib_1.CfnResource(stack, 'MyResource', { type: 'R' });
        dependingResource.node.addDependency(c1, c2);
        dependingResource.node.addDependency(c3);
        lib_1.ConstructNode.prepare(stack.node);
        test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyC1R1FB2A562F: { Type: 'T1' },
                MyC1R2AE2B5066: { Type: 'T2' },
                MyC2R3809EEAD6: { Type: 'T3' },
                MyC3C2R38CE6F9F7: { Type: 'T3' },
                MyResource: { Type: 'R',
                    DependsOn: ['MyC1R1FB2A562F',
                        'MyC1R2AE2B5066',
                        'MyC2R3809EEAD6',
                        'MyC3C2R38CE6F9F7'] } } });
        test.done();
    },
    'resource.ref returns the {Ref} token'(test) {
        const stack = new lib_1.Stack();
        const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'R' });
        test.deepEqual(stack.resolve(r.ref), { Ref: 'MyResource' });
        test.done();
    },
    'overrides': {
        'addOverride(p, v) allows assigning arbitrary values to synthesized resource definitions'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'AWS::Resource::Type' });
            // WHEN
            r.addOverride('Type', 'YouCanEvenOverrideTheType');
            r.addOverride('Metadata', { Key: 12 });
            r.addOverride('Use.Dot.Notation', 'To create subtrees');
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'YouCanEvenOverrideTheType',
                        Use: { Dot: { Notation: 'To create subtrees' } },
                        Metadata: { Key: 12 } } } });
            test.done();
        },
        'addOverride(p, null) will assign an "null" value'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                        }
                    }
                }
            });
            // WHEN
            r.addOverride('Properties.Hello.World.Value2', null);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello', Value2: null } } } } } });
            test.done();
        },
        'addOverride(p, undefined) can be used to delete a value'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                        }
                    }
                }
            });
            // WHEN
            r.addOverride('Properties.Hello.World.Value2', undefined);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello' } } } } } });
            test.done();
        },
        'addOverride(p, undefined) will not create empty trees'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', { type: 'AWS::Resource::Type' });
            // WHEN
            r.addPropertyOverride('Tree.Exists', 42);
            r.addPropertyOverride('Tree.Does.Not.Exist', undefined);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Tree: { Exists: 42 } } } } });
            test.done();
        },
        'addDeletionOverride(p) and addPropertyDeletionOverride(pp) are sugar `undefined`'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: {
                            Value1: 'Hello',
                            Value2: 129,
                            Value3: ['foo', 'bar']
                        }
                    }
                }
            });
            // WHEN
            r.addDeletionOverride('Properties.Hello.World.Value2');
            r.addPropertyDeletionOverride('Hello.World.Value3');
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Value1: 'Hello' } } } } } });
            test.done();
        },
        'addOverride(p, v) will overwrite any non-objects along the path'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: {
                    Hello: {
                        World: 42
                    }
                }
            });
            // WHEN
            r.addOverride('Properties.Override1', ['Hello', 123]);
            r.addOverride('Properties.Override1.Override2', { Heyy: [1] });
            r.addOverride('Properties.Hello.World.Foo.Bar', 42);
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Foo: { Bar: 42 } } },
                            Override1: {
                                Override2: { Heyy: [1] }
                            } } } } });
            test.done();
        },
        'addPropertyOverride(pp, v) is a sugar for overriding properties'(test) {
            // GIVEN
            const stack = new lib_1.Stack();
            const r = new lib_1.CfnResource(stack, 'MyResource', {
                type: 'AWS::Resource::Type',
                properties: { Hello: { World: 42 } }
            });
            // WHEN
            r.addPropertyOverride('Hello.World', { Hey: 'Jude' });
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'AWS::Resource::Type',
                        Properties: { Hello: { World: { Hey: 'Jude' } } } } } });
            test.done();
        },
        'overrides are applied after render'(test) {
            // GIVEN
            class MyResource extends lib_1.CfnResource {
                renderProperties() {
                    return { Fixed: 123 };
                }
            }
            const stack = new lib_1.Stack();
            const cfn = new MyResource(stack, 'rr', { type: 'AWS::Resource::Type' });
            // WHEN
            cfn.addPropertyOverride('Boom', 'Hi');
            cfn.addOverride('Properties.Foo.Bar', 'Bar');
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack), {
                Resources: {
                    rr: {
                        Type: 'AWS::Resource::Type',
                        Properties: {
                            Fixed: 123,
                            Boom: 'Hi',
                            Foo: {
                                Bar: 'Bar'
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'using mutable properties': {
            'can be used by derived classes to specify overrides before render()'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource', {
                    prop1: 'foo'
                });
                r.prop2 = 'bar';
                test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'MyResourceType',
                            Properties: { PROP1: 'foo', PROP2: 'bar' } } } });
                test.done();
            },
            '"properties" is undefined'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource');
                r.prop3 = 'zoo';
                test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'MyResourceType',
                            Properties: { PROP3: 'zoo' } } } });
                test.done();
            },
            '"properties" is empty'(test) {
                const stack = new lib_1.Stack();
                const r = new CustomizableResource(stack, 'MyResource', {});
                r.prop3 = 'zoo';
                r.prop2 = 'hey';
                test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'MyResourceType',
                            Properties: { PROP2: 'hey', PROP3: 'zoo' } } } });
                test.done();
            },
        }
    },
    '"aws:cdk:path" metadata is added if "aws:cdk:path-metadata" context is set to true'(test) {
        const stack = new lib_1.Stack();
        stack.node.setContext(cxapi.PATH_METADATA_ENABLE_CONTEXT, true);
        const parent = new lib_1.Construct(stack, 'Parent');
        new lib_1.CfnResource(parent, 'MyResource', {
            type: 'MyResourceType',
        });
        test.deepEqual(util_1.toCloudFormation(stack), { Resources: { ParentMyResource4B1FDBCC: { Type: 'MyResourceType',
                    Metadata: { [cxapi.PATH_METADATA_KEY]: 'Parent/MyResource' } } } });
        test.done();
    },
    'cross-stack construct dependencies are not rendered but turned into stack dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stackA = new lib_1.Stack(app, 'StackA');
        const resA = new lib_1.CfnResource(stackA, 'Resource', { type: 'R' });
        const stackB = new lib_1.Stack(app, 'StackB');
        const resB = new lib_1.CfnResource(stackB, 'Resource', { type: 'R' });
        // WHEN
        resB.node.addDependency(resA);
        // THEN
        const assembly = app.synth();
        const templateB = assembly.getStackByName(stackB.stackName).template;
        test.deepEqual(templateB, {
            Resources: {
                Resource: {
                    Type: 'R'
                    // Notice absence of 'DependsOn'
                }
            }
        });
        test.deepEqual(stackB.dependencies.map(s => s.node.id), ['StackA']);
        test.done();
    },
    'enableVersionUpgrade can be set on a resource'(test) {
        const stack = new lib_1.Stack();
        const r1 = new lib_1.CfnResource(stack, 'Resource', { type: 'Type' });
        r1.cfnOptions.updatePolicy = {
            enableVersionUpgrade: true
        };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                Resource: {
                    Type: 'Type',
                    UpdatePolicy: {
                        EnableVersionUpgrade: true
                    }
                }
            }
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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