"use strict";
const lib_1 = require("../lib");
const context_provider_1 = require("../lib/context-provider");
/**
 * Get the expected context key from a stack with missing parameters
 */
function expectedContextKey(stack) {
    const missing = lib_1.ConstructNode.synth(stack.node).manifest.missing;
    if (!missing || missing.length !== 1) {
        throw new Error('Expecting assembly to include a single missing context report');
    }
    return missing[0].key;
}
module.exports = {
    'AvailabilityZoneProvider returns a list with dummy values if the context is not available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['dummy1a', 'dummy1b', 'dummy1c']);
        test.done();
    },
    'AvailabilityZoneProvider will return context list if available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, ['us-east-1a', 'us-east-1b']);
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['us-east-1a', 'us-east-1b']);
        test.done();
    },
    'AvailabilityZoneProvider will complain if not given a list'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, 'not-a-list');
        test.throws(() => stack.availabilityZones);
        test.done();
    },
    'ContextProvider consistently generates a key'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'ssm',
            props: {
                parameterName: 'foo',
                anyStringParam: 'bar'
            },
        });
        test.deepEqual(key, {
            key: 'ssm:account=12345:anyStringParam=bar:parameterName=foo:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                parameterName: 'foo',
                anyStringParam: 'bar'
            }
        });
        const complexKey = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'vpc',
            props: {
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            }
        });
        test.deepEqual(complexKey, {
            key: 'vpc:account=12345:cidrBlock=192.168.0.16:igw=false:region=us-east-1:tags.Env=Preprod:tags.Name=MyVPC',
            props: {
                account: '12345',
                region: 'us-east-1',
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            }
        });
        test.done();
    },
    'Key generation can contain arbitrarily deep structures'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'provider',
            props: {
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            }
        });
        // THEN
        test.deepEqual(key, {
            key: 'provider:account=12345:list.0.key=key1:list.0.value=value1:list.1.key=key2:list.1.value=value2:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            }
        });
        test.done();
    },
    'Keys with undefined values are not serialized'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const result = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'provider',
            props: {
                p1: 42,
                p2: undefined,
            },
        });
        // THEN
        test.deepEqual(result, {
            key: 'provider:account=12345:p1=42:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                p1: 42,
                p2: undefined,
            },
        });
        test.done();
    },
    'context provider errors are attached to tree'(test) {
        const contextProps = { provider: 'bloop' };
        const contextKey = 'bloop:account=12345:region=us-east-1'; // Depends on the mangling algo
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // NOTE: error key is inlined here because it's part of the CX-API
        // compatibility surface.
        stack.node.setContext(contextKey, { $providerError: 'I had a boo-boo' });
        const construct = new lib_1.Construct(stack, 'Child');
        // Verify that we got the right hardcoded key above, give a descriptive error if not
        test.equals(context_provider_1.ContextProvider.getKey(construct, contextProps).key, contextKey);
        // WHEN
        context_provider_1.ContextProvider.getValue(construct, {
            ...contextProps,
            dummyValue: undefined,
        });
        // THEN
        const error = construct.node.metadata.find(m => m.type === 'aws:cdk:error');
        test.equals(error && error.data, 'I had a boo-boo');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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