"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
function identity(x) {
    return x;
}
exports.stringToCloudFormation = identity;
exports.booleanToCloudFormation = identity;
exports.objectToCloudFormation = identity;
exports.numberToCloudFormation = identity;
/**
 * The date needs to be formatted as an ISO date in UTC
 *
 * Some usage sites require a date, some require a timestamp. We'll
 * always output a timestamp and hope the parser on the other end
 * is smart enough to ignore the time part... (?)
 */
function dateToCloudFormation(x) {
    if (!x) {
        return undefined;
    }
    // tslint:disable-next-line:max-line-length
    return `${x.getUTCFullYear()}-${pad(x.getUTCMonth() + 1)}-${pad(x.getUTCDate())}T${pad(x.getUTCHours())}:${pad(x.getUTCMinutes())}:${pad(x.getUTCSeconds())}`;
}
exports.dateToCloudFormation = dateToCloudFormation;
/**
 * Pad a number to 2 decimal places
 */
function pad(x) {
    if (x < 10) {
        return '0' + x.toString();
    }
    return x.toString();
}
/**
 * Turn a tag object into the proper CloudFormation representation
 */
function cfnTagToCloudFormation(x) {
    return {
        Key: x.key,
        Value: x.value
    };
}
exports.cfnTagToCloudFormation = cfnTagToCloudFormation;
function listMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        return x.map(elementMapper);
    };
}
exports.listMapper = listMapper;
function hashMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        const ret = {};
        Object.keys(x).forEach((key) => {
            ret[key] = elementMapper(x[key]);
        });
        return ret;
    };
}
exports.hashMapper = hashMapper;
/**
 * Return a union mapper
 *
 * Takes a list of validators and a list of mappers, which should correspond pairwise.
 *
 * The mapper of the first successful validator will be called.
 */
function unionMapper(validators, mappers) {
    if (validators.length !== mappers.length) {
        throw Error('Not the same amount of validators and mappers passed to unionMapper()');
    }
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        for (let i = 0; i < validators.length; i++) {
            if (validators[i](x).isSuccess) {
                return mappers[i](x);
            }
        }
        // Should not be possible because the union must have passed validation before this function
        // will be called, but catch it anyway.
        throw new TypeError('No validators matched in the union()');
    };
}
exports.unionMapper = unionMapper;
// ----------------------------------------------------------------------
// VALIDATORS
//
// These are used while checking that supplied property bags match the expected schema
//
// We have a couple of datatypes that model validation errors and collections of validation
// errors (together forming a tree of errors so that we can trace validation errors through
// an object graph), and validators.
//
// Validators are simply functions that take a value and return a validation results. Then
// we have some combinators to turn primitive validators into more complex validators.
//
/**
 * Representation of validation results
 *
 * Models a tree of validation errors so that we have as much information as possible
 * about the failure that occurred.
 */
class ValidationResult {
    constructor(errorMessage = '', results = new ValidationResults()) {
        this.errorMessage = errorMessage;
        this.results = results;
    }
    get isSuccess() {
        return !this.errorMessage && this.results.isSuccess;
    }
    /**
     * Turn a failed validation into an exception
     */
    assertSuccess() {
        if (!this.isSuccess) {
            let message = this.errorTree();
            // The first letter will be lowercase, so uppercase it for a nicer error message
            message = message.substr(0, 1).toUpperCase() + message.substr(1);
            throw new CfnSynthesisError(message);
        }
    }
    /**
     * Return a string rendering of the tree of validation failures
     */
    errorTree() {
        const childMessages = this.results.errorTreeList();
        return this.errorMessage + (childMessages.length ? `\n  ${childMessages.replace(/\n/g, '\n  ')}` : '');
    }
    /**
     * Wrap this result with an error message, if it concerns an error
     */
    prefix(message) {
        if (this.isSuccess) {
            return this;
        }
        return new ValidationResult(`${message}: ${this.errorMessage}`, this.results);
    }
}
exports.ValidationResult = ValidationResult;
/**
 * A collection of validation results
 */
class ValidationResults {
    constructor(results = []) {
        this.results = results;
    }
    collect(result) {
        // Only collect failures
        if (!result.isSuccess) {
            this.results.push(result);
        }
    }
    get isSuccess() {
        return this.results.every(x => x.isSuccess);
    }
    errorTreeList() {
        return this.results.map(child => child.errorTree()).join('\n');
    }
    /**
     * Wrap up all validation results into a single tree node
     *
     * If there are failures in the collection, add a message, otherwise
     * return a success.
     */
    wrap(message) {
        if (this.isSuccess) {
            return exports.VALIDATION_SUCCESS;
        }
        return new ValidationResult(message, this);
    }
}
exports.ValidationResults = ValidationResults;
// Singleton object to save on allocations
exports.VALIDATION_SUCCESS = new ValidationResult();
/**
 * Return whether this object can be validated at all
 *
 * True unless it's undefined or a CloudFormation intrinsic
 */
function canInspect(x) {
    // Note: using weak equality on purpose, we also want to catch undefined
    return (x != null && !isCloudFormationIntrinsic(x));
}
exports.canInspect = canInspect;
// CloudFormation validators for primitive types
function validateString(x) {
    if (canInspect(x) && typeof x !== 'string') {
        return new ValidationResult(`${JSON.stringify(x)} should be a string`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateString = validateString;
function validateNumber(x) {
    if (canInspect(x) && typeof x !== 'number') {
        return new ValidationResult(`${JSON.stringify(x)} should be a number`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateNumber = validateNumber;
function validateBoolean(x) {
    if (canInspect(x) && typeof x !== 'boolean') {
        return new ValidationResult(`${JSON.stringify(x)} should be a boolean`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateBoolean = validateBoolean;
function validateDate(x) {
    if (canInspect(x) && !(x instanceof Date)) {
        return new ValidationResult(`${JSON.stringify(x)} should be a Date`);
    }
    if (x !== undefined && isNaN(x.getTime())) {
        return new ValidationResult('got an unparseable Date');
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateDate = validateDate;
function validateObject(x) {
    if (canInspect(x) && typeof x !== 'object') {
        return new ValidationResult(`${JSON.stringify(x)} should be an 'object'`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateObject = validateObject;
function validateCfnTag(x) {
    if (!canInspect(x)) {
        return exports.VALIDATION_SUCCESS;
    }
    if (x.key == null || x.value == null) {
        return new ValidationResult(`${JSON.stringify(x)} should have a 'key' and a 'value' property`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateCfnTag = validateCfnTag;
/**
 * Return a list validator based on the given element validator
 */
function listValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        if (!x.forEach) {
            return new ValidationResult(`${JSON.stringify(x)} should be a list`);
        }
        for (let i = 0; i < x.length; i++) {
            const element = x[i];
            const result = elementValidator(element);
            if (!result.isSuccess) {
                return result.prefix(`element ${i}`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.listValidator = listValidator;
/**
 * Return a hash validator based on the given element validator
 */
function hashValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        for (const key of Object.keys(x)) {
            const result = elementValidator(x[key]);
            if (!result.isSuccess) {
                return result.prefix(`element '${key}'`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.hashValidator = hashValidator;
/**
 * Decorate a validator with a message clarifying the property the failure is for.
 */
function propertyValidator(propName, validator) {
    return (x) => {
        return validator(x).prefix(propName);
    };
}
exports.propertyValidator = propertyValidator;
/**
 * Return a validator that will fail if the passed property is not present
 *
 * Does not distinguish between the property actually not being present, vs being present but 'null'
 * or 'undefined' (courtesy of JavaScript), which is generally the behavior that we want.
 *
 * Empty strings are considered "present"--don't know if this agrees with how CloudFormation looks
 * at the world.
 */
function requiredValidator(x) {
    if (x == null) {
        return new ValidationResult('required but missing');
    }
    return exports.VALIDATION_SUCCESS;
}
exports.requiredValidator = requiredValidator;
/**
 * Require a property from a property bag.
 *
 * @param props  the property bag from which a property is required.
 * @param name   the name of the required property.
 * @param typeName the name of the construct type that requires the property
 *
 * @returns the value of ``props[name]``
 *
 * @throws if the property ``name`` is not present in ``props``.
 */
function requireProperty(props, name, context) {
    if (!(name in props)) {
        throw new Error(`${context.toString()} is missing required property: ${name}`);
    }
    const value = props[name];
    // Possibly add type-checking here...
    return value;
}
exports.requireProperty = requireProperty;
/**
 * Validates if any of the given validators matches
 *
 * We add either/or words to the front of the error mesages so that they read
 * more nicely. Example:
 *
 *   Properties not correct for 'FunctionProps'
 *     codeUri: not one of the possible types
 *       either: properties not correct for 'S3LocationProperty'
 *         bucket: required but missing
 *         key: required but missing
 *         version: required but missing
 *       or: '3' should be a 'string'
 *
 */
function unionValidator(...validators) {
    return (x) => {
        const results = new ValidationResults();
        let eitherOr = 'either';
        for (const validator of validators) {
            const result = validator(x);
            if (result.isSuccess) {
                return result;
            }
            results.collect(result.prefix(eitherOr));
            eitherOr = 'or';
        }
        return results.wrap('not one of the possible types');
    };
}
exports.unionValidator = unionValidator;
/**
 * Return whether the indicated value represents a CloudFormation intrinsic.
 *
 * CloudFormation intrinsics are modeled as objects with a single key, which
 * look like: { "Fn::GetAtt": [...] } or similar.
 */
function isCloudFormationIntrinsic(x) {
    if (!(typeof x === 'object')) {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || keys[0].substr(0, 4) === 'Fn::';
}
// Cannot be public because JSII gets confused about es5.d.ts
class CfnSynthesisError extends Error {
    constructor() {
        super(...arguments);
        this.type = 'CfnSynthesisError';
    }
}
//# sourceMappingURL=data:application/json;base64,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