"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stack_1 = require("./stack");
const util_1 = require("./util");
/**
 * Adds a dependency between two resources or stacks, across stack and nested
 * stack boundaries.
 *
 * The algorithm consists of:
 * - Try to find the deepest common stack between the two elements
 * - If there isn't a common stack, it means the elements belong to two
 *   disjoined stack-trees and therefore we apply the dependency at the
 *   assembly/app level between the two topl-level stacks.
 * - If we did find a common stack, we apply the dependency as a CloudFormation
 *   "DependsOn" between the resources that "represent" our source and target
 *   either directly or through the AWS::CloudFormation::Stack resources that
 *   "lead" to them.
 *
 * @param source The source resource/stack (the depedent)
 * @param target The target resource/stack (the dependency)
 * @param reason Optional resource to associate with the dependency for
 * diagnostics
 */
function addDependency(source, target, reason) {
    if (source === target) {
        return;
    }
    const sourceStack = stack_1.Stack.of(source);
    const targetStack = stack_1.Stack.of(target);
    // find the deepest common stack between the two elements
    const sourcePath = util_1.pathToTopLevelStack(sourceStack);
    const targetPath = util_1.pathToTopLevelStack(targetStack);
    const commonStack = util_1.findLastCommonElement(sourcePath, targetPath);
    // if there is no common stack, then define an assembly-level dependency
    // between the two top-level stacks
    if (!commonStack) {
        const topLevelSource = sourcePath[0]; // first path element is the top-level stack
        const topLevelTarget = targetPath[0];
        topLevelSource._addAssemblyDependency(topLevelTarget, reason);
        return;
    }
    // assertion: at this point if source and target are stacks, both are nested stacks.
    // since we have a common stack, it is impossible that both are top-level
    // stacks, so let's examine the two cases where one of them is top-level and
    // the other is nested.
    // case 1 - source is top-level and target is nested: this implies that
    // `target` is a direct or indirect nested stack of `source`, and an explicit
    // dependency is not required because nested stacks will always be deployed
    // before their parents.
    if (commonStack === source) {
        return;
    }
    // case 2 - source is nested and target is top-level: this implies that
    // `source` is a direct or indirect nested stack of `target`, and this is not
    // possible (nested stacks cannot depend on their parents).
    if (commonStack === target) {
        throw new Error(`Nested stack '${sourceStack.node.path}' cannot depend on a parent stack '${targetStack.node.path}'`);
    }
    // we have a common stack from which we can reach both `source` and `target`
    // now we need to find two resources which are defined directly in this stack
    // and which can "lead us" to the source/target.
    const sourceResource = resourceInCommonStackFor(source);
    const targetResource = resourceInCommonStackFor(target);
    sourceResource._addResourceDependency(targetResource);
    function resourceInCommonStackFor(element) {
        const resource = stack_1.Stack.isStack(element) ? element.nestedStackResource : element;
        if (!resource) {
            throw new Error('assertion failure'); // see "assertion" above
        }
        const resourceStack = stack_1.Stack.of(resource);
        // we reached a resource defined in the common stack
        if (commonStack === resourceStack) {
            return resource;
        }
        return resourceInCommonStackFor(resourceStack);
    }
}
exports.addDependency = addDependency;
//# sourceMappingURL=data:application/json;base64,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