"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const path = require("path");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const synthetics = require("../lib");
describe(synthetics.Code.fromInline, () => {
    test('fromInline works', () => {
        // GIVEN
        const stack = new core_1.Stack(new core_1.App(), 'canaries');
        // WHEN
        const inline = synthetics.Code.fromInline(`
      exports.handler = async () => {
        console.log(\'hello world\');
      };`);
        // THEN
        expect(inline.bind(stack, 'index.handler').inlineCode).toEqual(`
      exports.handler = async () => {
        console.log(\'hello world\');
      };`);
    });
    test('fails if empty', () => {
        expect(() => synthetics.Code.fromInline(''))
            .toThrowError('Canary inline code cannot be empty');
    });
    test('fails if handler is not "index.handler"', () => {
        // GIVEN
        const stack = new core_1.Stack(new core_1.App(), 'canaries');
        // THEN
        expect(() => synthetics.Code.fromInline('code').bind(stack, 'canary.handler'))
            .toThrowError('The handler for inline code must be "index.handler" (got "canary.handler")');
    });
});
describe(synthetics.Code.fromAsset, () => {
    test('fromAsset works', () => {
        var _a, _b;
        // GIVEN
        const stack = new core_1.Stack(new core_1.App(), 'canaries');
        // WHEN
        const directoryAsset = synthetics.Code.fromAsset(path.join(__dirname, 'canaries'));
        new synthetics.Canary(stack, 'Canary', {
            test: synthetics.Test.custom({
                handler: 'canary.handler',
                code: directoryAsset,
            }),
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
            Code: {
                Handler: 'canary.handler',
                S3Bucket: stack.resolve((_a = directoryAsset.bind(stack, 'canary.handler').s3Location) === null || _a === void 0 ? void 0 : _a.bucketName),
                S3Key: stack.resolve((_b = directoryAsset.bind(stack, 'canary.handler').s3Location) === null || _b === void 0 ? void 0 : _b.objectKey),
            },
        });
    });
    test('only one Asset object gets created even if multiple canaries use the same AssetCode', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'canaries');
        // WHEN
        const directoryAsset = synthetics.Code.fromAsset(path.join(__dirname, 'canaries'));
        new synthetics.Canary(stack, 'Canary1', {
            test: synthetics.Test.custom({
                handler: 'canary.handler',
                code: directoryAsset,
            }),
        });
        new synthetics.Canary(stack, 'Canary2', {
            test: synthetics.Test.custom({
                handler: 'canary.handler',
                code: directoryAsset,
            }),
        });
        // THEN
        const assembly = app.synth();
        const synthesized = assembly.stacks[0];
        expect(synthesized.assets.length).toEqual(1);
    });
    test('fails if path does not exist', () => {
        const assetPath = path.join(__dirname, 'does-not-exist');
        expect(() => synthetics.Code.fromAsset(assetPath))
            .toThrowError(`${assetPath} is not a valid path`);
    });
    test('fails if non-zip asset is used', () => {
        // GIVEN
        const stack = new core_1.Stack(new core_1.App(), 'canaries');
        // THEN
        const assetPath = path.join(__dirname, 'canaries', 'nodejs', 'node_modules', 'canary.js');
        expect(() => synthetics.Code.fromAsset(assetPath).bind(stack, 'canary.handler'))
            .toThrowError(`Asset must be a .zip file or a directory (${assetPath})`);
    });
    test('fails if "nodejs/node_modules" folder structure not used', () => {
        // GIVEN
        const stack = new core_1.Stack(new core_1.App(), 'canaries');
        // THEN
        const assetPath = path.join(__dirname, 'canaries', 'nodejs', 'node_modules');
        expect(() => synthetics.Code.fromAsset(assetPath).bind(stack, 'canary.handler'))
            .toThrowError(`The canary resource requires that the handler is present at "nodejs/node_modules/canary.js" but not found at ${assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch)`);
    });
    test('fails if handler is specified incorrectly', () => {
        // GIVEN
        const stack = new core_1.Stack(new core_1.App(), 'canaries');
        // THEN
        const assetPath = path.join(__dirname, 'canaries', 'nodejs', 'node_modules');
        expect(() => synthetics.Code.fromAsset(assetPath).bind(stack, 'incorrect.handler'))
            .toThrowError(`The canary resource requires that the handler is present at "nodejs/node_modules/incorrect.js" but not found at ${assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch)`);
    });
});
describe(synthetics.Code.fromBucket, () => {
    test('fromBucket works', () => {
        var _a, _b;
        // GIVEN
        const stack = new core_1.Stack(new core_1.App(), 'canaries');
        const bucket = new s3.Bucket(stack, 'CodeBucket');
        // WHEN
        const code = synthetics.Code.fromBucket(bucket, 'code.js');
        const codeConfig = code.bind(stack, 'code.handler');
        // THEN
        expect((_a = codeConfig.s3Location) === null || _a === void 0 ? void 0 : _a.bucketName).toEqual(bucket.bucketName);
        expect((_b = codeConfig.s3Location) === null || _b === void 0 ? void 0 : _b.objectKey).toEqual('code.js');
    });
});
//# sourceMappingURL=data:application/json;base64,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