"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const synthetics = require("../lib");
test('Basic canary properties work', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        canaryName: 'mycanary',
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
        successRetentionPeriod: core_1.Duration.days(10),
        failureRetentionPeriod: core_1.Duration.days(10),
        startAfterCreation: false,
        timeToLive: core_1.Duration.minutes(30),
        runtime: synthetics.Runtime.SYNTHETICS_1_0,
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        Name: 'mycanary',
        SuccessRetentionPeriod: 10,
        FailureRetentionPeriod: 10,
        StartCanaryAfterCreation: false,
        Schedule: assert_1.objectLike({ DurationInSeconds: '1800' }),
        RuntimeVersion: 'syn-1.0',
    });
});
test('Canary can have generated name', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        Name: 'canariescanary8dfb794',
    });
});
test('Name validation does not fail when using Tokens', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        canaryName: core_1.Lazy.stringValue({ produce: () => 'My Canary' }),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN: no exception
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary');
});
test('Throws when name is specified incorrectly', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // THEN
    expect(() => new synthetics.Canary(stack, 'Canary', {
        canaryName: 'My Canary',
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    }))
        .toThrowError('Canary name must be lowercase, numbers, hyphens, or underscores (got "My Canary")');
});
test('Throws when name has more than 21 characters', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // THEN
    expect(() => new synthetics.Canary(stack, 'Canary', {
        canaryName: 'a'.repeat(22),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    }))
        .toThrowError(`Canary name is too large, must be between 1 and 21 characters, but is 22 (got "${'a'.repeat(22)}")`);
});
test('An existing role can be specified instead of auto-created', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    const role = new iam.Role(stack, 'role', {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
    });
    // role.addToPolicy(/* required permissions per the documentation */);
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        role,
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        ExecutionRoleArn: stack.resolve(role.roleArn),
    });
});
test('An existing bucket and prefix can be specified instead of auto-created', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    const bucket = new s3.Bucket(stack, 'mytestbucket');
    const prefix = 'canary';
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        artifactsBucketLocation: { bucket, prefix },
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        ArtifactS3Location: stack.resolve(bucket.s3UrlForObject(prefix)),
    });
});
test('Runtime can be specified', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        runtime: synthetics.Runtime.SYNTHETICS_1_0,
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        RuntimeVersion: 'syn-1.0',
    });
});
test('Runtime can be customized', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        runtime: new synthetics.Runtime('fancy-future-runtime-1337.42'),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        RuntimeVersion: 'fancy-future-runtime-1337.42',
    });
});
test('Schedule can be set with Rate', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        schedule: synthetics.Schedule.rate(core_1.Duration.minutes(3)),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        Schedule: assert_1.objectLike({ Expression: 'rate(3 minutes)' }),
    });
});
test('Schedule can be set to 1 minute', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        schedule: synthetics.Schedule.rate(core_1.Duration.minutes(1)),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        Schedule: assert_1.objectLike({ Expression: 'rate(1 minute)' }),
    });
});
test('Schedule can be set with Expression', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        schedule: synthetics.Schedule.expression('rate(1 hour)'),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        Schedule: assert_1.objectLike({ Expression: 'rate(1 hour)' }),
    });
});
test('Schedule can be set to run once', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        schedule: synthetics.Schedule.once(),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        Schedule: assert_1.objectLike({ Expression: 'rate(0 minutes)' }),
    });
});
test('Throws when rate above 60 minutes', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // THEN
    expect(() => new synthetics.Canary(stack, 'Canary', {
        schedule: synthetics.Schedule.rate(core_1.Duration.minutes(61)),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    }))
        .toThrowError('Schedule duration must be between 1 and 60 minutes');
});
test('Throws when rate above is not a whole number of minutes', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // THEN
    expect(() => new synthetics.Canary(stack, 'Canary', {
        schedule: synthetics.Schedule.rate(core_1.Duration.seconds(59)),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    }))
        .toThrowError('\'59 seconds\' cannot be converted into a whole number of minutes.');
});
test('Can share artifacts bucket between canaries', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    const canary1 = new synthetics.Canary(stack, 'Canary1', {
        schedule: synthetics.Schedule.once(),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
    });
    const canary2 = new synthetics.Canary(stack, 'Canary2', {
        schedule: synthetics.Schedule.once(),
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline('/* Synthetics handler code */'),
        }),
        artifactsBucketLocation: { bucket: canary1.artifactsBucket },
    });
    // THEN
    expect(canary1.artifactsBucket).toEqual(canary2.artifactsBucket);
});
test('can specify custom test', () => {
    // GIVEN
    const stack = new core_1.Stack(new core_1.App(), 'canaries');
    // WHEN
    new synthetics.Canary(stack, 'Canary', {
        test: synthetics.Test.custom({
            handler: 'index.handler',
            code: synthetics.Code.fromInline(`
        exports.handler = async () => {
          console.log(\'hello world\');
        };`),
        }),
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Synthetics::Canary', {
        Code: {
            Handler: 'index.handler',
            Script: `
        exports.handler = async () => {
          console.log(\'hello world\');
        };`,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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