import typer
import typing
import pathlib
import tempfile
import fnmatch
import yaml
from rich import print
import conan_cmake_cpp_project_tools.config as config
import conan_cmake_cpp_project_tools.utils as utils
import conan_cmake_cpp_project_tools.steps as steps


APP_NAME="ccc"
app = typer.Typer(name=APP_NAME)
cfg = config.ConfSettings()
progress = config.fspathtree()


@app.callback()
def main(config_file:pathlib.Path=typer.Option(None,help='ccc project config file to use.')
        ,build_type:str=typer.Option("Debug",help='The build type to compile/test.')
        ,release:bool=typer.Option(None,"--release/--debug","-R/-D",help='Set build type to "Release" or "Debug"')
        ,build_dir:pathlib.Path=typer.Option(None,help='The build directory to compile/test in. Auto-generated by default.')
        ,conanfile:pathlib.Path=typer.Option(None,help='The conanfile (conanfile.txt|conanfile.py) to install dependencies with. Detected by default.')
        ,cmakefile:pathlib.Path=typer.Option(None,help='The CMakeLists.txt file to use.')
        ,write_scripts:bool=typer.Option(False,"--write-scripts","-w",help='Write shell scripts to perform each build/test phase.')
        ,config_settings:typing.Optional[typing.List[str]] = typer.Option(None,help='Override project configuration settings.')
        ):

    if release:
        build_type = "Release"
    if release is not None and not release:
        build_type = "Debug"

    config.set_defaults(cfg)
    root_dir = utils.find_project_root(pathlib.Path())
    if root_dir is None:
        print(f"[red]Could not determine root project directory for '{pathlib.Path()}'[/red]")
        raise typer.Exit(code=1)
    config.load_config_files(cfg,utils.find_project_root(pathlib.Path()).absolute(),"ccc")

    cfg['/build_type'] = build_type
    cfg['/directories/root'] = utils.find_project_root(pathlib.Path()).absolute()

    config.set_default_build_dir(cfg)
    config.set_default_conanfile(cfg)
    config.set_default_cmakefile(cfg)

    if build_dir is not None:
        cfg['directories/build'] = build_dir.absolute()
    if conanfile is not None:
        cfg['files/conanfile'] = conanfile.absolute()
    if cmakefile is not None:
        cfg['files/CMakeLists.txt'] = conanfile.absolute()

    if write_scripts:
        cfg['directories/scripts'] = cfg['directories/build']

    if cfg.get('/files/progress',None) is None:
        cfg['/files/progress'] = cfg['directories/build'] / 'ccc-progress.yml'

    if config_settings:
        for config_setting in config_settings:
            settings = utils.parse_option_to_config_entry(config_setting)
            for key in settings:
                cfg[key] = settings[key]

    if not cfg['/files/progress'].exists():
        cfg['/files/progress'].parent.mkdir(parents=True,exist_ok=True)
        cfg['/files/progress'].write_text('{}')


    progress.tree.update( yaml.safe_load( cfg['/files/progress'].read_text() ) )


    
def run_step_if_pending(name,error_msg,force_run=False):
    if force_run == False and progress.get(f'/steps/{name}', "incomplete") == "complete":
        print(f'"{name}" step has already completed. Skipping.')
        return 0
    if getattr(steps,name)(cfg) != 0:
        print(f"{error_msg}")
        progress[f'/steps/{name}'] = "error"
        return 1

    progress[f'/steps/{name}'] = "complete"
    return 0

    
@app.command()
def install_deps(force:bool=typer.Option(False,"-f",help="Force all steps to run, even if it has been completed.")):
    '''
    Install project dependencies into build directory with Conan.
    '''

    with tempfile.TemporaryDirectory() as tmpdir:
        if not cfg.get('/directories/scripts',False):
            cfg['/directories/scripts'] = pathlib.Path(tmpdir).absolute()
        with utils.working_directory(tmpdir):
            run_step_if_pending('install_deps',"[red]There was an error installing dependencies.[/red]",force_run=force)

    cfg['/files/progress'].write_text( yaml.dump(progress.tree) )


@app.command()
def configure(force:bool=typer.Option(False,"-f",help="Force all steps to run, even if it has been completed.")):
    '''
    Configure project build.
    '''

    with tempfile.TemporaryDirectory() as tmpdir:
        if not cfg.get('directories/scripts',False):
            cfg['directories/scripts'] = pathlib.Path(tmpdir).absolute()
        with utils.working_directory(tmpdir):
            if run_step_if_pending('install_deps',"[red]There was an error installing dependencies. Halting.[/red]",force_run=force) == 0:
                run_step_if_pending('configure_build',"[red]There was an error configuring build.[/red]",force_run=force)

    cfg['/files/progress'].write_text( yaml.dump(progress.tree) )

@app.command()
def build(force:bool=typer.Option(False,"-f",help="Force all steps to run, even if it has been completed.")):
    '''
    Build project build.
    '''

    with tempfile.TemporaryDirectory() as tmpdir:
        if not cfg.get('directories/scripts',False):
            cfg['directories/scripts'] = pathlib.Path(tmpdir).absolute()
        with utils.working_directory(tmpdir):
            if run_step_if_pending('install_deps',"[red]There was an error installing dependencies. Halting.[/red]",force_run=force) == 0:
                if run_step_if_pending('configure_build',"[red]There was an error configuring build. Halting.[/red]",force_run=force) == 0:
                    run_step_if_pending('run_build',"[red]There was an error running build.[/red]",force_run=True)
    cfg['/files/progress'].write_text( yaml.dump(progress.tree) )


# we are not naming this function `test` because pytest will pick it up as a test to run, which it is not.
@app.command(name="test")
def run_test(force:bool=typer.Option(False,"-f",help="Force all steps to run, even if it has been completed.")
        , include:typing.Optional[typing.List[str]] = typer.Option(None,"--include","-i",help="Include pattern(s) to filter test executables that will run.")
        , exclude:typing.Optional[typing.List[str]] = typer.Option(None,"--exclude","-x",help="Exclude pattern(s) to filter test executables that will run.")
        ):
    '''
    Run project unit tests.
    '''
    if include:
        cfg['/run_tests/include'] = include
    if exclude:
        cfg['/run_tests/exclude'] = exclude

    with tempfile.TemporaryDirectory() as tmpdir:
        if not cfg.get('directories/scripts',False):
            cfg['directories/scripts'] = pathlib.Path(tmpdir).absolute()
        with utils.working_directory(tmpdir):
            if run_step_if_pending('install_deps',"[red]There was an error installing dependencies. Halting.[/red]",force_run=force) == 0:
                if run_step_if_pending('configure_build',"[red]There was an error configuring build. Halting.[/red]",force_run=force) == 0:
                    if run_step_if_pending('run_build',"[red]There was an error running build. Halting.[/red]",force_run=True) == 0:
                        if run_step_if_pending('run_tests',"[red]There was an error running tests.[/red]",force_run=True) == 0:
                            print("[green]All tests passed[/green]")

    cfg['/files/progress'].write_text( yaml.dump(progress.tree) )


@app.command()
def install(install_dir:pathlib.Path,force:bool=typer.Option(False,"-f",help="Force all steps to run, even if it has been completed.")):
    '''
    Install project into a directory.
    '''

    cfg['cmake/install/extra_args'] = ['--prefix',str(install_dir)]

    with tempfile.TemporaryDirectory() as tmpdir:
        if not cfg.get('directories/scripts',False):
            cfg['directories/scripts'] = pathlib.Path(tmpdir).absolute()
        with utils.working_directory(tmpdir):
            if run_step_if_pending('install_deps',"[red]There was an error installing dependencies. Halting.[/red]",force_run=force) == 0:
                if run_step_if_pending('configure_build',"[red]There was an error configuring build. Halting.[/red]",force_run=force) == 0:
                    if run_step_if_pending('run_build',"[red]There was an error running build. Halting.[/red]",force_run=force) == 0:
                        if run_step_if_pending('run_tests',"[red]There was an error running tests.[/red]",force_run=force) == 0:
                            if steps.install(cfg) != 0:
                                print("[red]There was an error installing.[/red]")
    cfg['/files/progress'].write_text( yaml.dump(progress.tree) )

@app.command()
def debug_tests(force:bool=typer.Option(False,"-f",help="Force all steps to run, even if it has been completed.")
        , include:typing.Optional[typing.List[str]] = typer.Option(None,"--include","-i",help="Include pattern(s) to filter test executables that will run.")
        , exclude:typing.Optional[typing.List[str]] = typer.Option(None,"--exclude","-x",help="Exclude pattern(s) to filter test executables that will run.")
        ):
    '''
    Run project unit tests through a debugger.
    '''

    if include:
        cfg['/run_tests/include'] = include
    if exclude:
        cfg['/run_tests/exclude'] = exclude

    with tempfile.TemporaryDirectory() as tmpdir:
        if not cfg.get('directories/scripts',False):
            cfg['directories/scripts'] = pathlib.Path(tmpdir).absolute()
        with utils.working_directory(tmpdir):
            if run_step_if_pending('install_deps',"[red]There was an error installing dependencies. Halting.[/red]",force_run=force) == 0:
                if run_step_if_pending('configure_build',"[red]There was an error configuring build. Halting.[/red]",force_run=force) == 0:
                    if run_step_if_pending('run_build',"[red]There was an error running build. Halting.[/red]",force_run=True) == 0:
                        run_step_if_pending('debug_tests',"[red]There was an error running tests through debugger.[/red]",force_run=True)

    cfg['/files/progress'].write_text( yaml.dump(progress.tree) )

@app.command()
def info(config_settings:bool=typer.Option(False,help="Print all configureations settings.")):
    '''
    Print some information about the project.
    '''

    print("directories:")
    print("\troot:",cfg['/directories/root'])
    print("\tbuild:",cfg['/directories/build'])
    print("files:")
    print("\tconan:",cfg['/files/conanfile'])
    print("\tcmake:",cfg['/files/CMakeLists.txt'])

    if config_settings:
        print()
        print("all configuration settings")
        for key in cfg.get_all_leaf_node_paths():
            val = cfg[key]
            print(key,':',val)



@app.command()
def add():
    '''
    Add a dependency to the project.
    '''
    pass

@app.command()
def make_editable():
    '''
    Create an editable Conan package for the project.
    '''
    pass

@app.command()
def list_sources():
    '''
    List the source files in a project.
    '''
    include_patterns = cfg.get('/list_sources/include',config.ConfSettings(['*'])).tree
    exclude_patterns = cfg.get('/list_sources/exclude',config.ConfSettings([])).tree
    

    sources = utils.get_source_files(cfg['/directories/root']) | utils.pfilter(utils.filename_matches_pattern_filter(include_patterns)) | -utils.pfilter(utils.filename_matches_pattern_filter(exclude_patterns))
    for file in sources:
        print(str(file))

