# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aws_secrets',
 'aws_secrets.cli',
 'aws_secrets.config',
 'aws_secrets.config.providers',
 'aws_secrets.config.providers.secretsmanager',
 'aws_secrets.config.providers.ssm',
 'aws_secrets.helpers',
 'aws_secrets.miscellaneous',
 'aws_secrets.representers',
 'aws_secrets.tags']

package_data = \
{'': ['*']}

install_requires = \
['PyYaml>=5.1,<6.0',
 'aws-encryption-sdk>=3.0.0,<4.0.0',
 'boto3>=1.3,<2.0',
 'click>=7.0,<8.0',
 'jsonschema>=3.2.0,<4.0.0']

entry_points = \
{'console_scripts': ['aws-secrets = aws_secrets.cli.cli:cli']}

setup_kwargs = {
    'name': 'aws-ssm-secrets-cli',
    'version': '2.1.0',
    'description': 'AWS Secret CLI for manage SSM SecureString and SecretsManager',
    'long_description': '# AWS Secrets CLI\n\n## About\n\nAWS Secrets CLI is a tool to manage [SSM Parameter Store](https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-parameter-store.html) (SecureString and String) using KMS to encrypt your information. This tool enables you to store your secrets information without exposing it to your git repository.\n\n## Motivation\n\nWhen you need to manage SSM parameter (SecureString) in multiple AWS Environments you need to create or update manually, because [CloudFormation](https://aws.amazon.com/pt/cloudformation/) doesn\'t support SSM parameter type Secure, you can use AWS CLI or boto3 to establish the parameters for you, but you need to read the secrets values from somewhere. You can\'t store into your git repository.\n\nAWS Secrets CLI provides you a Command Line Interface that manages your secrets using [KMS](https://aws.amazon.com/pt/kms/), so you can store the config file into your git repository because your secrets will not expose, only for people that have access to KMS Key.\n\n## Install\n\n```shell\npip install aws-ssm-secrets-cli\n```\n\n## Requirements\n\nIt is necessary to create a KMS key before starting to create the parameter using the CLI.\n\nYou can create this key using AWS CLI, AWS SDK, console, or CloudFormation:\n\nExample using CloudFormation:\n\n```yaml\nDescription: "KMS Key for Secrest"\nResources:\n  Key:\n    Type: AWS::KMS::Key\n    Properties:\n      KeyPolicy:\n        Statement:\n          - Action:\n              - kms:Create*\n              - kms:Describe*\n              - kms:Enable*\n              - kms:List*\n              - kms:Put*\n              - kms:Update*\n              - kms:Revoke*\n              - kms:Disable*\n              - kms:Get*\n              - kms:Delete*\n              - kms:ScheduleKeyDeletion\n              - kms:CancelKeyDeletion\n              - kms:GenerateDataKey\n              - kms:TagResource\n              - kms:UntagResource\n            Effect: Allow\n            Principal:\n              AWS: !Sub "arn:aws:iam::${AWS::AccountId}:root"\n            Resource: "*"\n          - Action:\n              - kms:Decrypt\n              - kms:Encrypt\n              - kms:ReEncrypt*\n              - kms:GenerateDataKey*\n            Effect: Allow\n            Principal:\n              AWS: !Sub "arn:aws:iam::${AWS::AccountId}:root"\n            Resource: "*"\n        Version: "2012-10-17"\n      Description: AWS KMS Key for secrets\n    UpdateReplacePolicy: Retain\n    DeletionPolicy: Retain\n\n  KeyAlias:\n    Type: AWS::KMS::Alias\n    Properties:\n      AliasName: alias/infra-scripts-secrets\n      TargetKeyId: !GetAtt Key.Arn\n\nOutputs:\n  KeyArn:\n    Description: KMS Key Arn\n    Value: !GetAtt Key.Arn\n```\n\n## Getting Started\n\n### Our fist config\n\nFor naming convention, you should give the environment name for the file name (e.g., dev.yaml)\n\n```yaml\nkms:\n  arn: KMS_KEY_ARN (String) #Required\nencryption_sdk: \'aws_encryption_sdk\'\nparameters:\n  - name: myparametername\n    value: "MySecretValueHere"\n    type: SecureString\nsecrets:\n  - name: mysecretname\n    value: "MySecretValueHere"\n```\n\nor AWS Secrets manager with object\n\n```yaml\nkms:\n  arn: KMS_KEY_ARN (String) #Required\nencryption_sdk: \'aws_encryption_sdk\'\nparameters:\n  - name: myparametername\n    value: "MySecretValueHere"\n    type: SecureString\nsecrets:\n  - name: mysecretname\n    value:\n      user: myusername\n      password: mypassword\n```\n\n### Encrypt\n\nTo encrypt the parameter values, you need to use this command:\n\n```shell\naws-secrets encrypt -e dev.yaml --profile myprofile --region eu-west-1\n```\n\n### Decrypt\n\nTo edit the values, you can decrypt and re-encrypt the parameter values. You need to use this command:\n\n```shell\naws-secrets decrypt -e dev.yaml --profile myprofile --region eu-west-1\n```\n\nAt this moment, a new file has created `dev.yaml.dec`. If you want to decrypt in overwrite mode put the `--output` option with the same file name that you are decrypting.\n\n```shell\naws-secrets decrypt -e dev.yaml --output dev.yaml --profile myprofile --region eu-west-1\n```\n\nAfter your changes you need to re-encrypt, you can do it using this command:\n\n```shell\naws-secrets encrypt -e dev.yaml --profile myprofile --region eu-west-1\n```\n\n### Create parameters into AWS Account\n\nTo deploy the parameter that you created on the last step, you need to execute this command:\n\n```shell\naws-secrets deploy -e dev.yaml --profile myaws-profile --region eu-west-1\n```\n\nNow your parameters have been created in AWS Account.\n\n## Migrate KMS API to AWS Encryption SDK\n\nThe AWS Encryption SDK is a client-side encryption library designed to make it easy for everyone to encrypt and decrypt data using industry standards and best practices. It enables you to focus on the core functionality of your application, rather than on how to best encrypt and decrypt your data. The AWS Encryption SDK is provided free of charge under the Apache 2.0 license.\n\nFull documentation: <https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/introduction.html>\n\nUsing AWS Encryption enables AWS Secrets CLI to encrypt data with more than 4KB.\n\n### Migration process\n\n1. Decrypt all SSM parameter and Secrets manager:\n\n```shell\naws-secrets decrypt -e dev.yaml --output dev.yaml --profile myprofile --region eu-west-1\n```\n\n2. Update YAML configuration to add the `encryption_sdk` with `aws_encryption_sdk` value.\n\n```yaml\nkms:\n  arn: KMS_KEY_ARN\nencryption_sdk: \'aws_encryption_sdk\'\nparameters:\n  - name: myparametername\n    value: "MySecretValueHere"\n    type: SecureString\nsecrets:\n  - name: mysecretname\n    value: "MySecretValueHere"\n```\n\n> Currently, the default value is `boto3`\n\n3. Re-encrypt the YAML configuration file\n\n```shell\naws-secrets encrypt -e dev.yaml --profile myprofile --region eu-west-1\n```\n\n## Configuration Schema\n\n```yaml\ntags: # Global tags, applied to all the resources\n  key: \'string\' # key/value pair\nkms:\n  arn: \'string\' # Required, KMS ARN\nencryption_sdk: \'aws_encryption_sdk\' | \'boto3\'\nparameters: # AWS SSM Parameter Section\n  - name: \'string\' # Required, Parameter Name\n    description: \'string\' # Optional, Parameter Description\n    type: \'String|SecureString\' # Required, Parameter Type\n    value: \'string\' # Required only for Type \'String\' or if it is some YAML tag (e.g. !file or !cmd)\n    tier: \'Standard|Advanced|Intelligent-Tiering\' # Optional, Parameter Tier, default \'Standard\'\n    tags: # Optional, Parameter Tags, it is extended with the global tags\n      key: \'string\'\nsecrets: # AWS Secrets Manager secrets Section\n  - name: \'string\' # Required, Secret Name\n    description: \'string\' # Optional, Secret Description\n    value: \'string\' # Required only if it is some YAML tag (e.g. !file or !cmd)\n    tags: # Optional, Secret Tags, it is extended with the global tags\n      key: \'string\'\nsecrets_file: \'Path\' # Optional, Secrets file path, default \'<config-filename>.secrets.yaml\'\n```\n\n## Command Line Interface\n\nCommand options differ depending on the command, and can be found by running:\n\n```shell\naws-secrets --help\naws-secrets COMMAND --help\n```\n\n### encrypt\n\nTo encrypt SecureString parameters and secrets values in the environment file.\n\n```shell\naws-secrets encrypt\n  --env-file\n  [--profile]\n  [--region]\n```\n\n#### Options\n\n| Option               | Description           | Data Type | Required | Options | Default |\n| -------------------- | --------------------- | --------- | -------- | ------- | ------- |\n| `--env-file` or `-e` | Environment file path | `String`  | `true`   |         |         |\n| `--profile`          | AWS Profile           | `String`  | `false`  |         |         |\n| `--region`           | AWS Region            | `String`  | `false`  |         |         |\n\n### decrypt\n\nTo decrypt SecureString parameters and secrets values in the environment file.\n\n```shell\naws-secrets decrypt\n  --env-file\n  [--profile]\n  [--region]\n```\n\n#### Options\n\n| Option               | Description           | Data Type | Required | Options | Default |\n| -------------------- | --------------------- | --------- | -------- | ------- | ------- |\n| `--env-file` or `-e` | Environment file path | `String`  | `true`   |         |         |\n| `--profile`          | AWS Profile           | `String`  | `false`  |         |         |\n| `--region`           | AWS Region            | `String`  | `false`  |         |         |\n\n### set-parameter\n\nCreate or modify the SSM parameter in the environment file.\n\n```shell\naws-secrets set-parameter\n  --env-file\n  --name\n  [--description]\n  [--kms]\n  [--type]\n  [--profile]\n  [--region]\n```\n\n#### Options\n\n| Option                  | Description               | Data Type | Required | Options                     | Default        |\n| ----------------------- | ------------------------- | --------- | -------- | --------------------------- | -------------- |\n| `--env-file` or `-e`    | Environment file path     | `String`  | `true`   |\n| `--name` or `-n`        | SSM Parameter Name        | `String`  | `true`   |                             |                |\n| `--description` or `-d` | SSM Parameter Description | `String`  | `false`  |                             |                |\n| `--type` or `-t`        | SSM Parameter Type        | `String`  | `true`   | `String` and `SecureString` | `SecureString` |\n| `--kms` or `-k`         | KMS Id or ARN             | `String`  | `true`   |                             |                |\n| `--profile`             | AWS Profile               | `String`  | `false`  |                             |                |\n| `--region`              | AWS Region                | `String`  | `false`  |                             |                |\n\n### set-secret\n\nCreate or modify secrets in the environment file.\n\n```shell\naws-secrets set-secret\n  --env-file\n  --name\n  [--description]\n  [--kms]\n  [--profile]\n  [--region]\n```\n\n#### Options\n\n| Option                  | Description           | Data Type | Required | Options | Default |\n| ----------------------- | --------------------- | --------- | -------- | ------- | ------- |\n| `--env-file` or `-e`    | Environment file path | `String`  | `true`   |         |         |\n| `--name` or `-n`        | Secret Name           | `String`  | `true`   |         |         |\n| `--description` or `-d` | Secret Description    | `String`  | `false`  |         |         |\n| `--kms` or `-k`         | KMS Id or ARN         | `String`  | `true`   |         |         |\n| `--profile`             | AWS Profile           | `String`  | `false`  |         |         |\n| `--region`              | AWS Region            | `String`  | `false`  |         |         |\n\n### view-parameter\n\nView the SSM parameter value in the environment file.\n\n```shell\naws-secrets view-parameter\n  --env-file\n  --name\n  [--profile]\n  [--region]\n```\n\n#### Options\n\n| Option               | Description           | Data Type | Required | Options | Default |\n| -------------------- | --------------------- | --------- | -------- | ------- | ------- |\n| `--env-file` or `-e` | Environment file path | `String`  | `true`   |         |         |\n| `--name` or `-n`     | SSM Parameter Name    | `String`  | `true`   |         |         |\n| `--profile`          | AWS Profile           | `String`  | `false`  |         |         |\n| `--region`           | AWS Region            | `String`  | `false`  |         |         |\n\n### deploy\n\nCreate or update SSM parameters and secrets in the AWS Account.\n\n```shell\naws-secrets deploy\n  --env-file\n  [--filter-pattern]\n  [--dry-run]\n  [--confirm]\n  [--only-secrets]\n  [--only-parameters]\n  [--profile]\n  [--region]\n```\n\n#### Options\n\n| Option               | Description                                                                                             | Data Type | Required | Options | Default |\n| -------------------- | ------------------------------------------------------------------------------------------------------- | --------- | -------- | ------- | ------- |\n| `--env-file` or `-e` | Environment file path                                                                                   | `String`  | `true`   |         |         |\n| `--filter-pattern`   | Filter Pattern (e.g `/app/db/*/password` match with `/app/db/dev/password` or `/app/db/prod/password` ) | `String`  | `false`  |         |         |\n| `--dry-run`          | Execution without apply the changes on the environment                                                  | `Boolean` | `false`  |         | `false` |\n| `--confirm`          | Confirm prompt before apply the changes                                                                 | `Boolean` | `false`  |         | `false` |\n| `--only-secrets`     | Apply changes just for AWS Secrets                                                                      | `Boolean` | `false`  |         | `false` |\n| `--only-parameters`  | Apply changes just for SSM Parameters                                                                   | `Boolean` | `false`  |         | `false` |\n| `--profile`          | AWS Profile                                                                                             | `String`  | `false`  |         |         |\n| `--region`           | AWS Region                                                                                              | `String`  | `false`  |         |         |\n\n#### Resolvers\n\nThis CLI implements resolvers, which can be used to resolve the value of a command output or a CloudFormation output value.\n\n##### !file\n\nThis resolver is designed to load a file content to the SSM Parameter or Secrets Manager Value.\n\nExample:\n\n```yaml\n...\nsecrets:\n  - name: mysecret\n    value: !file myfile.txt\n...\n```\n\n##### !cf_output\n\nThis resolver can be used in `parameters[*].value`, `secrets[*].value` and `kms.arn` properties.\n\nExample:\n\n```yaml\nkms:\n  arn: !cf_output "mystack.MyOutputKey"\nparameters:\n  - name: myparameter-name\n    type: String\n    value: !cf_output "mystack.MyOutputKey"\n```\n\n##### !cmd\n\nThis resolver can be used in `parameters[*].value` and `secrets[*].value` properties.\n\nExample:\n\n```yaml\nkms:\n  arn: "arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"\nparameters:\n  - name: myparameter-name\n    type: SecureString\n    value: !cmd \'echo "Teste"\'\n```\n\n###### providers\n\n###### cf\n\nCloudFormation Stack Output resolver\n\nUsage\n\n```text\n${cf:stack-name.output-name}\n```\n\nWith default values\n\n```text\n${cf:stack-name.output-name, \'mydefaultvalue\'}\n```\n\n###### session\n\nAWS Credentials Session resolver\n\nUsage\n\n```text\n${session:profile} or ${session:region}\n```\n\nWith default values\n\n```text\n${session:profile, \'myprofile\'} or ${session:region, \'us-east-1\'}\n```\n\n###### aws\n\nAWS Provider resolves the AWS CLI `--profile` and `--region` based on the `aws-secrets` CLI.\n\nUsage\n\n```text\n${aws:profile} or ${aws:region}\n```\n\nWith default values\n\n```text\n${aws:profile, \'myprofile\'} or ${aws:region, \'us-east-1\'}\n```\n\n**Example**:\n\nWith the config file:\n\n```yaml\nkms:\n  arn: !cf_output "mystack.KeyArn"\nparameters:\n  - description: My SSM Parameter\n    name: /my/ssm/param\n    type: SecureString\n    value: !cmd \'aws s3 ls ${aws:profile} ${aws:region, "eu-west-1"}\'\n```\n\nWhen run the `aws-secrets` with the `--profile` or `--region`\n\n```shell\naws-secrets -e conf.yaml --profile myprofile --region us-east-1\n```\n\nThe AWS CLI command will be execute this command:\n\n```shell\naws s3 ls --profile myprofile --region us-east-1\n```\n\nIf `--profile` not be speficied in the `aws-secrets` execution, like this:\n\n```shell\naws-secrets -e conf.yaml --region us-east-1\n```\n\nThe AWS CLI command will be execute this command:\n\n```shell\naws s3 ls --region eu-west-1\n```\n\n> The `--region` continue in the command because the resolver has the default value with `eu-west-1` in the config file.\n\n### Global Tags\n\nYou also can include Tags on a global level:\n\n```yaml\ntags:\n  SomeKey: SomeValue\nkms:\n  arn: "arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"\nparameters: ...\nsecrets: ...\n```\n',
    'author': 'Lucas Vieira',
    'author_email': 'lucas.vieira94@outlook.com',
    'maintainer': 'Lucas Vieira',
    'maintainer_email': 'lucas.vieira94@outlook.com',
    'url': 'https://github.com/lucasvieirasilva/aws-ssm-secrets-cli',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
