"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretValue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK.
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 *
 * @stability stable
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs.
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     *
     * @stability stable
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     *
     * @param secretId The ID or ARN of the secret.
     * @param options Options.
     * @stability stable
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error('secretId cannot be empty');
        }
        if (!secretId.startsWith('arn:') && secretId.includes(':')) {
            throw new Error(`secret id "${secretId}" is not an ARN but contains ":"`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || '',
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager Parameter Store.
     * @param version An integer that specifies the version of the parameter to use.
     * @stability stable
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     * @stability stable
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     * @stability stable
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error('CloudFormation parameter must be configured with "NoEcho"');
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
_a = JSII_RTTI_SYMBOL_1;
SecretValue[_a] = { fqn: "@aws-cdk/core.SecretValue", version: "1.96.0" };
//# sourceMappingURL=data:application/json;base64,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