"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatAnalytics = exports.MetadataResource = void 0;
const zlib = require("zlib");
const region_info_1 = require("@aws-cdk/region-info");
const cfn_condition_1 = require("../cfn-condition");
const cfn_fn_1 = require("../cfn-fn");
const cfn_pseudo_1 = require("../cfn-pseudo");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const lazy_1 = require("../lazy");
const token_1 = require("../token");
const runtime_info_1 = require("./runtime-info");
/**
 * Construct that will render the metadata resource
 */
class MetadataResource extends construct_compat_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        const metadataServiceExists = token_1.Token.isUnresolved(scope.region) || region_info_1.RegionInfo.get(scope.region).cdkMetadataResourceAvailable;
        if (metadataServiceExists) {
            const resource = new cfn_resource_1.CfnResource(this, 'Default', {
                type: 'AWS::CDK::Metadata',
                properties: {
                    Analytics: lazy_1.Lazy.string({ produce: () => formatAnalytics(runtime_info_1.constructInfoFromStack(scope)) }),
                },
            });
            // In case we don't actually know the region, add a condition to determine it at deploy time
            if (token_1.Token.isUnresolved(scope.region)) {
                const condition = new cfn_condition_1.CfnCondition(this, 'Condition', {
                    expression: makeCdkMetadataAvailableCondition(),
                });
                // To not cause undue template changes
                condition.overrideLogicalId('CDKMetadataAvailable');
                resource.cfnOptions.condition = condition;
            }
        }
    }
}
exports.MetadataResource = MetadataResource;
function makeCdkMetadataAvailableCondition() {
    return cfn_fn_1.Fn.conditionOr(...region_info_1.RegionInfo.regions
        .filter(ri => ri.cdkMetadataResourceAvailable)
        .map(ri => cfn_fn_1.Fn.conditionEquals(cfn_pseudo_1.Aws.REGION, ri.name)));
}
/** Convenience type for arbitrarily-nested map */
class Trie extends Map {
}
/**
 * Formats a list of construct fully-qualified names (FQNs) and versions into a (possibly compressed) prefix-encoded string.
 *
 * The list of ConstructInfos is logically formatted into:
 * ${version}!${fqn} (e.g., "1.90.0!aws-cdk-lib.Stack")
 * and then all of the construct-versions are grouped with common prefixes together, grouping common parts in '{}' and separating items with ','.
 *
 * Example:
 * [1.90.0!aws-cdk-lib.Stack, 1.90.0!aws-cdk-lib.Construct, 1.90.0!aws-cdk-lib.service.Resource, 0.42.1!aws-cdk-lib-experiments.NewStuff]
 * Becomes:
 * 1.90.0!aws-cdk-lib.{Stack,Construct,service.Resource},0.42.1!aws-cdk-lib-experiments.NewStuff
 *
 * The whole thing is then either included directly as plaintext as:
 * v2:plaintext:{prefixEncodedList}
 * Or is compressed and base64-encoded, and then formatted as:
 * v2:deflate64:{prefixEncodedListCompressedAndEncoded}
 *
 * Exported/visible for ease of testing.
 */
function formatAnalytics(infos) {
    const trie = new Trie();
    infos.forEach(info => insertFqnInTrie(`${info.version}!${info.fqn}`, trie));
    const plaintextEncodedConstructs = prefixEncodeTrie(trie);
    const compressedConstructs = zlib.gzipSync(Buffer.from(plaintextEncodedConstructs)).toString('base64');
    return `v2:deflate64:${compressedConstructs}`;
}
exports.formatAnalytics = formatAnalytics;
/**
 * Splits after non-alphanumeric characters (e.g., '.', '/') in the FQN
 * and insert each piece of the FQN in nested map (i.e., simple trie).
 */
function insertFqnInTrie(fqn, trie) {
    var _a;
    for (const fqnPart of fqn.replace(/[^a-z0-9]/gi, '$& ').split(' ')) {
        const nextLevelTreeRef = (_a = trie.get(fqnPart)) !== null && _a !== void 0 ? _a : new Trie();
        trie.set(fqnPart, nextLevelTreeRef);
        trie = nextLevelTreeRef;
    }
    return trie;
}
/**
 * Prefix-encodes a "trie-ish" structure, using '{}' to group and ',' to separate siblings.
 *
 * Example input:
 * ABC,ABD,AEF
 *
 * Example trie:
 * A --> B --> C
 *  |     \--> D
 *  \--> E --> F
 *
 * Becomes:
 * A{B{C,D},EF}
 */
function prefixEncodeTrie(trie) {
    let prefixEncoded = '';
    let isFirstEntryAtLevel = true;
    [...trie.entries()].forEach(([key, value]) => {
        if (!isFirstEntryAtLevel) {
            prefixEncoded += ',';
        }
        isFirstEntryAtLevel = false;
        prefixEncoded += key;
        if (value.size > 1) {
            prefixEncoded += '{';
            prefixEncoded += prefixEncodeTrie(value);
            prefixEncoded += '}';
        }
        else {
            prefixEncoded += prefixEncodeTrie(value);
        }
    });
    return prefixEncoded;
}
//# sourceMappingURL=data:application/json;base64,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